from dataclasses import dataclass
from typing import ClassVar, Optional

from enforce_typing import enforce_types

from .base_model import BaseModel
from .core import User
from .options import AuthenticationInitializationRequestOptions, CredentialAssertion, PublicKeyCredentialRequestOptions


@enforce_types
@dataclass
class AuthenticationInitializationRequest(BaseModel):
    """ Used to initialize an authentication operation. """

    user: Optional[User]
    USER_KEY: ClassVar[str] = "user"

    options: Optional[AuthenticationInitializationRequestOptions] = None
    OPTIONS_KEY: ClassVar[str] = "options"

    def to_json_serializable_internal(self) -> dict:
        return {
            AuthenticationInitializationRequest.USER_KEY: self.user,
            AuthenticationInitializationRequest.OPTIONS_KEY: self.options
        }

    @classmethod
    def from_json_serializable(cls, d: dict):
        if d is None:
            return None

        return AuthenticationInitializationRequest(
            User.from_json_serializable(d.get(AuthenticationInitializationRequest.USER_KEY, None)),
            AuthenticationInitializationRequestOptions.from_json_serializable(d.get(AuthenticationInitializationRequest.OPTIONS_KEY, None))
        )


@dataclass
class AuthenticationInitializationResponse(CredentialAssertion):
    """ Contains the representation of ``CredentialRequestOptions`` generated by the Hanko
        Authentication API that must be passed to browser's WebAuthn API via ``navigator.credentials.get()`` in order
        to authenticate with a credential/create an assertion.

        See also: https://www.w3.org/TR/webauthn-2/#sctn-credentialrequestoptions-extension """

    @classmethod
    def from_json_serializable(cls, d: dict):
        if d is None:
            return None

        return AuthenticationInitializationResponse(
            PublicKeyCredentialRequestOptions.from_json_serializable(d.get(AuthenticationInitializationResponse.RESPONSE_KEY, None))
        )
