from dataclasses import dataclass
from enum import Enum, unique
from typing import ClassVar, List, Optional

from enforce_typing import enforce_types

from .authenticator import AuthenticatorTransport
from .base64_entities import UrlEncodedBase64
from .base_model import BaseModel
from .attachment import AuthenticatorAttachment, UserVerificationRequirement
from .challenge import Challenge
from .entities import RelyingPartyEntity, UserEntity
from .webauthn_cose import COSEAlgorithmIdentifier


@enforce_types
@dataclass
class AuthenticatorSelection(BaseModel):
    """ Specifies the requirements regarding authenticator attributes for credential creation. """

    authenticator_attachment: Optional[AuthenticatorAttachment]
    """ If this member is present, eligible authenticators are filtered to only 
        authenticators attached with the specified ``AuthenticatorAttachment`` enum value. """
    AUTHENTICATOR_ATTACHMENT_KEY: ClassVar[str] = "authenticatorAttachment"

    require_resident_key: bool
    """ This member describes the Relying Party's requirements regarding resident
        credentials. If the parameter is set to true, the authenticator MUST create a client-side-resident
        public key credential source when creating a public key credential. """
    REQUIRE_RESIDENT_KEY_KEY: ClassVar[str] = "requireResidentKey"

    user_verification: Optional[UserVerificationRequirement]
    """ This member describes the Relying Party's requirements regarding user verification for
        the ``create()`` operation. Eligible authenticators are filtered to only those capable of satisfying this
        requirement. """
    USER_VERIFICATION_KEY: ClassVar[str] = "userVerification"

    def to_json_serializable_internal(self) -> dict:
        return {
            AuthenticatorSelection.AUTHENTICATOR_ATTACHMENT_KEY: self.authenticator_attachment,
            AuthenticatorSelection.REQUIRE_RESIDENT_KEY_KEY: self.require_resident_key,
            AuthenticatorSelection.USER_VERIFICATION_KEY: self.user_verification
        }

    @classmethod
    def from_json_serializable(cls, obj: dict):
        if obj is None:
            return None

        return AuthenticatorSelection(
            AuthenticatorAttachment.from_json_serializable(obj.get(AuthenticatorSelection.AUTHENTICATOR_ATTACHMENT_KEY, None)),
            obj.get(AuthenticatorSelection.REQUIRE_RESIDENT_KEY_KEY, None),
            UserVerificationRequirement.from_json_serializable(obj.get(AuthenticatorSelection.USER_VERIFICATION_KEY, None))
        )


@unique
class ConveyancePreference(BaseModel, Enum):
    """ WebAuthn relying parties may use :py:class:`ConveyancePreference` to specify their preference regarding
        attestation conveyance during credential generation.

        See also: https://www.w3.org/TR/webauthn/#enum-attestation-convey """

    NONE = "none"
    """ Indicates that the relying party is not interested in authenticator attestation. """

    DIRECT = "direct"
    """ Indicates that the relying party wants to receive the attestation statement as generated by the
        authenticator. """

    INDIRECT = "indirect"
    """ Indicates that the relying party prefers an attestation conveyance yielding verifiable attestation
        statements, but allows the client to decide how to obtain such attestation statements. Note: There is
        no guarantee that the RP will obtain a verifiable attestation statement in this case. For example, in
        the case that the authenticator employs self attestation. """

    def to_json_serializable_internal(self):
        return self.value

    @classmethod
    def from_json_serializable(cls, obj):
        if obj is None:
            return None

        return ConveyancePreference(obj)


@enforce_types
@dataclass
class RegistrationInitializationRequestOptions(BaseModel):
    """ Contains additional authenticator attributes for a registration initialization request. """

    authenticator_selection: AuthenticatorSelection
    AUTHENTICATOR_SELECTION_KEY: ClassVar[str] = "authenticatorSelection"

    conveyance_preference: Optional[ConveyancePreference] = None
    CONVEYANCE_PREFERENCE_KEY: ClassVar[str] = "attestation"

    def to_json_serializable_internal(self) -> dict:
        return {
            RegistrationInitializationRequestOptions.AUTHENTICATOR_SELECTION_KEY: self.authenticator_selection,
            RegistrationInitializationRequestOptions.CONVEYANCE_PREFERENCE_KEY: self.conveyance_preference
        }

    @classmethod
    def from_json_serializable(cls, obj: dict):
        if obj is None:
            return None

        return RegistrationInitializationRequestOptions(
            AuthenticatorSelection.from_json_serializable(obj.get(RegistrationInitializationRequestOptions.AUTHENTICATOR_SELECTION_KEY, None)),
            ConveyancePreference.from_json_serializable(obj.get(RegistrationInitializationRequestOptions.CONVEYANCE_PREFERENCE_KEY, None))
        )


@unique
class CredentialType(BaseModel, Enum):
    """ Defines the valid credential types.
        It is an extension point; values can be added to it in the future, as
        more credential types are defined. The values of this enumeration are used
        for versioning the Authentication Assertion and attestation structures according
        to the type of the authenticator.

        See `§5.10.3. Credential Descriptor <https://www.w3.org/TR/webauthn/#credentialType>`_ """

    PUBLIC_KEY = "public-key"
    """ Currently one credential type is defined, namely ``PUBLIC_KEY``. """

    def to_json_serializable_internal(self):
        return self.value

    @classmethod
    def from_json_serializable(cls, obj):
        if obj is None:
            return None

        return CredentialType(obj)


@enforce_types
@dataclass
class CredentialParameter(BaseModel):
    """ Represents the credential type and algorithm
        that the relying party wants the authenticator to create."""

    type: CredentialType
    TYPE_KEY: ClassVar[str] = "type"

    algorithm: COSEAlgorithmIdentifier
    ALGORITHM_KEY: ClassVar[str] = "alg"

    def to_json_serializable_internal(self):
        return {
            CredentialParameter.TYPE_KEY: self.type,
            CredentialParameter.ALGORITHM_KEY: self.algorithm
        }

    @classmethod
    def from_json_serializable(cls, d: dict):
        if d is None:
            return None

        return CredentialParameter(
            CredentialType.from_json_serializable(d.get(CredentialParameter.TYPE_KEY, None)),
            COSEAlgorithmIdentifier.from_json_serializable(d.get(CredentialParameter.ALGORITHM_KEY, None))
        )


@dataclass
class CredentialDescriptor(BaseModel):
    """ Contains the attributes that are specified by a caller when referring to a public
        key credential as an input parameter to the ``create()`` or ``get()`` methods. It mirrors the fields of
        the ``PublicKeyCredential`` object returned by the latter methods.
        See `§5.10.3. Credential Descriptor <https://www.w3.org/TR/webauthn/#credential-dictionary>`_"""

    type: CredentialType
    """ The valid credential types. """
    TYPE_KEY: ClassVar[str] = "type"

    credential_id: UrlEncodedBase64
    """ The ID of a credential to allow/disallow. """
    CREDENTIAL_ID_KEY: ClassVar[str] = "id"

    transport: List[AuthenticatorTransport]
    """ The authenticator transports that can be used. """
    TRANSPORT_KEY: ClassVar[str] = "transports"

    def to_json_serializable_internal(self):
        return {
            CredentialDescriptor.TYPE_KEY: self.type,
            CredentialDescriptor.CREDENTIAL_ID_KEY: self.credential_id,
            CredentialDescriptor.TRANSPORT_KEY: self.transport
        }

    @classmethod
    def from_json_serializable(cls, d: dict):
        if d is None:
            return None

        return CredentialDescriptor(
            CredentialType.from_json_serializable(d.get(CredentialDescriptor.TYPE_KEY, None)),
            UrlEncodedBase64.from_json_serializable(d.get(CredentialDescriptor.CREDENTIAL_ID_KEY, None)),
            AuthenticatorTransport.from_json_serializable_sequence(d.get(CredentialDescriptor.TRANSPORT_KEY, None))
        )


@dataclass
class PublicKeyCredentialCreationOptions(BaseModel):
    """ In order to create a ``Credential`` via ``create()``, the caller specifies a few parameters in a
        :py:class:`PublicKeyCredentialCreationOptions` object.

        See `§5.4. Options for Credential Creation <https://www.w3.org/TR/webauthn/#dictionary-makecredentialoptions>`_ ."""

    challenge: Challenge
    CHALLENGE_KEY: ClassVar[str] = "challenge"

    relying_party: RelyingPartyEntity
    RELYING_PARTY_KEY: ClassVar[str] = "rp"

    user: UserEntity
    USER_KEY: ClassVar[str] = "user"

    parameters: List[CredentialParameter]
    PARAMETERS_KEY: ClassVar[str] = "pubKeyCredParams"

    authenticator_selection: AuthenticatorSelection
    AUTHENTICATOR_SELECTION_KEY: ClassVar[str] = "authenticatorSelection"

    timeout: int
    TIMEOUT_KEY: ClassVar[str] = "timeout"

    credential_exclude_list: List[CredentialDescriptor]
    CREDENTIAL_EXCLUDE_LIST_KEY: ClassVar[str] = "excludeCredentials"

    extensions: Optional[dict]
    EXTENSIONS_KEY: ClassVar[str] = "extensions"

    attestation: ConveyancePreference
    ATTESTATION_KEY: ClassVar[str] = "attestation"

    def to_json_serializable_internal(self):
        return {
            PublicKeyCredentialCreationOptions.CHALLENGE_KEY: self.challenge,
            PublicKeyCredentialCreationOptions.RELYING_PARTY_KEY: self.relying_party,
            PublicKeyCredentialCreationOptions.USER_KEY: self.user,
            PublicKeyCredentialCreationOptions.PARAMETERS_KEY: self.parameters,
            PublicKeyCredentialCreationOptions.AUTHENTICATOR_SELECTION_KEY: self.authenticator_selection,
            PublicKeyCredentialCreationOptions.TIMEOUT_KEY: self.timeout,
            PublicKeyCredentialCreationOptions.CREDENTIAL_EXCLUDE_LIST_KEY: self.credential_exclude_list,
            PublicKeyCredentialCreationOptions.EXTENSIONS_KEY: self.extensions,
            PublicKeyCredentialCreationOptions.ATTESTATION_KEY: self.attestation
        }

    @classmethod
    def from_json_serializable(cls, d: dict):
        return PublicKeyCredentialCreationOptions(
            Challenge.from_json_serializable(d.get(PublicKeyCredentialCreationOptions.CHALLENGE_KEY, None)),
            RelyingPartyEntity.from_json_serializable(d.get(PublicKeyCredentialCreationOptions.RELYING_PARTY_KEY, None)),
            UserEntity.from_json_serializable(d.get(PublicKeyCredentialCreationOptions.USER_KEY, None)),
            CredentialParameter.from_json_serializable_sequence(d.get(PublicKeyCredentialCreationOptions.PARAMETERS_KEY, None)),
            AuthenticatorSelection.from_json_serializable(d.get(PublicKeyCredentialCreationOptions.AUTHENTICATOR_SELECTION_KEY, None)),
            d.get(PublicKeyCredentialCreationOptions.TIMEOUT_KEY, None),
            CredentialDescriptor.from_json_serializable_sequence(d.get(PublicKeyCredentialCreationOptions.CREDENTIAL_EXCLUDE_LIST_KEY, None)),
            d.get(PublicKeyCredentialCreationOptions.EXTENSIONS_KEY, None),
            ConveyancePreference.from_json_serializable(d.get(PublicKeyCredentialCreationOptions.ATTESTATION_KEY, None))
        )


@dataclass
class CredentialCreation(BaseModel):
    """ Contains the representation of :py:class:`PublicKeyCredentialCreationOptions` generated by the Hanko
        Authentication API that must be passed to browser's WebAuthn API via ``navigator.credentials.create()`` in order
        to create a credential.

        See also: https://www.w3.org/TR/webauthn/#sctn-credentialcreationoptions-extension """

    response: PublicKeyCredentialCreationOptions
    RESPONSE_KEY: ClassVar[str] = "publicKey"

    def to_json_serializable_internal(self) -> dict:
        return {
            CredentialCreation.RESPONSE_KEY: self.response
        }

    @classmethod
    def from_json_serializable(cls, d: dict):
        if d is None:
            return None

        return CredentialCreation(
            PublicKeyCredentialCreationOptions.from_json_serializable(d.get(CredentialCreation.RESPONSE_KEY, None))
        )


@enforce_types
@dataclass
class AuthenticationInitializationRequestOptions(BaseModel):
    """ Includes additional authenticator attributes for the authentication initialization."""

    user_verification: Optional[UserVerificationRequirement] = None
    USER_VERIFICATION_KEY: ClassVar[str] = "userVerification"

    authenticator_attachment: Optional[AuthenticatorAttachment] = None
    AUTHENTICATOR_ATTACHMENT_KEY: ClassVar[str] = "authenticatorAttachment"

    def to_json_serializable_internal(self) -> dict:
        return {
            AuthenticationInitializationRequestOptions.USER_VERIFICATION_KEY: self.user_verification,
            AuthenticationInitializationRequestOptions.AUTHENTICATOR_ATTACHMENT_KEY: self.authenticator_attachment
        }

    @classmethod
    def from_json_serializable(cls, d):
        if d is None:
            return None

        return AuthenticationInitializationRequestOptions(
            UserVerificationRequirement.from_json_serializable(d.get(AuthenticationInitializationRequestOptions.USER_VERIFICATION_KEY, None)),
            AuthenticatorAttachment.from_json_serializable(d.get(AuthenticationInitializationRequestOptions.AUTHENTICATOR_ATTACHMENT_KEY, None))
        )


@dataclass
class PublicKeyCredentialRequestOptions(BaseModel):
    """ Contains the data needed for assertion generation.
        ``Challenge`` must be present, while the other members are optional.

        See `§5.5. Options for Assertion Generation <https://www.w3.org/TR/webauthn/#assertion-options>`_ """

    challenge: Challenge
    CHALLENGE_KEY: ClassVar[str] = "challenge"

    timeout: int
    TIMEOUT_KEY: ClassVar[str] = "timeout"

    relying_party_id: Optional[str]
    RELYING_PARTY_ID_KEY: ClassVar[str] = "rpId"

    allowed_credentials: Optional[List[CredentialDescriptor]]
    ALLOWED_CREDENTIALS_KEY: ClassVar[str] = "allowCredentials"

    user_verification_requirement: Optional[UserVerificationRequirement]
    USER_VERIFICATION_REQUIREMENT_KEY: ClassVar[str] = "userVerification"

    extensions: Optional[dict] = None
    EXTENSIONS_KEY: ClassVar[str] = "extensions"

    def to_json_serializable_internal(self) -> dict:
        return {
            PublicKeyCredentialRequestOptions.CHALLENGE_KEY: self.challenge,
            PublicKeyCredentialRequestOptions.TIMEOUT_KEY: self.timeout,
            PublicKeyCredentialRequestOptions.RELYING_PARTY_ID_KEY: self.relying_party_id,
            PublicKeyCredentialRequestOptions.ALLOWED_CREDENTIALS_KEY: self.allowed_credentials,
            PublicKeyCredentialRequestOptions.USER_VERIFICATION_REQUIREMENT_KEY: self.user_verification_requirement,
            PublicKeyCredentialRequestOptions.EXTENSIONS_KEY: self.extensions
        }

    @classmethod
    def from_json_serializable(cls, d: dict):
        if d is None:
            return None

        return PublicKeyCredentialRequestOptions(
            Challenge.from_json_serializable(d.get(PublicKeyCredentialRequestOptions.CHALLENGE_KEY, None)),
            d.get(PublicKeyCredentialRequestOptions.TIMEOUT_KEY, None),
            d.get(PublicKeyCredentialRequestOptions.RELYING_PARTY_ID_KEY, None),
            CredentialDescriptor.from_json_serializable_sequence(d.get(PublicKeyCredentialRequestOptions.ALLOWED_CREDENTIALS_KEY, None)),
            UserVerificationRequirement.from_json_serializable(d.get(PublicKeyCredentialRequestOptions.USER_VERIFICATION_REQUIREMENT_KEY, None)),
            d.get(PublicKeyCredentialRequestOptions.EXTENSIONS_KEY, None)
        )


@dataclass
class CredentialAssertion(BaseModel):
    """ Contains the representation of :py:class:`PublicKeyCredentialRequestOptions` generated by the Hanko
        Authentication API that must be passed to browser's WebAuthn API via ``navigator.credentials.get()`` in order
        to authenticate with a credential/create an assertion.

        See also: https://www.w3.org/TR/webauthn-2/#sctn-credentialrequestoptions-extension """

    response: PublicKeyCredentialRequestOptions
    RESPONSE_KEY: ClassVar[str] = "publicKey"

    def to_json_serializable_internal(self) -> dict:
        return {
            CredentialAssertion.RESPONSE_KEY: self.response
        }

    @classmethod
    def from_json_serializable(cls, d: dict):
        return CredentialAssertion(
            PublicKeyCredentialRequestOptions.from_json_serializable(d.get(CredentialAssertion.RESPONSE_KEY, None))
        )
