from dataclasses import dataclass
from typing import ClassVar

from enforce_typing import enforce_types

from .base_model import BaseModel
from .core import User
from .options import RegistrationInitializationRequestOptions, CredentialCreation, PublicKeyCredentialCreationOptions


@enforce_types
@dataclass
class RegistrationInitializationRequest(BaseModel):
    """ Used for initiating a credential registration operation. """

    user: User
    USER_KEY: ClassVar[str] = "user"

    options: RegistrationInitializationRequestOptions = None
    OPTIONS_KEY: ClassVar[str] = "options"

    def to_json_serializable_internal(self) -> dict:
        return {
            RegistrationInitializationRequest.USER_KEY: self.user,
            RegistrationInitializationRequest.OPTIONS_KEY: self.options
        }

    @classmethod
    def from_json_serializable(cls, d: dict):
        if d is None:
            return None

        return RegistrationInitializationRequest(
            User.from_json_serializable(d.get(RegistrationInitializationRequest.USER_KEY, None)),
            RegistrationInitializationRequestOptions.from_json_serializable(d.get(RegistrationInitializationRequest.OPTIONS_KEY, None))
        )


@dataclass
class RegistrationInitializationResponse(CredentialCreation):
    """ Contains the representation of a :py:class:`CredentialCreation` generated by the Hanko
        Authentication API that must be passed to browser's WebAuthn API via ``navigator.credentials.create()`` in order
        to create a credential.

        See also: https://www.w3.org/TR/webauthn/#sctn-credentialcreationoptions-extension """

    @classmethod
    def from_json_serializable(cls, d: dict):
        return RegistrationInitializationResponse(
            PublicKeyCredentialCreationOptions.from_json_serializable(d.get(RegistrationInitializationResponse.RESPONSE_KEY, None))
        )
