from dataclasses import dataclass
from typing import ClassVar, Optional

from enforce_typing import enforce_types

from .authentication_initialization import AuthenticationInitializationResponse
from .base_model import BaseModel
from .core import User
from .options import AuthenticationInitializationRequestOptions, PublicKeyCredentialRequestOptions


@enforce_types
@dataclass
class TransactionInitializationRequest(BaseModel):
    """ Used for initializing transaction operations. """

    user: User
    USER_KEY: ClassVar[str] = "user"

    transaction: str
    TRANSACTION_KEY: ClassVar[str] = "transaction"

    options: Optional[AuthenticationInitializationRequestOptions] = None
    OPTIONS_KEY: ClassVar[str] = "options"

    def to_json_serializable_internal(self) -> dict:
        return {
            TransactionInitializationRequest.USER_KEY: self.user,
            TransactionInitializationRequest.TRANSACTION_KEY: self.transaction,
            TransactionInitializationRequest.OPTIONS_KEY: self.options
        }

    @classmethod
    def from_json_serializable(cls, d: dict):
        if d is None:
            return None

        return TransactionInitializationRequest(
            User.from_json_serializable(d.get(TransactionInitializationRequest.USER_KEY, None)),
            d.get(TransactionInitializationRequest.TRANSACTION_KEY, None),
            AuthenticationInitializationRequestOptions.from_json_serializable(d.get(TransactionInitializationRequest.OPTIONS_KEY, None))
        )


@dataclass
class TransactionInitializationResponse(AuthenticationInitializationResponse):
    """ Contains the representation of :py:class:`PublicKeyCredentialRequestOptions` generated by the Hanko
        Authentication API that must be passed to browser's WebAuthn API via ``navigator.credentials.get()`` in order
        to authenticate with a credential/create an assertion.

        See also: https://www.w3.org/TR/webauthn-2/#sctn-credentialrequestoptions-extension """

    @classmethod
    def from_json_serializable(cls, d: dict):
        return TransactionInitializationResponse(
            PublicKeyCredentialRequestOptions.from_json_serializable(d.get(TransactionInitializationResponse.RESPONSE_KEY, None))
        )
