from OpenSSL import SSL
from cryptography.x509.ocsp import (
    OCSPResponseStatus,
    OCSPCertStatus,
)

__module__ = "trivialscan.constants"

FAKE_PROTOCOLS: list[str] = ["TLSv1.4", "TLSv1.8", "TLSv2", "TLSv2.1", "TLSv2.3"]
DTLS_1_0 = "DTLSv1.2 (0xfeff)"
DTLS_1_2 = "DTLSv1.2 (0xfefd)"
SSL2_LABEL = "SSLv2 (0x2ff)"
SSL3_LABEL = "SSLv3 (0x300)"
TLS1_0_LABEL = "TLSv1 (0x301)"
TLS1_1_LABEL = "TLSv1.1 (0x302)"
TLS1_2_LABEL = "TLSv1.2 (0x303)"
TLS1_3_LABEL = "TLSv1.3 (0x304)"
VALIDATION_OID: dict[str, str] = {
    # https://opensource.apple.com/source/security_certificates/security_certificates-55024.2/evroot.config
    # https://src.chromium.org/viewvc/chrome/trunk/src/net/cert/ev_root_ca_metadata.cc?view=markup
    # https://certs.opera.com/03/ev-oids.xml
    # https://searchfox.org/mozilla-central/source/security/certverifier/ExtendedValidation.cpp
    "2.16.840.1.114414.1.7.23.1": "DV",
    "1.3.6.1.4.1.46222.1.10": "DV",
    "1.3.6.1.4.1.34697.1.1": "EV",
    "1.3.6.1.4.1.29836.1.10": "EV",
    "2.16.840.1.114412.1.3.0.2": "EV",
    "1.3.6.1.4.1.14370.1.6": "EV",
    "1.3.6.1.4.1.4146.1.1": "EV",
    "2.16.840.1.114414.1.7.24.3": "EV",
    "1.3.6.1.4.1.14777.6.1.1": "EV",
    "1.3.6.1.4.1.22234.2.5.2.3.1": "EV",
    "1.3.6.1.4.1.23223.2": "EV",
    "1.3.6.1.4.1.23223.1.1.1": "EV",
    "1.2.276.0.44.1.1.1.4": "EV",
    "2.16.840.1.113733.1.7.48.1": "EV",
    "2.16.840.1.113733.1.7.23.6": "EV",
    "2.16.840.1.114171.500.9": "EV",
    "1.3.6.1.4.1.17326.10.14.2.1.2": "EV",
    "1.3.6.1.4.1.17326.10.14.2.2.2": "EV",
    "1.3.6.1.4.1.17326.10.8.12.1.2": "EV",
    "1.3.6.1.4.1.17326.10.8.12.2.2": "EV",
    "2.16.840.1.114404.1.1.2.4.1": "EV",
    "1.3.6.1.4.1.7879.13.24.1": "EV",
    "2.16.792.3.0.4.1.1.4": "EV",
    "1.3.6.1.4.1.34697.2.1": "EV",
    "1.2.616.1.113527.2.5.1.1": "EV",
    "2.16.756.1.89.1.2.1.1": "EV",
    "2.16.840.1.114412.2.1": "EV",
    "2.16.578.1.26.1.3.3": "EV",
    "1.3.6.1.4.1.34697.2.2": "EV",
    "1.2.156.112559.1.1.6.1": "EV",
    "2.16.156.112554.3": "EV",
    "1.3.6.1.4.1.34697.2.4": "EV",
    "1.3.6.1.4.1.40869.1.1.22.3": "EV",
    "1.3.159.1.17.1": "EV",
    "2.16.840.1.114413.1.7.23.3": "EV",
    "1.2.392.200091.100.721.1": "EV",
    "2.16.756.5.14.7.4.8": "EV",
    "2.23.140.1.1": "EV",
    "1.3.6.1.4.1.13177.10.1.3.10": "EV",
    "2.16.840.1.114028.10.1.2": "EV",
    "1.3.6.1.4.1.6449.1.2.1.5.1": "EV",
    "1.3.6.1.4.1.8024.0.2.100.1.2": "EV",
    "2.16.528.1.1003.1.2.7": "EV",
    "1.3.6.1.4.1.782.1.2.1.8.1": "EV",
    "1.3.6.1.4.1.6334.1.100.1": "EV",
    "2.16.840.1.114414.1.7.23.3": "EV",
    "1.3.6.1.4.1.14777.6.1.2": "EV",
    "1.3.6.1.4.1.34697.2.3": "EV",
    "1.3.6.1.4.1.13769.666.666.666.1.500.9.1": "EV",
    "2.16.840.1.113839.0.6.3": "EV",
    "2.16.792.3.0.3.1.1.5": "EV",
    "1.3.6.1.4.1.5237.1.1.3": "EV",
    "2.16.840.1.101.3.2.1.1.5": "EV",
    "1.3.6.1.4.1.30360.3.3.3.3.4.4.3.0": "EV",
    "1.3.6.1.4.1.46222.1.1": "EV",
    "1.3.6.1.4.1.311.60": "EV",
    "1.3.6.1.4.1.48679.100": "EV",
    "1.3.6.1.4.1.55594.1.1.1": "EV",
    "1.3.6.1.4.1.4788.2.200.1": "EV",
    "1.3.6.1.4.1.4788.2.202.1": "EV",
    "1.3.6.1.4.1.31247.1.3": "EV",
    "1.3.6.1.4.1.52331.2": "EV",
    "2.16.840.1.114414.1.7.23.2": "OV",
    "2.16.792.3.0.3.1.1.2": "OV",
    "1.3.6.1.4.1.46222.1.20": "OV",
    "2.23.140.1.2.1": "DV",
    "2.23.140.1.2.2": "OV",
    "2.23.140.1.2.3": "EV",
}
# https://ccadb-public.secure.force.com/ccadb/AllCAAIdentifiersReport
CAA_DOMAINS = {
    "camerfirma.com": ["AC Camerfirma, S.A."],
    "actalis.it": ["Actalis"],
    "amazon.com": ["Amazon Trust Services"],
    "amazontrust.com": ["Amazon Trust Services"],
    "awstrust.com": ["Amazon Trust Services"],
    "amazonaws.com": ["Amazon Trust Services"],
    "aws.amazon.com": ["Amazon Trust Services"],
    "pki.apple.com": [
        "Apple Public Server ECC CA 11 - G1",
        "Apple Public Server ECC CA 12 - G1",
        "Apple Public Server RSA CA 11 - G1",
        "Apple Public Server RSA CA 12 - G1",
    ],
    "certum.pl": ["Asseco Data Systems S.A. (previously Unizeto Certum)"],
    "certum.eu": ["Asseco Data Systems S.A. (previously Unizeto Certum)"],
    "yandex.ru": ["Asseco Data Systems S.A. (previously Unizeto Certum)"],
    "atos.net": ["Atos"],
    "firmaprofesional.com": ["Autoridad de Certificacion Firmaprofesional"],
    "anf.es": ["Autoridad de Certificación (ANF AC)"],
    "buypass.com": ["Buypass"],
    "buypass.no": ["Buypass"],
    "certicamara.com": ["Certicámara"],
    "certigna.fr": [
        "Dhimyotis / Certigna",
        "Certigna Entity Code Signing CA",
        "Certigna Identity CA",
        "Certigna Identity Plus CA",
        "Certigna Services CA",
    ],
    "www.certinomis.com": ["Certinomis / Docapost"],
    "www.certinomis.fr": ["Certinomis / Docapost"],
    "certsign.ro": ["certSIGN"],
    "cfca.com.cn": ["China Financial Certification Authority (CFCA)"],
    "pki.hinet.net": ["Chunghwa Telecom"],
    "tls.hinet.net": ["Chunghwa Telecom"],
    "eca.hinet.net": ["Chunghwa Telecom"],
    "epki.com.tw": ["Chunghwa Telecom"],
    "publicca.hinet.net": ["Chunghwa Telecom"],
    "cisco.com": ["Cisco", "Cisco XSSL-R2"],
    "Comsign.co.il": ["ComSign"],
    "Comsign.co.uk": ["ComSign"],
    "Comsigneurope.com": ["ComSign"],
    "aoc.cat": ["Consorci Administració Oberta de Catalunya (Consorci AOC, CATCert)"],
    "jcsinc.co.jp": [
        "Cybertrust Japan / JCSI",
        "Cybertrust Japan SureCode CA G1",
        "Cybertrust Japan SureCode CA G2",
        "Cybertrust Japan SureCode CA G3",
        "Cybertrust Japan SureCode CA G4",
        "Cybertrust Japan SureCode EV CA G1",
        "Cybertrust Japan SureCode EV CA G2",
        "Cybertrust Japan SureCode EV CA G3",
        "Cybertrust Japan SureCode EV CA G4",
        "Cybertrust Japan SureMail CA G5",
        "Cybertrust Japan SureMail CA G6",
        "Cybertrust Japan SureServer CA G4",
        "Cybertrust Japan SureServer CA G5",
        "Cybertrust Japan SureServer CA G6",
        "Cybertrust Japan SureServer CA G7",
        "Cybertrust Japan SureServer CA G8",
        "Cybertrust Japan SureServer EV CA G3",
        "Cybertrust Japan SureServer EV CA G4",
        "Cybertrust Japan SureServer EV CA G5",
        "Cybertrust Japan SureServer EV CA G6",
        "Cybertrust Japan SureServer EV CA G7",
        "Cybertrust Japan SureServer EV CA G8",
        "Cybertrust Japan SureServer EV CA G9",
        "Cybertrust Japan SureTime CA G1",
        "Cybertrust Japan SureTime CA G2",
        "Cybertrust Japan SureTime CA G3",
        "Cybertrust Japan SureTime CA G4",
    ],
    "dtrust.de": ["D-TRUST"],
    "d-trust.de": ["D-TRUST"],
    "dtrust.net": ["D-TRUST"],
    "d-trust.net": ["D-TRUST"],
    "telesec.de": ["Deutsche Telekom Security GmbH"],
    "pki.dfn.de": ["Deutsche Telekom Security GmbH"],
    "dfn.de": ["Deutsche Telekom Security GmbH"],
    "digicert.com": [
        "QuoVadis",
        "Symantec",
        "Symantec / GeoTrust",
        "Symantec / VeriSign",
        "DigiCert",
        "DigiCert Inc",
    ],
    "geotrust.com": [
        "Symantec",
        "Symantec / GeoTrust",
        "Symantec / VeriSign",
        "QuoVadis",
        "DigiCert",
        "DigiCert Inc",
    ],
    "rapidssl.com": [
        "DigiCert",
        "QuoVadis",
        "Symantec",
        "Symantec / GeoTrust",
        "Symantec / VeriSign",
        "DigiCert Inc",
    ],
    "digitalcertvalidation.com": [
        "Symantec",
        "Symantec / GeoTrust",
        "Symantec / VeriSign",
        "QuoVadis",
        "DigiCert",
        "DigiCert Inc",
    ],
    "volusion.digitalcertvalidation.com": [
        "Symantec",
        "Symantec / GeoTrust",
        "Symantec / VeriSign",
        "DigiCert",
        "DigiCert Inc",
    ],
    "stratossl.digitalcertvalidation.com": [
        "Symantec",
        "Symantec / GeoTrust",
        "Symantec / VeriSign",
        "DigiCert",
        "DigiCert Inc",
    ],
    "intermediatecertificate.digitalcertvalidation.com": [
        "DigiCert",
        "Symantec",
        "Symantec / GeoTrust",
        "Symantec / VeriSign",
        "DigiCert Inc",
    ],
    "1and1.digitalcertvalidation.com": [
        "Symantec",
        "Symantec / GeoTrust",
        "Symantec / VeriSign",
        "DigiCert",
        "Digidentity B.V.",
        "DigiCert Inc",
    ],
    "digidentity.com": [
        "Digidentity BV PKIoverheid Burger CA - 2021",
        "Digidentity BV PKIoverheid Organisatie Server CA - G3",
        "Digidentity BV PKIoverheid Organisatie Server CA - G3",
        "Digidentity BV PKIoverheid Organisatie Services CA - 2021",
        "Digidentity Organisatie CA - G2",
        "Digidentity PKIoverheid Organisatie Server CA - G3",
        "Digidentity PKIoverheid Server CA 2020",
    ],
    "disig.sk": ["Disig, a.s."],
    "globaltrust.eu": ["e-commerce monitoring GmbH"],
    "e-tugra.com.tr": ["E-Tugra"],
    "e-tugra.com": ["E-Tugra"],
    "etugra.com": ["E-Tugra"],
    "etugra.com.tr": ["E-Tugra"],
    "edicomgroup.com": ["EDICOM"],
    "emsign.com": ["eMudhra Technologies Limited"],
    "entrust.net": ["Entrust"],
    "affirmtrust.com": ["Entrust"],
    "fina.hr": ["Financijska agencija (Fina)"],
    "gdca.com.cn": [
        "Global Digital Cybersecurity Authority Co., Ltd. (Formerly Guang Dong Certificate Authority (GDCA))"
    ],
    "globalsign.com": ["GlobalSign nv-sa"],
    "godaddy.com": ["GoDaddy"],
    "starfieldtech.com": ["GoDaddy"],
    "pki.goog": ["Google Trust Services LLC"],
    "google.com": ["Google Trust Services LLC"],
    "eCert.gov.hk": ["Government of Hong Kong (SAR), Hongkong Post, Certizen"],
    "hongkongpost.gov.hk": ["Government of Hong Kong (SAR), Hongkong Post, Certizen"],
    "gpki.go.kr": ["Government of Korea, KLID"],
    "accv.es": [
        "Government of Spain, Autoritat de Certificació de la Comunitat Valenciana (ACCV)"
    ],
    "fnmt.es": ["Government of Spain, Fábrica Nacional de Moneda y Timbre (FNMT)"],
    "efos.se": ["Government of Sweden (Försäkringskassan)"],
    "myndighetsca.se": ["Government of Sweden (Försäkringskassan)"],
    "gca.nat.gov.tw": [
        "Government of Taiwan, Government Root Certification Authority (GRCA)"
    ],
    "www.pkioverheid.nl": [
        "Staat der Nederlanden Domein Server CA 2020",
        "Government of The Netherlands, PKIoverheid (Logius)",
        "QuoVadis CSP - PKI Overheid CA - G2",
        "QuoVadis PKIoverheid EV CA",
        "QuoVadis PKIoverheid Organisatie Server CA - G3",
        "QuoVadis PKIoverheid Organisatie Server CA - G3",
        "QuoVadis PKIoverheid Server CA 2020",
        "QuoVadis PKIoverheid Server CA 2020",
    ],
    "tuntrust.tn": [
        "Government of Tunisia, Agence National de Certification Electronique / National Digital Certification Agency (ANCE/NDCA)"
    ],
    "kamusm.gov.tr": ["Government of Turkey, Kamu Sertifikasyon Merkezi (Kamu SM)"],
    "harica.gr": ["HARICA"],
    "identrust.com": ["IdenTrust Commercial Root CA 1", "IdenTrust Services, LLC"],
    "letsencrypt.org": ["R3", "R4", "Internet Security Research Group", "ISRG Root X1"],
    "izenpe.com": ["CA de Certificados SSL EV", "Izenpe S.A."],
    "izenpe.eus": ["CA de Certificados SSL EV", "Izenpe S.A."],
    "jprs.jp": [
        "JPRS Domain Validation Authority - G3",
        "JPRS Domain Validation Authority - G3",
        "JPRS Domain Validation Authority - G3",
        "JPRS Domain Validation Authority - G4",
        "JPRS Organization Validation Authority - G3",
        "JPRS Organization Validation Authority - G3",
        "JPRS Organization Validation Authority - G3",
        "JPRS Organization Validation Authority - G4",
        "KPN BV PKIoverheid Organisatie Server CA - G3",
    ],
    "kpn.com": [
        "KPN BV PKIoverheid Organisatie Server CA - G3",
        "KPN CM PKIoverheid EV CA",
        "KPN Corporate Market CSP Organisatie CA - G2",
        "KPN Corporate Market CSP Organisatie Services CA - G3",
        "KPN PKIoverheid EV CA",
        "KPN PKIoverheid EV CA",
        "KPN PKIoverheid Organisatie CA - G2",
        "KPN PKIoverheid Organisatie CA - G2",
        "KPN PKIoverheid Organisatie CA - G2",
        "KPN PKIoverheid Organisatie Persoon CA - G3",
        "KPN PKIoverheid Organisatie Persoon CA - G3",
        "KPN PKIoverheid Organisatie Server CA - G3",
        "KPN PKIoverheid Organisatie Server CA - G3",
        "KPN PKIoverheid Organisatie Services CA - G3",
        "KPN PKIoverheid Organisatie Services CA - G3",
        "KPN PKIoverheid Server CA 2020",
        "KPN PKIoverheid Server CA 2020",
    ],
    "elektronicznypodpis.pl": ["Krajowa Izba Rozliczeniowa S.A. (KIR)"],
    "e-szigno.hu": ["Microsec Ltd."],
    "microsoft.com": ["Microsoft Corporation", "Microsoft RSA TLS Issuing AOC CA 02"],
    "multicert.com": ["MULTICERT"],
    "certificate.naver.com": ["NAVER Cloud"],
    "netlock.hu": ["NETLOCK Kft."],
    "netlock.net": ["NETLOCK Kft."],
    "netlock.eu": ["NETLOCK Kft."],
    "web.com": ["Network Solutions", "Network Solutions Certificate Authority"],
    "networksolutions.com": [
        "Network Solutions",
        "Network Solutions Certificate Authority",
    ],
    "wisekey.com": ["OISTE"],
    "hightrusted.com": ["OISTE"],
    "certifyid.com": ["OISTE"],
    "oiste.org": ["OISTE"],
    "oaticerts.com": ["Open Access Technology International, Inc. (OATI)"],
    "quovadisglobal.com": [
        "QuoVadis",
        "QuoVadis CSP - PKI Overheid EV CA",
        "QuoVadis CSP - PKI Overheid CA - G2",
        "QuoVadis PKIoverheid EV CA",
        "QuoVadis PKIoverheid Organisatie Server CA - G3",
        "QuoVadis PKIoverheid Organisatie Server CA - G3",
        "QuoVadis PKIoverheid Organisatie Services CA - G3",
        "QuoVadis PKIoverheid Server CA 2020",
        "QuoVadis PKIoverheid Server CA 2020",
    ],
    "digicert.ne.jp": ["QuoVadis"],
    "cybertrust.ne.jp": [
        "QuoVadis",
        "Cybertrust Japan / JCSI",
        "Cybertrust Japan SureCode CA G1",
        "Cybertrust Japan SureCode CA G2",
        "Cybertrust Japan SureCode CA G3",
        "Cybertrust Japan SureCode CA G4",
        "Cybertrust Japan SureCode EV CA G1",
        "Cybertrust Japan SureCode EV CA G2",
        "Cybertrust Japan SureCode EV CA G3",
        "Cybertrust Japan SureCode EV CA G4",
        "Cybertrust Japan SureMail CA G5",
        "Cybertrust Japan SureMail CA G6",
        "Cybertrust Japan SureServer CA G4",
        "Cybertrust Japan SureServer CA G5",
        "Cybertrust Japan SureServer CA G6",
        "Cybertrust Japan SureServer CA G7",
        "Cybertrust Japan SureServer CA G8",
        "Cybertrust Japan SureServer EV CA G3",
        "Cybertrust Japan SureServer EV CA G4",
        "Cybertrust Japan SureServer EV CA G5",
        "Cybertrust Japan SureServer EV CA G6",
        "Cybertrust Japan SureServer EV CA G7",
        "Cybertrust Japan SureServer EV CA G8",
        "Cybertrust Japan SureServer EV CA G9",
        "Cybertrust Japan SureTime CA G1",
        "Cybertrust Japan SureTime CA G2",
        "Cybertrust Japan SureTime CA G3",
        "Cybertrust Japan SureTime CA G4",
    ],
    "symantec.com": [
        "QuoVadis",
        "DigiCert",
        "Symantec",
        "Symantec / GeoTrust",
        "Symantec / VeriSign",
        "DigiCert Inc",
    ],
    "thawte.com": [
        "QuoVadis",
        "Symantec",
        "Symantec / GeoTrust",
        "Symantec / VeriSign",
        "DigiCert",
        "DigiCert Inc",
    ],
    "secomtrust.net": ["SECOM Trust Systems CO., LTD."],
    "sectigo.com": ["Sectigo"],
    "comodo.com": ["Sectigo"],
    "comodoca.com": ["Sectigo"],
    "usertrust.com": ["Sectigo"],
    "trust-provider.com": ["Sectigo"],
    "trustwave.com": ["SecureTrust"],
    "securetrust.com": ["SecureTrust"],
    "sheca.com": ["Shanghai Electronic Certification Authority Co., Ltd."],
    "imtrust.cn": ["Shanghai Electronic Certification Authority Co., Ltd."],
    "wwwtrust.cn": ["Shanghai Electronic Certification Authority Co., Ltd."],
    "skidsolutions.eu": ["SK ID Solutions AS"],
    "ssl.com": ["SSL.com"],
    "pkioverheid.nl": ["Staat der Nederlanden Domein Server CA 2020"],
    "admin.ch": [
        "Swiss BIT, Swiss Federal Office of Information Technology, Systems and Telecommunication (FOITT)"
    ],
    "swisssign.com": ["SwissSign AG"],
    "swisssign.net": ["SwissSign AG"],
    "swissign.com": ["SwissSign AG"],
    "swisssign.ch": ["SwissSign AG"],
    "swisssign.li": ["SwissSign AG"],
    "swissign.li": ["SwissSign AG"],
    "swisssign.org": ["SwissSign AG"],
    "swisssign.biz": ["SwissSign AG"],
    "swisstsa.ch": ["SwissSign AG"],
    "swisstsa.li": ["SwissSign AG"],
    "digitalid.ch": ["SwissSign AG"],
    "digital-id.ch": ["SwissSign AG"],
    "zert.ch": ["SwissSign AG"],
    "rootsigning.com": ["SwissSign AG"],
    "root-signing.ch": ["SwissSign AG"],
    "ssl-certificate.ch": ["SwissSign AG"],
    "managed-pki.ch": ["SwissSign AG"],
    "managed-pki.de": ["SwissSign AG"],
    "swissstick.com": ["SwissSign AG"],
    "swisssigner.ch": ["SwissSign AG"],
    "pki-posta.ch": ["SwissSign AG"],
    "pki-poste.ch": ["SwissSign AG"],
    "pki-post.ch": ["SwissSign AG"],
    "trustdoc.ch": ["SwissSign AG"],
    "trustsign.ch": ["SwissSign AG"],
    "swisssigner.com": ["SwissSign AG"],
    "postsuisseid.ch": ["SwissSign AG"],
    "suisseid-service.ch": ["SwissSign AG"],
    "signdemo.com": ["SwissSign AG"],
    "sirb.com": ["SwissSign AG"],
    "twca.com.tw": ["Taiwan-CA Inc. (TWCA)"],
    "telia.com": ["Telia Company"],
    "telia.fi": ["Telia Company"],
    "telia.se": ["Telia Company"],
    "trustcor.ca": ["TrustCor Systems"],
    "trustfactory.net": [
        "TrustFactory Client Issuing Certificate Authority",
        "TrustFactory SSL Issuing Certificate Authority",
        "TrustFactory(Pty)Ltd",
    ],
    "gtlsca.nat.gov.tw": ["行政院/政府伺服器數位憑證管理中心 - G1"],
}
VALIDATION_TYPES: dict[str, str] = {
    "DV": "Domain Validation (DV)",
    "OV": "Organization Validation (OV)",
    "EV": "Extended Validation (EV)",
}
X509_DATE_FMT = r"%Y%m%d%H%M%SZ"
WEAK_KEY_SIZE: dict[str, int] = {
    "RSA": 2048,
    "DSA": 4096,
    "EC": 244,
    "DH": 244,
}
RSA_PUBLIC_EXPONENT_CONSTANT = 65537
KNOWN_WEAK_KEYS: dict[str, str] = {
    "RSA": "The use RSA Encryption is considered vulnerable in certain context. 2000: Factorization of a 512-bit RSA Modulus, essentially derive a private key knowing only the public key. Verified bt EFF in 2001. Later in 2009 factorization of up to 1024-bit keys",
    "DSA": "The use DSA Encryption is considered vulnerable. 1999: HPL Laboratories demonstrated lattice attacks on DSA, a non-trivial example of the known message attack that is a total break and message forgery technique. 2010 Dimitrios Poulakis demonstrated a lattice reduction technique for single or multiple message forgery",
    "EC": "The use Elliptic-curve Encryption is considered vulnerable in certain context. 2010 Dimitrios Poulakis demonstrated a lattice reduction technique to attack ECDSA for single or multiple message forgery",
}
KNOWN_WEAK_SIGNATURE_ALGORITHMS: dict[str, str] = {
    "sha1WithRSAEncryption": "The use of SHA1 with RSA Encryption is considered vulnerable. Macquarie University Australia 2009: identified vulnerabilities to collision attacks, later in 2017 Marc Stevens demonstrated collision proofs",
    "md5WithRSAEncryption": "The use of MD5 with RSA Encryption is considered vulnerable. Arjen Lenstra and Benne de Weger 2005: vulnerable to hash collision attacks",
    "md2WithRSAEncryption": "The use of MD2 with RSA Encryption is considered vulnerable. Rogier, N. and Chauvaud, P. in 1995: vulnerable to collision, later preimage resistance, and second-preimage resistance attacks were demonstrated at BlackHat 2008 by Mark Twain",
}
DEPRECATED_OCSP_ALGO: dict[str, str] = {
    "SHA1": "The use of SHA1 is considered vulnerable. Macquarie University Australia 2009: identified vulnerabilities to collision attacks, later in 2017 Marc Stevens demonstrated collision proofs",
}

WEAK_PROTOCOL: dict[str, str] = {
    SSL2_LABEL: f"{SSL2_LABEL} Deprecated in 2011 (rfc6176) with undetectable manipulator-in-the-middle exploits",
    SSL3_LABEL: f"{SSL3_LABEL} Deprecated in 2015 (rfc7568) mainly due to POODLE, a manipulator-in-the-middle exploit",
    TLS1_0_LABEL: f"{TLS1_0_LABEL} 2018 deprecated by PCI Council. Also in 2018, Apple, Google, Microsoft, and Mozilla jointly announced deprecation. Officially deprecated in 2020 (rfc8996)",
    TLS1_1_LABEL: f"{TLS1_1_LABEL} No longer supported by Firefox 24 or newer and Chrome 29 or newer. Deprecated in 2020 (rfc8996)",
}
OCSP_RESP_STATUS: dict[int, str] = {
    OCSPResponseStatus.SUCCESSFUL: "Successful",
    OCSPResponseStatus.MALFORMED_REQUEST: "Malformed Request",
    OCSPResponseStatus.INTERNAL_ERROR: "Internal Error",
    OCSPResponseStatus.TRY_LATER: "Try Later",
    OCSPResponseStatus.SIG_REQUIRED: "Signature Required",
    OCSPResponseStatus.UNAUTHORIZED: "Unauthorized",
}
OCSP_CERT_STATUS: dict[int, str] = {
    OCSPCertStatus.GOOD: "Good",
    OCSPCertStatus.REVOKED: "Revoked",
    OCSPCertStatus.UNKNOWN: "Unknown",
}
SCT_PRESENT = "SCT Extension present"
SCT_MISSING = "Missing SCT extension"
SCT_GOOD = "SCT extension Good"
SCT_INSUFFICIENT = "SCT extension Insufficient"
SCT_STATUS_MAP = {
    SCT_PRESENT: "Signed Certificate Timestamps (SCT) Extension present",
    SCT_MISSING: "Missing Signed Certificate Timestamps (SCT) Extension",
    SCT_GOOD: "3 or more SCT's are sufficient",
    SCT_INSUFFICIENT: "Including less than 3 SCT's is widely scruitinised as insuffient for trust purposes",
}
OCSP_NO_ASSERTION = "NoAssertion"
OCSP_INVALID_ASSERTION_DATE = "InvalidAssertionDate"
OCSP_STAPLED_NO_RESPONDER_URL = "StapledNoResponderURL"
OCSP_STAPLED_BUT_NOT_REQUIRED = "StapledButNotRequired"
OCSP_NO_RESPONDER_URL = "NoResponderURL"
OCSP_MUST_STAPLE_NOT_STAPLED = "MustStapleNotStapled"
OCSP_STATUS_REASON_MAP = {
    OCSP_NO_ASSERTION: "No OCSP responses from any responders or stapled",
    OCSP_INVALID_ASSERTION_DATE: "OCSP response 'thisUpdate' is future dated",
    OCSP_STAPLED_NO_RESPONDER_URL: "stapled response is expected though not required, without any responders it is still a valid assertion",
    OCSP_STAPLED_BUT_NOT_REQUIRED: "stapled response is expected though not required, not very good but still a valid assertion",
    OCSP_NO_RESPONDER_URL: "without any responders it is still a valid assertion",
    OCSP_MUST_STAPLE_NOT_STAPLED: "stapled response was expected and required but not provided duritn TLS negotiation",
}
SESSION_CACHE_MODE: dict[int, str] = {
    SSL.SESS_CACHE_OFF: "no caching",
    SSL.SESS_CACHE_CLIENT: "session_resumption_tickets",
    SSL.SESS_CACHE_SERVER: "session_resumption_caching",
    SSL.SESS_CACHE_BOTH: "session_resumption_both",
}
NOT_KNOWN_WEAK_CIPHERS: list[str] = [
    "ECDHE-RSA-AES256-GCM-SHA384",
    "ECDHE-RSA-AES256-SHA384",
    "ECDHE-RSA-AES128-GCM-SHA256",
    "ECDHE-RSA-AES128-SHA256",
    "ECDHE-ECDSA-AES256-GCM-SHA384",
    "ECDHE-ECDSA-AES256-SHA384",
    "ECDHE-ECDSA-AES128-GCM-SHA256",
    "ECDHE-ECDSA-AES128-SHA256",
    "DHE-DSS-AES256-GCM-SHA384",
    "DHE-RSA-AES256-GCM-SHA384",
    "DHE-RSA-AES256-SHA256",
    "DHE-DSS-AES256-SHA256",
    "DHE-DSS-AES128-GCM-SHA256",
    "DHE-RSA-AES128-GCM-SHA256",
    "DHE-RSA-AES128-SHA256",
    "DHE-DSS-AES128-SHA256",
]
STRONG_CIPHERS: list[str] = [
    "TLS_CHACHA20_POLY1305_SHA256",
    "TLS_AES_128_CCM_8_SHA256",
    "TLS_AES_128_CCM_SHA2",
]
DNSSEC_ALGORITHMS: dict[int, str] = {
    1: "RSA/MD5",
    2: "DH",
    3: "DSA/SHA-1",
    5: "RSA/SHA-1",
    6: "DSA-NSEC3-SHA1",
    7: "RSASHA1-NSEC3-SHA1",
    8: "RSA/SHA-256",
    10: "RSA/SHA-512",
    12: "GHOST R 34.10-2001",
    13: "ECDSA/SHA-256",
    14: "ECDSA/SHA-384",
    15: "Ed25519",
    16: "Ed448",
    253: "private algorithm DNS",
    254: "private algorithm OID",
}
WEAK_DNSSEC_ALGORITHMS: dict[str, str] = {
    "RSA/MD5": "DNSSEC Algorithm RSA/MD5 was deprecated in 2005",
    "DSA/SHA-1": "DNSSEC Algorithm DSA/SHA-1 was deprecated in 2004",
    "RSA/SHA-1": KNOWN_WEAK_SIGNATURE_ALGORITHMS["sha1WithRSAEncryption"],
    "DSA-NSEC3-SHA1": "DNSSEC Algorithm was DSA-NSEC3-SHA1 deprecated in 2008",
    "GHOST R 34.10-2001": "DNSSEC Algorithm GHOST R 34.10-2001 was deprecated in 2010",
}
STRONG_DNSSEC_ALGORITHMS: list[str] = [
    "ECDSA/SHA-384",
    "Ed25519",
    "Ed448",
]
NIST_SP800_131A_KEY_SIZES = {
    "RSA": 2048,
    "DSA": 2048,
    "EC": 224,
    "DH": 224,
}
NIST_SP800_131A_ALLOWED_RFC_CIPHERS = {
    "AES128-GCM-SHA256": "TLS_RSA_WITH_AES_128_GCM_SHA256",
    "AES256-GCM-SHA384": "TLS_RSA_WITH_AES_256_GCM_SHA384",
    "DHE-RSA-AES256-GCM-SHA384": "TLS_DHE_RSA_WITH_AES_256_GCM_SHA384",
    "DHE-RSA-AES128-GCM-SHA256": "TLS_DHE_RSA_WITH_AES_128_GCM_SHA256",
}
NIST_SP800_131A_ALLOWED_TRANSITION_CIPHERS = {
    "AES256-SHA": "TLS_RSA_WITH_AES_256_CBC_SHA",
    "DES-CBC3-SHA": "TLS_RSA_WITH_3DES_EDGE_CBC_SHA",
    "AES128-SHA": "TLS_RSA_WITH_AES_128_CBC_SHA",
}
FIPS_140_2_CIPHERS = [
    "ECDHE-RSA-AES256-GCM-SHA384",
    "ECDHE-ECDSA-AES256-GCM-SHA384",
    "ECDHE-RSA-AES256-SHA384",
    "ECDHE-ECDSA-AES256-SHA384",
    "DHE-DSS-AES256-GCM-SHA384",
    "DHE-RSA-AES256-GCM-SHA384",
    "DHE-RSA-AES256-SHA256",
    "DHE-DSS-AES256-SHA256",
    "ADH-AES256-GCM-SHA384",
    "ADH-AES256-SHA256",
    "ECDH-RSA-AES256-GCM-SHA384",
    "ECDH-ECDSA-AES256-GCM-SHA384",
    "ECDH-RSA-AES256-SHA384",
    "ECDH-ECDSA-AES256-SHA384",
    "AES256-GCM-SHA384",
    "AES256-SHA256",
    "ECDHE-RSA-AES128-GCM-SHA256",
    "ECDHE-ECDSA-AES128-GCM-SHA256",
    "ECDHE-RSA-AES128-SHA256",
    "ECDHE-ECDSA-AES128-SHA256",
    "DHE-DSS-AES128-GCM-SHA256",
    "DHE-RSA-AES128-GCM-SHA256",
    "DHE-RSA-AES128-SHA256",
    "DHE-DSS-AES128-SHA256",
    "ADH-AES128-GCM-SHA256",
    "ADH-AES128-SHA256",
    "ECDH-RSA-AES128-GCM-SHA256",
    "ECDH-ECDSA-AES128-GCM-SHA256",
    "ECDH-RSA-AES128-SHA256",
    "ECDH-ECDSA-AES128-SHA256",
    "AES128-GCM-SHA256",
    "AES128-SHA256",
]
FIPS_140_2_RFC_CIPHERS = [
    "TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384",
    "TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384",
    "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384",
    "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384",
    "TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256",
    "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256",
    "TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256",
    "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256",
    "TLS_DHE_RSA_WITH_AES_256_GCM_SHA384",
    "TLS_DHE_DSS_WITH_AES_256_GCM_SHA384",
    "TLS_DHE_DSS_WITH_AES_256_CBC_SHA256",
    "TLS_DHE_RSA_WITH_AES_256_CBC_SHA256",
    "TLS_DHE_RSA_WITH_AES_128_GCM_SHA256",
    "TLS_DHE_DSS_WITH_AES_128_GCM_SHA256",
    "TLS_DHE_DSS_WITH_AES_128_CBC_SHA256",
    "TLS_DHE_RSA_WITH_AES_128_CBC_SHA256",
    "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA",
    "TLS_ECDH_RSA_WITH_AES_256_CBC_SHA",
    "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA",
    "TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA",
    "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA",
    "TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA",
    "TLS_ECDHE_ECDSA_WITH_3DES_EDGE_CBC_SHA",
    "TLS_ECDH_ECDSA_WITH_3DES_EDGE_CBC_SHA",
    "TLS_ECDH_RSA_WITH_3DES_EDGE_CBC_SHA",
    "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA",
    "TLS_ECDH_RSA_WITH_AES_128_CBC_SHA",
    "TLS_ECDHE_RSA_WITH_3DES_EDGE_CBC_SHA",
    "TLS_DHE_RSA_WITH_AES_256_CBC_SHA",
    "TLS_DHE_DSS_WITH_AES_256_CBC_SHA",
    "TLS_RSA_WITH_AES_256_CBC_SHA",
    "TLS_DHE_DSS_WITH_AES_128_CBC_SHA",
    "TLS_DHE_RSA_WITH_AES_128_CBC_SHA",
    "TLS_RSA_WITH_AES_128_CBC_SHA",
]

TLSA_EXTENSION_OID = "1.3.6.1.4.1.11129.2.1.4"
PROTOCOL_VERSION: dict[str, int] = {
    "DTLSv1": 0xFEFF,
    "DTLSv1.2": 0xFEFD,
    "SSLv2": 0x02FF,
    "SSLv3": 0x0300,
    "TLSv1": 0x0301,
    "TLSv1.1": 0x0302,
    "TLSv1.2": 0x0303,
    "TLSv1.3": 0x0304,
    "TLSv1.3_DRAFT_14": 0x7F0E,
    "TLSv1.3_DRAFT_15": 0x7F0F,
    "TLSv1.3_DRAFT_16": 0x7F10,
    "TLSv1.3_DRAFT_17": 0x7F11,
    "TLSV1.3_DRAFT_18": 0x7F12,
    "TLSV1.3_DRAFT_18_mozilla_pr_1092": 0x7E02,
    "TLSv1.3_DRAFT_19": 0x7F13,
    "TLSv1.3_DRAFT_20": 0x7F14,
    "TLSv1.3_DRAFT_21": 0x7F15,
    "TLSv1.3_DRAFT_22": 0x7F16,
    "TLSv1.3_DRAFT_23": 0x7F17,
    "TLSv1.3_DRAFT_24": 0x7F18,
    "TLSv1.3_DRAFT_25": 0x7F19,
    "TLSv1.3_DRAFT_26": 0x7F1A,
    "TLSv1.3_DRAFT_27": 0x7F1B,
    "TLSv1.3_DRAFT_28": 0x7F1C,
    "TLSv1.4": 0x0305,
    "TLSv1.8": 0x0309,
    "TLSv2": 0x0400,
    "TLSv2.1": 0x0401,
    "TLSv2.3": 0x0404,
    "GREASE": 0x0A0A,
}
OPENSSL_VERSION_LOOKUP: dict[int, str] = {
    int(key): value for (value, key) in PROTOCOL_VERSION.items()
}
PROTOCOL_TEXT_MAP: dict[str, int] = {
    DTLS_1_0: 0xFEFF,
    DTLS_1_2: 0xFEFD,
    SSL2_LABEL: 0x02FF,
    SSL3_LABEL: 0x0300,
    TLS1_0_LABEL: 0x0301,
    TLS1_1_LABEL: 0x0302,
    TLS1_2_LABEL: 0x0303,
    TLS1_3_LABEL: 0x0304,
}
ALL_CIPHERS = "DHE-RSA-AES128-CCM:ECDHE-ECDSA-CHACHA20-POLY1305:DHE-PSK-CAMELLIA128-SHA256:SRP-3DES-EDGE-CBC-SHA:AES256-SHA:RSA-PSK-3DES-EDGE-CBC-SHA:DHE-PSK-AES128-GCM-SHA256:ADH-AES256-SHA:CAMELLIA128-SHA:RSA-PSK-AES128-GCM-SHA256:ECDHE-ECDSA-AES128-SHA:IDEA-CBC-SHA:DHE-PSK-NULL-SHA256:CAMELLIA128-SHA256:ECDHE-PSK-CAMELLIA256-SHA384:DHE-RSA-AES256-SHA256:RSA-PSK-AES256-CBC-SHA:AES128-SHA:PSK-AES256-CBC-SHA384:DHE-DSS-SEED-SHA:DHE-RSA-AES256-CCM:SRP-AES-128-CBC-SHA:AES128-CCM:AES256-CCM:AES128-SHA256:ECDHE-RSA-AES128-SHA:RSA-PSK-CAMELLIA128-SHA256:ADH-CAMELLIA128-SHA256:DHE-DSS-AES128-SHA256:ECDHE-ECDSA-AES256-SHA:DHE-PSK-AES128-CCM8:ECDHE-RSA-DES-CBC3-SHA:DHE-RSA-CAMELLIA256-SHA256:NULL-SHA256:RSA-PSK-CAMELLIA256-SHA384:PSK-NULL-SHA384:DHE-DSS-DES-CBC3-SHA:DHE-RSA-CAMELLIA128-SHA:CAMELLIA256-SHA:ECDHE-RSA-CAMELLIA128-SHA256:PSK-AES256-CBC-SHA:SRP-RSA-AES-256-CBC-SHA:ECDHE-ECDSA-CAMELLIA128-SHA256:DHE-RSA-DES-CBC3-SHA:DHE-DSS-AES128-GCM-SHA256:PSK-CHACHA20-POLY1305:ADH-AES256-SHA256:ECDHE-ECDSA-AES128-SHA256:ECDHE-ECDSA-CAMELLIA256-SHA384:SRP-AES-256-CBC-SHA:ECDHE-PSK-NULL-SHA:PSK-AES128-CCM8:SRP-DSS-3DES-EDGE-CBC-SHA:RSA-PSK-NULL-SHA:TLS_AES_128_GCM_SHA256:CAMELLIA256-SHA256:AES256-CCM8:SRP-RSA-AES-128-CBC-SHA:DHE-PSK-AES256-GCM-SHA384:DHE-PSK-AES128-CBC-SHA:DHE-RSA-AES256-SHA:DHE-RSA-SEED-SHA:DHE-PSK-AES256-CCM8:DHE-RSA-AES128-CCM8:AECDH-DES-CBC3-SHA:ECDHE-ECDSA-AES256-GCM-SHA384:DHE-PSK-AES256-CBC-SHA384:ECDHE-ECDSA-AES256-SHA384:ADH-AES128-SHA256:ECDHE-PSK-AES128-CBC-SHA:DHE-PSK-AES128-CCM:RSA-PSK-NULL-SHA384:TLS_AES_256_GCM_SHA384:DHE-RSA-AES128-SHA:DHE-PSK-NULL-SHA384:ADH-CAMELLIA128-SHA:AES128-CCM8:ADH-AES128-GCM-SHA256:DHE-RSA-CAMELLIA256-SHA:ECDHE-ECDSA-DES-CBC3-SHA:ECDHE-PSK-CAMELLIA128-SHA256:DHE-RSA-AES128-GCM-SHA256:ECDHE-PSK-CHACHA20-POLY1305:DHE-RSA-AES128-SHA256:DHE-PSK-CHACHA20-POLY1305:DHE-DSS-CAMELLIA128-SHA256:DHE-RSA-AES256-CCM8:PSK-AES128-CCM:DHE-DSS-AES256-SHA256:ADH-CAMELLIA256-SHA:DHE-RSA-AES256-GCM-SHA384:ECDHE-ECDSA-AES256-CCM:ADH-CAMELLIA256-SHA256:ECDHE-PSK-3DES-EDGE-CBC-SHA:PSK-NULL-SHA:DHE-PSK-CAMELLIA256-SHA384:ECDHE-ECDSA-NULL-SHA:ECDHE-RSA-AES256-SHA384:SRP-DSS-AES-256-CBC-SHA:ECDHE-PSK-AES256-CBC-SHA384:PSK-AES128-CBC-SHA:ECDHE-RSA-AES256-SHA:PSK-AES128-CBC-SHA256:NULL-MD5:ECDHE-ECDSA-AES128-GCM-SHA256:DHE-DSS-CAMELLIA256-SHA:RSA-PSK-AES128-CBC-SHA256:SRP-RSA-3DES-EDGE-CBC-SHA:ADH-AES256-GCM-SHA384:DHE-PSK-3DES-EDGE-CBC-SHA:DHE-PSK-AES256-CCM:TLS_AES_128_CCM_SHA256:ECDHE-RSA-AES128-SHA256:DHE-PSK-AES128-CBC-SHA256:ADH-AES128-SHA:PSK-CAMELLIA256-SHA384:ECDHE-ECDSA-AES128-CCM8:DHE-PSK-NULL-SHA:ECDHE-PSK-AES128-CBC-SHA256:DHE-DSS-CAMELLIA256-SHA256:ADH-SEED-SHA:DHE-RSA-CHACHA20-POLY1305:ECDHE-RSA-AES256-GCM-SHA384:RSA-PSK-AES128-CBC-SHA:PSK-3DES-EDGE-CBC-SHA:PSK-NULL-SHA256:AECDH-AES128-SHA:RSA-PSK-AES256-GCM-SHA384:AECDH-NULL-SHA:RSA-PSK-AES256-CBC-SHA384:AECDH-AES256-SHA:DHE-RSA-CAMELLIA128-SHA256:ECDHE-RSA-NULL-SHA:PSK-CAMELLIA128-SHA256:RSA-PSK-CHACHA20-POLY1305:ECDHE-RSA-CAMELLIA256-SHA384:AES256-SHA256:DHE-DSS-AES256-GCM-SHA384:ECDHE-ECDSA-AES256-CCM8:SRP-DSS-AES-128-CBC-SHA:ECDHE-PSK-NULL-SHA384:DHE-DSS-CAMELLIA128-SHA:DHE-DSS-AES128-SHA:DHE-DSS-AES256-SHA:DHE-PSK-AES256-CBC-SHA:ECDHE-RSA-CHACHA20-POLY1305:PSK-AES128-GCM-SHA256:PSK-AES256-GCM-SHA384:AES256-GCM-SHA384:ADH-DES-CBC3-SHA:RSA-PSK-NULL-SHA256:ECDHE-PSK-NULL-SHA256:SEED-SHA:ECDHE-ECDSA-AES128-CCM:PSK-AES256-CCM:ECDHE-RSA-AES128-GCM-SHA256:DES-CBC3-SHA:ECDHE-PSK-AES256-CBC-SHA:PSK-AES256-CCM8:AES128-GCM-SHA256:NULL-SHA"
COMPROMISED_SHA1: dict[str, str] = {
    "C864484869D41D2B0D32319C5A62F9315AAF2CBD": "Superfish",
    "1148E7C94E7019A577B46881520CA71922EB476E": "Superfish",
    "7E873E428883D071343E4CAC5A204BEECE5489E0": "eDellRoot",
    "98A04E4163357790C4A79E6D713FF0AF51FE6927": "eDellRoot",
    "3D2C76AABBA3879391FDDF5FB8CC960C7ED73574": "DSDTestProvider",
    "02C2D931062D7B1DC2A5C7F5F0685064081FB221": "DSDTestProvider",
    "407D40E1A0D9D25BB8196644DDFDA715A850B236": "Dekart Certificate Authority",
    "70043C289339603792DA928F73F55086603FBF27": "PortableWares Issued by Google",
    "08E4987249BC450748A4A78133CBF041A3510033": "COMODO RSA Domain Validation Secure Server CA",
    "0C08B84A1F26F50549DAAB363308D3565268F160": "DigiNotar Root CA",
    "13DB1BA7073B75443C52A14C3BAE0CC9D478567C": "Entrust Certification Authority - L1C",
    "17C0C59AB5D8D5852043E8EC692C409D8062AA53": "TRIAL PKIoverheid Organisatie TEST CA - G2",
    "1E8E8806AA605544CDA2BBB906B5D0CC7FB6FFF7": "RCS Certification Authority",
    "2B84BFBB34EE2EF949FE1CBE30AA026416EB2216": "GTE CyberTrust Global Root",
    "2DC7ABF67E9E639A22DBDF4EEE9BC52A48A2CC2D": "DigiNotar Root CA",
    "367D4B3B4FCBBC0B767B2EC0CDB2A36EAB71A4EB": "Entrust.net Secure Server Certification Authority",
    "3F018E6FE336096A791B81E769BE701AAF21E307": "DigiNotar Root CA",
    "40AA38731BD189F9CDB5B9DC35E2136F38777AF4": "Staat der Nederlanden Overheid CA",
    "43D9BCB568E039D073A74A71D8511F7476089CC3": "DigiNotar Root CA G2",
    "4D8547B7F864132A7F62D9B75B068521F10B68E3": "EGO BILGI ISLEM",
    "51C3247D60F356C7CA3BAF4C3F429DAC93EE7B74": "GTE CyberTrust Global Root",
    "5DE83EE82AC5090AEA9D6AC4E7A6E213F946E179": "Staat der Nederlanden Organisatie CA - G2",
    "5F6FDDAD652DBBAE1F842A8E3B5F35F6AC12F851": "COMODO RSA Domain Validation Secure Server CA",
    "6A2C5CB094D5E0B757FB0F5842AAC813A5802FE1": "VeriSign Class 3 Public Primary Certification Authority - G3",
    "70432DE43C9D1E79C1132520BC5843F7BB7D9295": "XRamp Global Certification Authority",
    "73F571F7F4FB49C150C467ADA1C9A9EAE6B19AAD": "Entrust.net Certification Authority (2048)",
    "8268993EDAEBB1E4FB77910F12CBD6C670F07CEA": "DigiNotar Root CA",
    "8323F14FBC9F9B80B79DED14CD0157CDFB0895D2": "VeriSign Class 3 Public Primary Certification Authority - G3",
    "86E817C81A5CA672FE000F36F878C19518D6F844": "Entrust.net Secure Server Certification Authority",
    "8E5BD50D6AE686D65252F843A9D4B96D197730AB": "Entrust.net Certification Authority (2048)",
    "9432BD9AEC1D75D1705C543AA34C4AF6A526C13D": "DigiNotar Root CA",
    "9845A431D51959CAF225322B4A4FE9F223CE6D15": "GTE CyberTrust Global Root",
    "9C655ED5FAE3B8964D8972F63A6353593F5EB44E": "VeriSign Class 3 Public Primary Certification Authority - G3",
    "A1DB6393916F17E4185509400415C70240B0AE6B": "Class 3 Public Primary Certification Authority",
    "A24841ABD6A0CA5CCD2AA3B190701ED64B39FE53": "DigiNotar Root CA",
    "B86E791620F759F17B8D25E38CA8BE32E7D5EAC2": "GTE CyberTrust Global Root",
    "C060ED44CBD881BD0EF86C0BA287DDCF8167478C": "DigiNotar Root CA",
    "C69F28C825139E65A646C434ACA5A1D200295DB1": "TÜRKTRUST Elektronik Sunucu Sertifikası Hizmetleri",
    "D43719B51B57CA4BB874167D4795231D3434FDA8": "TRIAL PKIoverheid TEST Root CA - G2",
    "E05F7C22598C1256A7B94D92D3D194508C8CBA71": "Trustwave Organization Issuing CA, Level 2",
    "E1F3591E769865C4E447ACC37EAFC9E2BFE4C576": "CNNIC ROOT",
    "EEEFAA0BCD11AF5C02FA96206AC5C62BA724D60A": "DigiNotar Root CA",
    "F0BB01D1D83198516A4FF16D8AB5EFAB74C7CCA3": "DigiNotar Root CA",
    "F4DB6D0281F204D36E2D2FBFA72F7940ED9D1ADC": "Certification Authority of WoSign",
    "F8A54E03AADC5692B850496A4C4630FFEAA29D83": "Entrust.net Secure Server Certification Authority",
    "F92BE5266CC05DB2DC0DC3F2DC74E02DEFD949CB": "TÜRKTRUST Elektronik Sunucu Sertifikası Hizmetleri",
    "FFAD0E26F05BBCD8063CCE1DFA60245E143D5380": "DigiNotar Root CA",
    "E4CDCDEECA764FC4196B4E6CF8BAAF7659140620": "Adguard CA R73",
    "3A8E038CFB17523CE3323D857DC18FC9E8A5CF0D": "Adguard CA R73",
    "2B52EB9E5F91360170543786190B66D1E47B6709": "Harmony",
    "0874A3219367D67070C0F6D15D8FB55E03AE581B": "Harmony",
    "D468C4971AD856CC96F8B9B4B2D6A1B8040E26BD": "KeepMyFamilySecure",
    "B49438B65F42E2CB43666BC23CFFE531CE7F6D46": "Kurupira.NET",
    "BE90F13A20F8DE5537BF62CFCD5B3CDEFD43B9EA": "Kurupira.NET",
    "7182D5BF77E74168D93C56C841791D1EF2D74506": "PrivDog Secure Connection Inspector CA",
    "46C50A91E5AC1E4D6D090CCA2DCD74B7FFF39040": "PrivDog Secure Connection Inspector CA",
    "653B739F5898BB9C031B1DBCED66E131FDC6BCB8": "Qustodio CA",
    "27980262382C7BA633E1E6879428D67B13FE7429": "Qustodio CA",
    "F17F6FB631DC99E3A3C87FFE1CF1811088D96033": "Malicious (China)",
    "C418F64D46D1DF003D2730137243A91211C675FB": "Malicious (China)",
    "B435D4E1119D1C6690A749EBB394BD637BA782B7": "Malicious (China)",
    "8BAF4C9B1DF02A92F7DA128EB91BACF498604B6F": "Malicious (CNNIC)",
    "8A5C8CEEA503E60556BAD81BD4F6C9B0EDE52FE0": "Malicious (China)",
    "7A4AA61E2A88704115E47748D8647DAEE6837559": "Malicious (China)",
    "7514436E903C901069980499CA70DE74FC06C83C": "Malicious (China)",
    "66F2DCFB3F814DDEE9B3206F11DEFE1BFBDFE132": "Malicious (China)",
    "4F99AA93FB2BD13726A1994ACE7FF005F2935D1E": "China Internet Network Information Center Root CA",
    "0A8F0029EA3CD051A30133BD7AA6ECCFF8FFEDC6": "Malicious (China)",
    "0B972C9EA6E7CC58D93B20BF71EC412E7209FABF": "Malicious (China)",
    "0F36385B811A25C39B314E83CAE9346670CC74B4": "Malicious (China)",
    "1632478D89F9213A92008563F5A4A7D312408AD6": "Malicious (China)",
    "23EB1BA46471A1E7E9F2DB5701FEF8F2F80CAAE9": "Malicious (China)",
    "28F97816197AFF182518AA44FEC1A0CE5CB64C8A": "Malicious (China)",
    "3120F295417730075F8CD42D0CAE008EB5726EF8": "Malicious (China)",
    "31F1FD68226320EEC63B3F9DEA4A3E537C7C3917": "Malicious (China)",
    "33A4D8BC38608EF52EF0E28A35091E9250907FB9": "Malicious (China)",
    "3E2BF7F2031B96F38CE6C4D8A85D3E2D58476A0F": "Malicious (China)",
    "689A1229D698D172E99E1BF00B9C19854119C9FC": "Malicious (China)",
    "6A174570A916FBE84453EED3D070A1D8DA442829": "Malicious (China)",
    "742CA08594ABA62CE76E94386EA75A98712F02EA": "Malicious (China)",
    "8250BED5A214433A66377CBC10EF83F669DA3A67": "Malicious (China)",
    "8E207F5DC0A69AA337658D2367433EC27D8506D0": "Malicious (China)",
    "A3A1B06F2461234AE336A5C237FCA6FFDDF0D73A": "Malicious (China)",
    "A3F1333FE242BFCFC5D14E8F394298406810D1A0": "Malicious (China)",
    "A8F2DFE36AE0CC2DB9DD38347D30AED9551DD25A": "Malicious (China)",
    "AB702CDF18EBE8B438C52869CD4A5DEF48B40E33": "Malicious (China)",
    "B51085D95B95DA912012D2C3B0019A2DC46D26E3": "Malicious (China)",
    "B94294BF91EA8FB64BE61097C7FB001359B676CB": "Malicious (China)",
    "B9C9F58B3BBEF575E2B58328770E7B0076C40B5E": "Malicious (China)",
    "C59DD79CC42245AC7216FDC18524F7D1E4CF1E92": "Malicious (China)",
    "D27AD2BEED94C0A13CC72521EA5D71BE8119F32B": "Malicious (China)",
    "E2B8294B5584AB6B58C290466CAC3FB8398F8483": "Malicious (China)",
    "EABDA240440ABBD694930A01D09764C6C2D77966": "Malicious (China)",
    "EB466CD37565F93CDE1062CD8D9826ED23730F12": "Malicious (China)",
    "F02B70BDE4EAE02B207377B9FD4785E4C9CC55DC": "Malicious (China)",
    "FBEDDC9065B7272037BC550C9C56DEBBF27894E1": "Malicious (China)",
    "0750D9E76816800CC02FD1F4E1C058B5E51AE32A": "Malicious (China)",
    "15C1776480382DEEFC869C2DA7F5BA8626B8A029": "Malicious (China)",
    "1C1ECDCCF764E6168177C5711F33EC9229A29F88": "Malicious (China)",
    "1E3159DAE31F0ED21050075C64E110807E9DA277": "Malicious (China)",
    "27D5BBE04301E1604839708D172CF09296ED9033": "Malicious (China)",
    "337DF96418F08A9355870513AFCEBDC68BCED767": "Malicious (China)",
    "3423C2D0496D3CFBBB1A39159DD218559DC1CD48": "Malicious (China)",
    "381CBC5048AFD9A02D3E5882D5F22D962B1A5F72": "Malicious (China)",
    "38CFE78D9F1F0B0637AFCAAA3D5549D87C0AA1D0": "Malicious (China)",
    "3E14B8BD6C568657D852D95D387249AE857B4A39": "Malicious (China)",
    "4245FAB05FAE296958760C3D10EF55E4F25C8681": "Malicious (China)",
    "46A762F3C3CF3732DE22A8BA1EBBA3BC048F9B8C": "Malicious (China)",
    "56FAADDC596DCF78D585D83A35BC04B690D12736": "Malicious (China)",
    "5B562E002F15E1617C416D1960218544C18F36B7": "Malicious (China)",
    "692790DA5189529CC5CE1E16E984277A03023E99": "Malicious (China)",
    "6D5A18050D56BFDE525CBE89E8C45DD1B53D12E9": "Malicious (China)",
    "73FFCA3F815B7A77717FE91912A4DC7B6BFB79CA": "Malicious (China)",
    "7C1913D189C46577D7513F980A2CFD7EDCBA0EC9": "Malicious (China)",
    "804E5FB7DE84F5F5B28347233EAF07846B6070D3": "Malicious (China)",
    "868241C8B85AF79E2DAC79EDADB723E82A36AFC3": "Malicious (China)",
    "87DD1B740366A877FD28BF7ED709F4ACE1FA43A5": "Malicious (China)",
    "87DFE0056D01EB9635395D44A856218484F492F0": "Malicious (China)",
    "B0B68AE97CFE2AFACD0DC2010B9D70ACE593E8A6": "Malicious (China)",
    "B1B3162F7CC867F6D47536DF3232E706E031357C": "Malicious (China)",
    "B2FBDA222493A93C38F77C90D4BE6DA17F15F0B0": "Malicious (China)",
    "B39191CFF08EB6FD8B447C21CAAEF6FC33F1D5AE": "Malicious (China)",
    "BCF8FDFBD8A8501D603A90356566C93BC83372EE": "Malicious (China)",
    "C0CDBE86C539F0A8BEAD82D82DF01B4DDA6FA349": "Malicious (China)",
    "C5447EDC0FAD9C60CA43D4C40F2464F2DA9BED3D": "Malicious (China)",
    "CAF415BAC356D194DA6FE89ECCB49DF4A216EACE": "Malicious (China)",
    "CE335662F0EA6764B95C7F50A995A514ACE8C815": "Malicious (China)",
    "CF37A5B5C9166BD6B57A56BF67165A584B057241": "Malicious (China)",
    "D1F58CDB9194A374AC91DAADD411FEB99C434CC9": "Malicious (China)",
    "D8EFF6C28BB508E4702565F42748454A872BD412": "Malicious (China)",
    "E3D569137E603E7BACB6BCC66AE943850C8ADF38": "Malicious (China)",
    "F519492BD08F146837D187160222BEA26D8F8AEF": "Malicious (China)",
    "FAD4319D4E173FF3853E51C98D21919BF3DA1A1E": "Malicious (China)",
    "FAEE1F690EDD55373C35E491EF0514F3EE9445A5": "Malicious (China)",
    "FD17DA1458D3514592C4F9544A6EA960B373E877": "Malicious (China)",
    "2290C311EA0F3F57E06DF45B698E18E828E59BC3": "Malicious (China)",
    "27A29C3A8B3261770E8B59448557DC9E9339E68C": "Malicious (China)",
    "30F3B3ADC6E570BDA606B9F96DE24190CE262C67": "Malicious (China)",
    "316076F2866588DBB233C7F9EB68B58125150C21": "Malicious (China)",
    "561422647B89BE22F203EBCAEF52B5007227510A": "Malicious (China)",
    "75F411595FE9A21A17A4967C7B666E5152791A32": "Malicious (China)",
    "DC6EE6EDC4C078E1B2C12F6D1985000E27CFD318": "Malicious (China)",
    "F468B5F3FED807974476A22B32EA3137D924F7BA": "Malicious (China)",
    "F6BEADB9BC02E0A152D71C318739CDECFC1C085D": "Malicious (China)",
    "6856BB1A6C4F76DACA362187CC2CCD484EDDC25D": "Malicious (China)",
    "8EDCEE98F5788D38B8D8AD0E0C6137A6FBD1666D": "Malicious (China)",
    "AF8484379E2A512CE9074AEBD6B900ED11A70DF7": "Malicious (China)",
    "06143151E02B45DDBADD5D8E56530DAAE328CF90": "Suspect (China)",
    "67650DF17E8E7E5B8240A4F4564BCFE23D69C6F0": "Suspect (China)",
    "81AC5DE150D1B8DE5D3E0E266A136B737862D322": "Suspect (China)",
    "89C32E6B524E4D65388B9ECEDC637134ED4193A3": "Suspect (China)",
    "8EF134D4BAD5498E348A7C6D9B66F67A09D56D9B": "Suspect (China)",
    "9CBB4853F6A4F6D352A4E83252556013F5ADAF65": "Suspect (China)",
    "9D319381546EA6A12811E09CF90A20C840BE944D": "Suspect (China)",
    "AE3F2E66D48FC6BD1DF131E89D768D505DF14302": "Suspect (China)",
    "B091AA913847F313D727BCEFC8179F086F3A8C0F": "TW Government Root CA",
    "CF9E876DD3EBFC422697A3B5A37AA076A9062348": "Suspect (China)",
    "D6DAA8208D09D2154D24B52FCB346EB258B28A58": "Suspect (China)",
    "D99B104298594763F0B9A927B79269CB47DD158B": "Suspect (China)",
    "DF646DCB7B0FD3A96AEE88C64E2D676711FF9D5F": "Suspect (China)",
    "E0925E18C7765E22DABD9427529DA6AF4E066428": "Suspect (China)",
    "F48B11BFDEABBE94542071E641DE6BBE882B40B9": "Suspect (China)",
    "339D811FEC673E7F731307A34C7C7523ABBE7DFE": "Suspect (China)",
    "3F3E6C4B33802A2FEA46C5CACA14770A40018899": "Suspect (China)",
    "5B404B6DB43E1F71557F75552E7668289B1B6309": "Suspect (China)",
    "905F942FD9F28F679B378180FD4F846347F645C1": "USA Federal Government",
    "FAA7D9FB31B746F200A85E65797613D816E063B5": "Finland Government Root CA",
}
QUESTIONABLE_DV_ISSUERS: list[str] = [
    "DST Root CA X3",
    "Digital Signature Trust Co.",
    "R3",
]
VERIFIED_TRUST = "Verified trusted Root CA"
NOT_TRUSTED = "Root CA is not trusted"
LIMITED_TRUST = "Root CA has limited trust"
UNVERIFIED_TRUST = "No CA Authorization detected, is a trusted Root CA"
BROWSER_TRUSTED = "Trusted by browsers but not some platforms"
SERVER_TRUSTED = "Trusted by platforms but not some browsers"
QUESTIONABLE_TRUST = "Questionable Trust"

# https://testssl.sh/openssl-rfc.mappping.html
OPENSSL_MAP_TO_RFC: dict[str, str] = {
    "NULL-MD5": "TLS_RSA_WITH_NULL_MD5",
    "NULL-SHA": "TLS_RSA_WITH_NULL_SHA",
    "EXP-RC4-MD5": "TLS_RSA_EXPORT_WITH_RC4_40_MD5",
    "RC4-MD5": "TLS_RSA_WITH_RC4_128_MD5",
    "RC4-SHA": "TLS_RSA_WITH_RC4_128_SHA",
    "EXP-RC2-CBC-MD5": "TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5",
    "IDEA-CBC-SHA": "TLS_RSA_WITH_IDEA_CBC_SHA",
    "EXP-DES-CBC-SHA": "TLS_RSA_EXPORT_WITH_DES40_CBC_SHA",
    "DES-CBC-SHA": "TLS_RSA_WITH_DES_CBC_SHA",
    "DES-CBC3-SHA": "TLS_RSA_WITH_3DES_EDGE_CBC_SHA",
    "EXP-DH-DSS-DES-CBC-SHA": "TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA",
    "DH-DSS-DES-CBC-SHA": "TLS_DH_DSS_WITH_DES_CBC_SHA",
    "DH-DSS-DES-CBC3-SHA": "TLS_DH_DSS_WITH_3DES_EDGE_CBC_SHA",
    "EXP-DH-RSA-DES-CBC-SHA": "TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA",
    "DH-RSA-DES-CBC-SHA": "TLS_DH_RSA_WITH_DES_CBC_SHA",
    "DH-RSA-DES-CBC3-SHA": "TLS_DH_RSA_WITH_3DES_EDGE_CBC_SHA",
    "EXP-EDH-DSS-DES-CBC-SHA": "TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA",
    "EDH-DSS-DES-CBC-SHA": "TLS_DHE_DSS_WITH_DES_CBC_SHA",
    "EDH-DSS-DES-CBC3-SHA": "TLS_DHE_DSS_WITH_3DES_EDGE_CBC_SHA",
    "EXP-EDH-RSA-DES-CBC-SHA": "TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA",
    "EDH-RSA-DES-CBC-SHA": "TLS_DHE_RSA_WITH_DES_CBC_SHA",
    "EDH-RSA-DES-CBC3-SHA": "TLS_DHE_RSA_WITH_3DES_EDGE_CBC_SHA",
    "EXP-ADH-RC4-MD5": "TLS_DH_anon_EXPORT_WITH_RC4_40_MD5",
    "ADH-RC4-MD5": "TLS_DH_anon_WITH_RC4_128_MD5",
    "EXP-ADH-DES-CBC-SHA": "TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA",
    "ADH-DES-CBC-SHA": "TLS_DH_anon_WITH_DES_CBC_SHA",
    "ADH-DES-CBC3-SHA": "TLS_DH_anon_WITH_3DES_EDGE_CBC_SHA",
    "KRB5-DES-CBC-SHA": "TLS_KRB5_WITH_DES_CBC_SHA",
    "KRB5-DES-CBC3-SHA": "TLS_KRB5_WITH_3DES_EDGE_CBC_SHA",
    "KRB5-RC4-SHA": "TLS_KRB5_WITH_RC4_128_SHA",
    "KRB5-IDEA-CBC-SHA": "TLS_KRB5_WITH_IDEA_CBC_SHA",
    "KRB5-DES-CBC-MD5": "TLS_KRB5_WITH_DES_CBC_MD5",
    "KRB5-DES-CBC3-MD5": "TLS_KRB5_WITH_3DES_EDGE_CBC_MD5",
    "KRB5-RC4-MD5": "TLS_KRB5_WITH_RC4_128_MD5",
    "KRB5-IDEA-CBC-MD5": "TLS_KRB5_WITH_IDEA_CBC_MD5",
    "EXP-KRB5-DES-CBC-SHA": "TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA",
    "EXP-KRB5-RC2-CBC-SHA": "TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA",
    "EXP-KRB5-RC4-SHA": "TLS_KRB5_EXPORT_WITH_RC4_40_SHA",
    "EXP-KRB5-DES-CBC-MD5": "TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5",
    "EXP-KRB5-RC2-CBC-MD5": "TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5",
    "EXP-KRB5-RC4-MD5": "TLS_KRB5_EXPORT_WITH_RC4_40_MD5",
    "AES128-SHA": "TLS_RSA_WITH_AES_128_CBC_SHA",
    "DH-DSS-AES128-SHA": "TLS_DH_DSS_WITH_AES_128_CBC_SHA",
    "DH-RSA-AES128-SHA": "TLS_DH_RSA_WITH_AES_128_CBC_SHA",
    "DHE-DSS-AES128-SHA": "TLS_DHE_DSS_WITH_AES_128_CBC_SHA",
    "DHE-RSA-AES128-SHA": "TLS_DHE_RSA_WITH_AES_128_CBC_SHA",
    "ADH-AES128-SHA": "TLS_DH_anon_WITH_AES_128_CBC_SHA",
    "AES256-SHA": "TLS_RSA_WITH_AES_256_CBC_SHA",
    "DH-DSS-AES256-SHA": "TLS_DH_DSS_WITH_AES_256_CBC_SHA",
    "DH-RSA-AES256-SHA": "TLS_DH_RSA_WITH_AES_256_CBC_SHA",
    "DHE-DSS-AES256-SHA": "TLS_DHE_DSS_WITH_AES_256_CBC_SHA",
    "DHE-RSA-AES256-SHA": "TLS_DHE_RSA_WITH_AES_256_CBC_SHA",
    "ADH-AES256-SHA": "TLS_DH_anon_WITH_AES_256_CBC_SHA",
    "NULL-SHA256": "TLS_RSA_WITH_NULL_SHA256",
    "AES128-SHA256": "TLS_RSA_WITH_AES_128_CBC_SHA256",
    "AES256-SHA256": "TLS_RSA_WITH_AES_256_CBC_SHA256",
    "DH-DSS-AES128-SHA256": "TLS_DH_DSS_WITH_AES_128_CBC_SHA256",
    "DH-RSA-AES128-SHA256": "TLS_DH_RSA_WITH_AES_128_CBC_SHA256",
    "DHE-DSS-AES128-SHA256": "TLS_DHE_DSS_WITH_AES_128_CBC_SHA256",
    "CAMELLIA128-SHA": "TLS_RSA_WITH_CAMELLIA_128_CBC_SHA",
    "DH-DSS-CAMELLIA128-SHA": "TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA",
    "DH-RSA-CAMELLIA128-SHA": "TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA",
    "DHE-DSS-CAMELLIA128-SHA": "TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA",
    "DHE-RSA-CAMELLIA128-SHA": "TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA",
    "ADH-CAMELLIA128-SHA": "TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA",
    "EXP1024-DES-CBC-SHA": "TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA",
    "EXP1024-DHE-DSS-DES-CBC-SHA": "TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA",
    "EXP1024-RC4-SHA": "TLS_RSA_EXPORT1024_WITH_RC4_56_SHA",
    "EXP1024-RC4-MD5": "TLS_RSA_EXPORT1024_WITH_RC4_56_MD5",
    "EXP1024-RC2-CBC-MD5": "TLS_RSA_EXPORT1024_WITH_RC2_CBC_56_MD5",
    "EXP1024-DHE-DSS-RC4-SHA": "TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA",
    "DHE-DSS-RC4-SHA": "TLS_DHE_DSS_WITH_RC4_128_SHA",
    "DHE-RSA-AES128-SHA256": "TLS_DHE_RSA_WITH_AES_128_CBC_SHA256",
    "DH-DSS-AES256-SHA256": "TLS_DH_DSS_WITH_AES_256_CBC_SHA256",
    "DH-RSA-AES256-SHA256": "TLS_DH_RSA_WITH_AES_256_CBC_SHA256",
    "DHE-DSS-AES256-SHA256": "TLS_DHE_DSS_WITH_AES_256_CBC_SHA256",
    "DHE-RSA-AES256-SHA256": "TLS_DHE_RSA_WITH_AES_256_CBC_SHA256",
    "ADH-AES128-SHA256": "TLS_DH_anon_WITH_AES_128_CBC_SHA256",
    "ADH-AES256-SHA256": "TLS_DH_anon_WITH_AES_256_CBC_SHA256",
    "GHOST94-GHOST89-GHOST89": "TLS_GOSTR341094_WITH_28147_CNT_IMIT",
    "GHOST2001-GHOST89-GHOST89": "TLS_GOSTR341001_WITH_28147_CNT_IMIT",
    "CAMELLIA256-SHA": "TLS_RSA_WITH_CAMELLIA_256_CBC_SHA",
    "DH-DSS-CAMELLIA256-SHA": "TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA",
    "DH-RSA-CAMELLIA256-SHA": "TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA",
    "DHE-DSS-CAMELLIA256-SHA": "TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA",
    "DHE-RSA-CAMELLIA256-SHA": "TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA",
    "ADH-CAMELLIA256-SHA": "TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA",
    "PSK-RC4-SHA": "TLS_PSK_WITH_RC4_128_SHA",
    "PSK-3DES-EDGE-CBC-SHA": "TLS_PSK_WITH_3DES_EDGE_CBC_SHA",
    "PSK-AES128-CBC-SHA": "TLS_PSK_WITH_AES_128_CBC_SHA",
    "PSK-AES256-CBC-SHA": "TLS_PSK_WITH_AES_256_CBC_SHA",
    "RSA-PSK-RC4-SHA": "TLS_RSA_PSK_WITH_RC4_128_SHA",
    "RSA-PSK-3DES-EDGE-CBC-SHA": "TLS_RSA_PSK_WITH_3DES_EDGE_CBC_SHA",
    "RSA-PSK-AES128-CBC-SHA": "TLS_RSA_PSK_WITH_AES_128_CBC_SHA",
    "RSA-PSK-AES256-CBC-SHA": "TLS_RSA_PSK_WITH_AES_256_CBC_SHA",
    "SEED-SHA": "TLS_RSA_WITH_SEED_CBC_SHA",
    "DH-DSS-SEED-SHA": "TLS_DH_DSS_WITH_SEED_CBC_SHA",
    "DH-RSA-SEED-SHA": "TLS_DH_RSA_WITH_SEED_CBC_SHA",
    "DHE-DSS-SEED-SHA": "TLS_DHE_DSS_WITH_SEED_CBC_SHA",
    "DHE-RSA-SEED-SHA": "TLS_DHE_RSA_WITH_SEED_CBC_SHA",
    "ADH-SEED-SHA": "TLS_DH_anon_WITH_SEED_CBC_SHA",
    "AES128-GCM-SHA256": "TLS_RSA_WITH_AES_128_GCM_SHA256",
    "AES256-GCM-SHA384": "TLS_RSA_WITH_AES_256_GCM_SHA384",
    "DHE-RSA-AES128-GCM-SHA256": "TLS_DHE_RSA_WITH_AES_128_GCM_SHA256",
    "DHE-RSA-AES256-GCM-SHA384": "TLS_DHE_RSA_WITH_AES_256_GCM_SHA384",
    "DH-RSA-AES128-GCM-SHA256": "TLS_DH_RSA_WITH_AES_128_GCM_SHA256",
    "DH-RSA-AES256-GCM-SHA384": "TLS_DH_RSA_WITH_AES_256_GCM_SHA384",
    "DHE-DSS-AES128-GCM-SHA256": "TLS_DHE_DSS_WITH_AES_128_GCM_SHA256",
    "DHE-DSS-AES256-GCM-SHA384": "TLS_DHE_DSS_WITH_AES_256_GCM_SHA384",
    "DH-DSS-AES128-GCM-SHA256": "TLS_DH_DSS_WITH_AES_128_GCM_SHA256",
    "DH-DSS-AES256-GCM-SHA384": "TLS_DH_DSS_WITH_AES_256_GCM_SHA384",
    "ADH-AES128-GCM-SHA256": "TLS_DH_anon_WITH_AES_128_GCM_SHA256",
    "ADH-AES256-GCM-SHA384": "TLS_DH_anon_WITH_AES_256_GCM_SHA384",
    "CAMELLIA128-SHA256": "TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256",
    "DH-DSS-CAMELLIA128-SHA256": "TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256",
    "DH-RSA-CAMELLIA128-SHA256": "TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256",
    "DHE-DSS-CAMELLIA128-SHA256": "TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256",
    "DHE-RSA-CAMELLIA128-SHA256": "TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256",
    "ADH-CAMELLIA128-SHA256": "TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA256",
    "CAMELLIA256-SHA256": "TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256",
    "DH-DSS-CAMELLIA256-SHA256": "TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256",
    "DH-RSA-CAMELLIA256-SHA256": "TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256",
    "DHE-DSS-CAMELLIA256-SHA256": "TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256",
    "DHE-RSA-CAMELLIA256-SHA256": "TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256",
    "ADH-CAMELLIA256-SHA256": "TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA256",
    "TLS_FALLBACK_SCSV": "TLS_FALLBACK_SCSV",
    "ECDH-ECDSA-NULL-SHA": "TLS_ECDH_ECDSA_WITH_NULL_SHA",
    "ECDH-ECDSA-RC4-SHA": "TLS_ECDH_ECDSA_WITH_RC4_128_SHA",
    "ECDH-ECDSA-DES-CBC3-SHA": "TLS_ECDH_ECDSA_WITH_3DES_EDGE_CBC_SHA",
    "ECDH-ECDSA-AES128-SHA": "TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA",
    "ECDH-ECDSA-AES256-SHA": "TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA",
    "ECDHE-ECDSA-NULL-SHA": "TLS_ECDHE_ECDSA_WITH_NULL_SHA",
    "ECDHE-ECDSA-RC4-SHA": "TLS_ECDHE_ECDSA_WITH_RC4_128_SHA",
    "ECDHE-ECDSA-DES-CBC3-SHA": "TLS_ECDHE_ECDSA_WITH_3DES_EDGE_CBC_SHA",
    "ECDHE-ECDSA-AES128-SHA": "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA",
    "ECDHE-ECDSA-AES256-SHA": "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA",
    "ECDH-RSA-NULL-SHA": "TLS_ECDH_RSA_WITH_NULL_SHA",
    "ECDH-RSA-RC4-SHA": "TLS_ECDH_RSA_WITH_RC4_128_SHA",
    "ECDH-RSA-DES-CBC3-SHA": "TLS_ECDH_RSA_WITH_3DES_EDGE_CBC_SHA",
    "ECDH-RSA-AES128-SHA": "TLS_ECDH_RSA_WITH_AES_128_CBC_SHA",
    "ECDH-RSA-AES256-SHA": "TLS_ECDH_RSA_WITH_AES_256_CBC_SHA",
    "ECDHE-RSA-NULL-SHA": "TLS_ECDHE_RSA_WITH_NULL_SHA",
    "ECDHE-RSA-RC4-SHA": "TLS_ECDHE_RSA_WITH_RC4_128_SHA",
    "ECDHE-RSA-DES-CBC3-SHA": "TLS_ECDHE_RSA_WITH_3DES_EDGE_CBC_SHA",
    "ECDHE-RSA-AES128-SHA": "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA",
    "ECDHE-RSA-AES256-SHA": "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA",
    "AECDH-NULL-SHA": "TLS_ECDH_anon_WITH_NULL_SHA",
    "AECDH-RC4-SHA": "TLS_ECDH_anon_WITH_RC4_128_SHA",
    "AECDH-DES-CBC3-SHA": "TLS_ECDH_anon_WITH_3DES_EDGE_CBC_SHA",
    "AECDH-AES128-SHA": "TLS_ECDH_anon_WITH_AES_128_CBC_SHA",
    "AECDH-AES256-SHA": "TLS_ECDH_anon_WITH_AES_256_CBC_SHA",
    "SRP-3DES-EDGE-CBC-SHA": "TLS_SRP_SHA_WITH_3DES_EDGE_CBC_SHA",
    "SRP-RSA-3DES-EDGE-CBC-SHA": "TLS_SRP_SHA_RSA_WITH_3DES_EDGE_CBC_SHA",
    "SRP-DSS-3DES-EDGE-CBC-SHA": "TLS_SRP_SHA_DSS_WITH_3DES_EDGE_CBC_SHA",
    "SRP-AES-128-CBC-SHA": "TLS_SRP_SHA_WITH_AES_128_CBC_SHA",
    "SRP-RSA-AES-128-CBC-SHA": "TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA",
    "SRP-DSS-AES-128-CBC-SHA": "TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA",
    "SRP-AES-256-CBC-SHA": "TLS_SRP_SHA_WITH_AES_256_CBC_SHA",
    "SRP-RSA-AES-256-CBC-SHA": "TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA",
    "SRP-DSS-AES-256-CBC-SHA": "TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA",
    "ECDHE-ECDSA-AES128-SHA256": "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256",
    "ECDHE-ECDSA-AES256-SHA384": "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384",
    "ECDH-ECDSA-AES128-SHA256": "TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256",
    "ECDH-ECDSA-AES256-SHA384": "TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384",
    "ECDHE-RSA-AES128-SHA256": "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256",
    "ECDHE-RSA-AES256-SHA384": "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384",
    "ECDH-RSA-AES128-SHA256": "TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256",
    "ECDH-RSA-AES256-SHA384": "TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384",
    "ECDHE-ECDSA-AES128-GCM-SHA256": "TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256",
    "ECDHE-ECDSA-AES256-GCM-SHA384": "TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384",
    "ECDH-ECDSA-AES128-GCM-SHA256": "TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256",
    "ECDH-ECDSA-AES256-GCM-SHA384": "TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384",
    "ECDHE-RSA-AES128-GCM-SHA256": "TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256",
    "ECDHE-RSA-AES256-GCM-SHA384": "TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384",
    "ECDH-RSA-AES128-GCM-SHA256": "TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256",
    "ECDH-RSA-AES256-GCM-SHA384": "TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384",
    "ECDHE-ECDSA-CAMELLIA128-SHA256": "TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256",
    "ECDHE-ECDSA-CAMELLIA256-SHA384": "TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384",
    "ECDH-ECDSA-CAMELLIA128-SHA256": "TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256",
    "ECDH-ECDSA-CAMELLIA256-SHA384": "TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384",
    "ECDHE-RSA-CAMELLIA128-SHA256": "TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256",
    "ECDHE-RSA-CAMELLIA256-SHA384": "TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384",
    "ECDH-RSA-CAMELLIA128-SHA256": "TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256",
    "ECDH-RSA-CAMELLIA256-SHA384": "TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384",
    "ECDHE-RSA-CHACHA20-POLY1305": "TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256",
    "ECDHE-ECDSA-CHACHA20-POLY1305": "TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256",
    "DHE-RSA-CHACHA20-POLY1305": "TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256",
    "ECDHE-RSA-CHACHA20-POLY1305-OLD": "OLD_TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256",
    "ECDHE-ECDSA-CHACHA20-POLY1305-OLD": "OLD_TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256",
    "DHE-RSA-CHACHA20-POLY1305-OLD": "OLD_TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256",
    "DHE-RSA-DES-CBC3-SHA": "TLS_DHE_RSA_WITH_3DES_EDGE_CBC_SHA",
    "DHE-DSS-DES-CBC3-SHA": "TLS_DHE_DSS_WITH_3DES_EDGE_CBC_SHA",
    "AES128-CCM": "TLS_RSA_WITH_AES_128_CCM",
    "AES256-CCM": "TLS_RSA_WITH_AES_256_CCM",
    "DHE-RSA-AES128-CCM": "TLS_DHE_RSA_WITH_AES_128_CCM",
    "DHE-RSA-AES256-CCM": "TLS_DHE_RSA_WITH_AES_256_CCM",
    "AES128-CCM8": "TLS_RSA_WITH_AES_128_CCM_8",
    "AES256-CCM8": "TLS_RSA_WITH_AES_256_CCM_8",
    "DHE-RSA-AES128-CCM8": "TLS_DHE_RSA_WITH_AES_128_CCM_8",
    "DHE-RSA-AES256-CCM8": "TLS_DHE_RSA_WITH_AES_256_CCM_8",
    "ECDHE-ECDSA-AES128-CCM": "TLS_ECDHE_ECDSA_WITH_AES_128_CCM",
    "ECDHE-ECDSA-AES256-CCM": "TLS_ECDHE_ECDSA_WITH_AES_256_CCM",
    "ECDHE-ECDSA-AES128-CCM8": "TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8",
    "ECDHE-ECDSA-AES256-CCM8": "TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8",
    "ARIA128-GCM-SHA256": "TLS_RSA_WITH_ARIA_128_GCM_SHA256",
    "ARIA256-GCM-SHA384": "TLS_RSA_WITH_ARIA_256_GCM_SHA384",
    "DHE-DSS-ARIA128-GCM-SHA256": "TLS_DHE_DSS_WITH_ARIA_128_GCM_SHA256",
    "DHE-DSS-ARIA256-GCM-SHA384": "TLS_DHE_DSS_WITH_ARIA_256_GCM_SHA384",
    "DHE-PSK-3DES-EDGE-CBC-SHA": "TLS_DHE_PSK_WITH_3DES_EDGE_CBC_SHA",
    "DHE-PSK-AES128-CBC-SHA": "TLS_DHE_PSK_WITH_AES_128_CBC_SHA",
    "DHE-PSK-AES128-CBC-SHA256": "TLS_DHE_PSK_WITH_AES_128_CBC_SHA256",
    "DHE-PSK-AES128-CCM": "TLS_DHE_PSK_WITH_AES_128_CCM",
    "DHE-PSK-AES128-CCM8": "TLS_PSK_DHE_WITH_AES_128_CCM_8",
    "DHE-PSK-AES128-GCM-SHA256": "TLS_DHE_PSK_WITH_AES_128_GCM_SHA256",
    "DHE-PSK-AES256-CBC-SHA": "TLS_DHE_PSK_WITH_AES_256_CBC_SHA",
    "DHE-PSK-AES256-CBC-SHA384": "TLS_DHE_PSK_WITH_AES_256_CBC_SHA384",
    "DHE-PSK-AES256-CCM": "TLS_DHE_PSK_WITH_AES_256_CCM",
    "DHE-PSK-AES256-CCM8": "TLS_PSK_DHE_WITH_AES_256_CCM_8",
    "DHE-PSK-AES256-GCM-SHA384": "TLS_DHE_PSK_WITH_AES_256_GCM_SHA384",
    "DHE-PSK-ARIA128-GCM-SHA256": "TLS_DHE_PSK_WITH_ARIA_128_GCM_SHA256",
    "DHE-PSK-ARIA256-GCM-SHA384": "TLS_DHE_PSK_WITH_ARIA_256_GCM_SHA384",
    "DHE-PSK-CAMELLIA128-SHA256": "TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256",
    "DHE-PSK-CAMELLIA256-SHA384": "TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384",
    "DHE-PSK-CHACHA20-POLY1305": "TLS_DHE_PSK_WITH_CHACHA20_POLY1305_SHA256",
    "DHE-PSK-NULL-SHA": "TLS_DHE_PSK_WITH_NULL_SHA",
    "DHE-PSK-NULL-SHA256": "TLS_DHE_PSK_WITH_NULL_SHA256",
    "DHE-PSK-NULL-SHA384": "TLS_DHE_PSK_WITH_NULL_SHA384",
    "DHE-PSK-RC4-SHA": "TLS_DHE_PSK_WITH_RC4_128_SHA",
    "DHE-RSA-ARIA128-GCM-SHA256": "TLS_DHE_RSA_WITH_ARIA_128_GCM_SHA256",
    "DHE-RSA-ARIA256-GCM-SHA384": "TLS_DHE_RSA_WITH_ARIA_256_GCM_SHA384",
    "ECDHE-ARIA128-GCM-SHA256": "TLS_ECDHE_RSA_WITH_ARIA_128_GCM_SHA256",
    "ECDHE-ARIA256-GCM-SHA384": "TLS_ECDHE_RSA_WITH_ARIA_256_GCM_SHA384",
    "ECDHE-ECDSA-ARIA128-GCM-SHA256": "TLS_ECDHE_ECDSA_WITH_ARIA_128_GCM_SHA256",
    "ECDHE-ECDSA-ARIA256-GCM-SHA384": "TLS_ECDHE_ECDSA_WITH_ARIA_256_GCM_SHA384",
    "ECDHE-PSK-3DES-EDGE-CBC-SHA": "TLS_ECDHE_PSK_WITH_3DES_EDGE_CBC_SHA",
    "ECDHE-PSK-AES128-CBC-SHA": "TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA",
    "ECDHE-PSK-AES128-CBC-SHA256": "TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256",
    "ECDHE-PSK-AES256-CBC-SHA": "TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA",
    "ECDHE-PSK-AES256-CBC-SHA384": "TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384",
    "ECDHE-PSK-CAMELLIA128-SHA256": "TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256",
    "ECDHE-PSK-CAMELLIA256-SHA384": "TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384",
    "ECDHE-PSK-CHACHA20-POLY1305": "TLS_ECDHE_PSK_WITH_CHACHA20_POLY1305_SHA256",
    "ECDHE-PSK-NULL-SHA": "TLS_ECDHE_PSK_WITH_NULL_SHA",
    "ECDHE-PSK-NULL-SHA256": "TLS_ECDHE_PSK_WITH_NULL_SHA256",
    "ECDHE-PSK-NULL-SHA384": "TLS_ECDHE_PSK_WITH_NULL_SHA384",
    "ECDHE-PSK-RC4-SHA": "TLS_ECDHE_PSK_WITH_RC4_128_SHA",
    "GHOST2001-NULL-GHOST94": "TLS_GOSTR341001_WITH_NULL_GOSTR3411",
    "GHOST94-NULL-GHOST94": "TLS_GOSTR341094_WITH_NULL_GOSTR3411",
    "PSK-AES128-CBC-SHA256": "TLS_PSK_WITH_AES_128_CBC_SHA256",
    "PSK-AES128-CCM": "TLS_PSK_WITH_AES_128_CCM",
    "PSK-AES128-CCM8": "TLS_PSK_WITH_AES_128_CCM_8",
    "PSK-AES128-GCM-SHA256": "TLS_PSK_WITH_AES_128_GCM_SHA256",
    "PSK-AES256-CBC-SHA384": "TLS_PSK_WITH_AES_256_CBC_SHA384",
    "PSK-AES256-CCM": "TLS_PSK_WITH_AES_256_CCM",
    "PSK-AES256-CCM8": "TLS_PSK_WITH_AES_256_CCM_8",
    "PSK-AES256-GCM-SHA384": "TLS_PSK_WITH_AES_256_GCM_SHA384",
    "PSK-ARIA128-GCM-SHA256": "TLS_PSK_WITH_ARIA_128_GCM_SHA256",
    "PSK-ARIA256-GCM-SHA384": "TLS_PSK_WITH_ARIA_256_GCM_SHA384",
    "PSK-CAMELLIA128-SHA256": "TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256",
    "PSK-CAMELLIA256-SHA384": "TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384",
    "PSK-CHACHA20-POLY1305": "TLS_PSK_WITH_CHACHA20_POLY1305_SHA256",
    "PSK-NULL-SHA": "TLS_PSK_WITH_NULL_SHA",
    "PSK-NULL-SHA256": "TLS_PSK_WITH_NULL_SHA256",
    "PSK-NULL-SHA384": "TLS_PSK_WITH_NULL_SHA384",
    "RSA-PSK-AES128-CBC-SHA256": "TLS_RSA_PSK_WITH_AES_128_CBC_SHA256",
    "RSA-PSK-AES128-GCM-SHA256": "TLS_RSA_PSK_WITH_AES_128_GCM_SHA256",
    "RSA-PSK-AES256-CBC-SHA384": "TLS_RSA_PSK_WITH_AES_256_CBC_SHA384",
    "RSA-PSK-AES256-GCM-SHA384": "TLS_RSA_PSK_WITH_AES_256_GCM_SHA384",
    "RSA-PSK-ARIA128-GCM-SHA256": "TLS_RSA_PSK_WITH_ARIA_128_GCM_SHA256",
    "RSA-PSK-ARIA256-GCM-SHA384": "TLS_RSA_PSK_WITH_ARIA_256_GCM_SHA384",
    "RSA-PSK-CAMELLIA128-SHA256": "TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256",
    "RSA-PSK-CAMELLIA256-SHA384": "TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384",
    "RSA-PSK-CHACHA20-POLY1305": "TLS_RSA_PSK_WITH_CHACHA20_POLY1305_SHA256",
    "RSA-PSK-NULL-SHA": "TLS_RSA_PSK_WITH_NULL_SHA",
    "RSA-PSK-NULL-SHA256": "TLS_RSA_PSK_WITH_NULL_SHA256",
    "RSA-PSK-NULL-SHA384": "TLS_RSA_PSK_WITH_NULL_SHA384",
}
KEY_SIZE_MAPPING: dict[str, int] = {
    "TLS_RSA_WITH_NULL_MD5": 0,
    "TLS_RSA_WITH_NULL_SHA": 0,
    "TLS_RSA_WITH_AES_128_CBC_SHA": 128,
    "TLS_DHE_DSS_WITH_AES_128_CBC_SHA": 128,
    "TLS_DHE_RSA_WITH_AES_128_CBC_SHA": 128,
    "TLS_DH_anon_WITH_AES_128_CBC_SHA": 128,
    "TLS_RSA_WITH_AES_256_CBC_SHA": 256,
    "TLS_DHE_DSS_WITH_AES_256_CBC_SHA": 256,
    "TLS_DHE_RSA_WITH_AES_256_CBC_SHA": 256,
    "TLS_DH_anon_WITH_AES_256_CBC_SHA": 256,
    "TLS_RSA_WITH_NULL_SHA256": 0,
    "TLS_RSA_WITH_AES_128_CBC_SHA256": 128,
    "TLS_RSA_WITH_AES_256_CBC_SHA256": 256,
    "TLS_DHE_DSS_WITH_AES_128_CBC_SHA256": 128,
    "TLS_RSA_WITH_CAMELLIA_128_CBC_SHA": 128,
    "TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA": 128,
    "TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA": 128,
    "TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA": 128,
    "TLS_DHE_RSA_WITH_AES_128_CBC_SHA256": 128,
    "TLS_DHE_DSS_WITH_AES_256_CBC_SHA256": 256,
    "TLS_DHE_RSA_WITH_AES_256_CBC_SHA256": 256,
    "TLS_DH_anon_WITH_AES_128_CBC_SHA256": 128,
    "TLS_DH_anon_WITH_AES_256_CBC_SHA256": 256,
    "TLS_RSA_WITH_CAMELLIA_256_CBC_SHA": 256,
    "TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA": 256,
    "TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA": 256,
    "TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA": 256,
    "TLS_PSK_WITH_AES_128_CBC_SHA": 128,
    "TLS_PSK_WITH_AES_256_CBC_SHA": 256,
    "TLS_RSA_PSK_WITH_AES_128_CBC_SHA": 128,
    "TLS_RSA_PSK_WITH_AES_256_CBC_SHA": 256,
    "TLS_RSA_WITH_SEED_CBC_SHA": 128,
    "TLS_DHE_DSS_WITH_SEED_CBC_SHA": 128,
    "TLS_DHE_RSA_WITH_SEED_CBC_SHA": 128,
    "TLS_DH_anon_WITH_SEED_CBC_SHA": 128,
    "TLS_RSA_WITH_AES_128_GCM_SHA256": 128,
    "TLS_RSA_WITH_AES_256_GCM_SHA384": 256,
    "TLS_DHE_RSA_WITH_AES_128_GCM_SHA256": 128,
    "TLS_DHE_RSA_WITH_AES_256_GCM_SHA384": 256,
    "TLS_DHE_DSS_WITH_AES_128_GCM_SHA256": 128,
    "TLS_DHE_DSS_WITH_AES_256_GCM_SHA384": 256,
    "TLS_DH_anon_WITH_AES_128_GCM_SHA256": 128,
    "TLS_DH_anon_WITH_AES_256_GCM_SHA384": 256,
    "TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256": 128,
    "TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256": 128,
    "TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256": 128,
    "TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA256": 128,
    "TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256": 256,
    "TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256": 256,
    "TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256": 256,
    "TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA256": 256,
    "TLS_ECDHE_ECDSA_WITH_NULL_SHA": 0,
    "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA": 128,
    "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA": 256,
    "TLS_ECDHE_RSA_WITH_NULL_SHA": 0,
    "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA": 128,
    "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA": 256,
    "TLS_ECDH_anon_WITH_NULL_SHA": 0,
    "TLS_ECDH_anon_WITH_AES_128_CBC_SHA": 128,
    "TLS_ECDH_anon_WITH_AES_256_CBC_SHA": 256,
    "TLS_SRP_SHA_WITH_AES_128_CBC_SHA": 128,
    "TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA": 128,
    "TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA": 128,
    "TLS_SRP_SHA_WITH_AES_256_CBC_SHA": 256,
    "TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA": 256,
    "TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA": 256,
    "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256": 128,
    "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384": 256,
    "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256": 128,
    "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384": 256,
    "TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256": 128,
    "TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384": 256,
    "TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256": 128,
    "TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384": 256,
    "TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256": 128,
    "TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384": 256,
    "TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256": 128,
    "TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384": 256,
    "TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256": 256,
    "TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256": 256,
    "TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256": 256,
    "RSA_WITH_AES_128_CCM": 128,
    "RSA_WITH_AES_256_CCM": 256,
    "DHE_RSA_WITH_AES_128_CCM": 128,
    "TLS_DHE_RSA_WITH_AES_256_CCM": 256,
    "RSA_WITH_AES_128_CCM_8": 128,
    "RSA_WITH_AES_256_CCM_8": 256,
    "DHE_RSA_WITH_AES_128_CCM_8": 128,
    "DHE_RSA_WITH_AES_256_CCM_8": 256,
    "ECDHE_ECDSA_WITH_AES_128_CCM": 128,
    "ECDHE_ECDSA_WITH_AES_256_CCM": 256,
    "ECDHE_ECDSA_WITH_AES_128_CCM_8": 128,
    "ECDHE_ECDSA_WITH_AES_256_CCM_8": 256,
    "TLS_RSA_WITH_RC4_128_SHA": 128,
    "TLS_RSA_WITH_IDEA_CBC_SHA": 128,
    "TLS_RSA_EXPORT_WITH_DES40_CBC_SHA": 40,
    "TLS_RSA_WITH_DES_CBC_SHA": 56,
    "TLS_RSA_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_DH_DSS_WITH_DES_CBC_SHA": 56,
    "TLS_DH_DSS_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_DH_RSA_WITH_DES_CBC_SHA": 56,
    "TLS_DH_RSA_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA": 40,
    "TLS_DHE_DSS_WITH_DES_CBC_SHA": 56,
    "TLS_DHE_DSS_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA": 40,
    "TLS_DHE_RSA_WITH_DES_CBC_SHA": 56,
    "TLS_DHE_RSA_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_DH_anon_EXPORT_WITH_RC4_40_MD5": 40,
    "TLS_DH_anon_WITH_RC4_128_MD5": 128,
    "TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA": 40,
    "TLS_DH_anon_WITH_DES_CBC_SHA": 56,
    "TLS_DH_anon_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_DH_DSS_WITH_AES_128_CBC_SHA": 128,
    "TLS_DH_RSA_WITH_AES_128_CBC_SHA": 128,
    "TLS_DH_DSS_WITH_AES_256_CBC_SHA": 256,
    "TLS_DH_RSA_WITH_AES_256_CBC_SHA": 256,
    "TLS_DH_DSS_WITH_AES_128_CBC_SHA256": 128,
    "TLS_DH_RSA_WITH_AES_128_CBC_SHA256": 128,
    "TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA": 128,
    "TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA": 128,
    "TLS_DH_DSS_WITH_AES_256_CBC_SHA256": 256,
    "TLS_DH_RSA_WITH_AES_256_CBC_SHA256": 256,
    "TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA": 256,
    "TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA": 256,
    "TLS_PSK_WITH_RC4_128_SHA": 128,
    "TLS_PSK_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_DH_DSS_WITH_SEED_CBC_SHA": 128,
    "TLS_DH_RSA_WITH_SEED_CBC_SHA": 128,
    "TLS_DH_RSA_WITH_AES_128_GCM_SHA256": 128,
    "TLS_DH_RSA_WITH_AES_256_GCM_SHA384": 256,
    "TLS_DH_DSS_WITH_AES_128_GCM_SHA256": 128,
    "TLS_DH_DSS_WITH_AES_256_GCM_SHA384": 256,
    "TLS_ECDH_ECDSA_WITH_NULL_SHA": 0,
    "TLS_ECDH_ECDSA_WITH_RC4_128_SHA": 128,
    "TLS_ECDH_ECDSA_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA": 128,
    "TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA": 256,
    "TLS_ECDHE_ECDSA_WITH_RC4_128_SHA": 128,
    "TLS_ECDHE_ECDSA_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_ECDH_RSA_WITH_NULL_SHA": 0,
    "TLS_ECDH_RSA_WITH_RC4_128_SHA": 128,
    "TLS_ECDH_RSA_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_ECDH_RSA_WITH_AES_128_CBC_SHA": 128,
    "TLS_ECDH_RSA_WITH_AES_256_CBC_SHA": 256,
    "TLS_ECDHE_RSA_WITH_RC4_128_SHA": 128,
    "TLS_ECDHE_RSA_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_ECDH_anon_WITH_RC4_128_SHA": 128,
    "TLS_ECDH_anon_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_SRP_SHA_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_SRP_SHA_RSA_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_SRP_SHA_DSS_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256": 128,
    "TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384": 256,
    "TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256": 128,
    "TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384": 256,
    "TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256": 128,
    "TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384": 256,
    "TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256": 128,
    "TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384": 256,
    "TLS_RSA_EXPORT_WITH_RC4_40_MD5": 40,
    "TLS_RSA_WITH_RC4_128_MD5": 128,
    "TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5": 40,
    "TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA": 40,
    "TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA": 40,
    "TLS_KRB5_WITH_RC4_128_SHA": 128,
    "TLS_KRB5_WITH_RC4_128_MD5": 128,
    "TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA": 40,
    "TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA": 40,
    "TLS_KRB5_EXPORT_WITH_RC4_40_SHA": 40,
    "TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5": 40,
    "TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5": 40,
    "TLS_KRB5_EXPORT_WITH_RC4_40_MD5": 40,
    "TLS_RSA_EXPORT1024_WITH_RC4_56_SHA": 56,
    "TLS_RSA_EXPORT1024_WITH_RC4_56_MD5": 56,
    "TLS_RSA_EXPORT1024_WITH_RC2_CBC_56_MD5": 56,
    "TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA": 56,
    "TLS_DHE_DSS_WITH_RC4_128_SHA": 128,
    "TLS_RSA_PSK_WITH_RC4_128_SHA": 128,
    "TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256": 128,
    "TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256": 128,
    "TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256": 256,
    "TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256": 256,
    "TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256": 128,
    "TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384": 256,
    "TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256": 128,
    "TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384": 256,
    "OLD_TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256": 256,
    "OLD_TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256": 256,
    "OLD_TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256": 256,
    "TLS_KRB5_WITH_DES_CBC_SHA": 56,
    "TLS_KRB5_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_KRB5_WITH_IDEA_CBC_SHA": 128,
    "TLS_KRB5_WITH_DES_CBC_MD5": 56,
    "TLS_KRB5_WITH_3DES_EDGE_CBC_MD5": 168,
    "TLS_KRB5_WITH_IDEA_CBC_MD5": 128,
    "TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA": 56,
    "TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA": 56,
    "TLS_GOSTR341094_WITH_28147_CNT_IMIT": 256,
    "TLS_GOSTR341001_WITH_28147_CNT_IMIT": 256,
    "TLS_RSA_PSK_WITH_3DES_EDGE_CBC_SHA": 168,
    "TLS_AES_256_GCM_SHA384": 256,
    "TLS_CHACHA20_POLY1305_SHA256": 256,
    "TLS_AES_128_GCM_SHA256": 128,
    "TLS_ECDHE_ECDSA_WITH_ARIA_256_GCM_SHA384": 256,
    "TLS_ECDHE_RSA_WITH_ARIA_256_GCM_SHA384": 256,
    "TLS_DHE_DSS_WITH_ARIA_256_GCM_SHA384": 256,
    "TLS_DHE_RSA_WITH_ARIA_256_GCM_SHA384": 256,
    "TLS_ECDHE_ECDSA_WITH_ARIA_128_GCM_SHA256": 128,
    "TLS_ECDHE_RSA_WITH_ARIA_128_GCM_SHA256": 128,
    "TLS_DHE_DSS_WITH_ARIA_128_GCM_SHA256": 128,
    "TLS_DHE_RSA_WITH_ARIA_128_GCM_SHA256": 256,
    "TLS_RSA_WITH_ARIA_256_GCM_SHA384": 256,
    "TLS_RSA_WITH_ARIA_128_GCM_SHA256": 128,
    "TLS_RSA_WITH_AES_256_CCM": 256,
    "TLS_ECDHE_ECDSA_WITH_AES_128_CCM": 128,
    "TLS_DHE_RSA_WITH_AES_128_CCM": 128,
    "TLS_RSA_WITH_AES_128_CCM": 128,
    "TLS_RSA_WITH_AES_256_CCM_8": 128,
    "TLS_ECDHE_ECDSA_WITH_AES_256_CCM": 256,
    "TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8": 256,
    "TLS_DHE_RSA_WITH_AES_128_CCM_8": 128,
    "TLS_DHE_RSA_WITH_AES_256_CCM_8": 256,
    "TLS_RSA_WITH_AES_128_CCM_8": 128,
    "TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8": 128,
    "TLS_AES_128_CCM_8_SHA256": 128,
    "TLS_AES_128_CCM_SHA256": 128,
    "SSL_CK_RC4_128_WITH_MD5": 128,
    "SSL_CK_RC4_128_EXPORT40_WITH_MD5": 40,
    "SSL_CK_RC2_128_CBC_WITH_MD5": 128,
    "SSL_CK_RC2_128_CBC_EXPORT40_WITH_MD5": 40,
    "SSL_CK_IDEA_128_CBC_WITH_MD5": 128,
    "SSL_CK_DES_64_CBC_WITH_MD5": 56,
    "SSL_CK_DES_192_EDGE3_CBC_WITH_MD5": 168,
    "SSL_CK_RC4_64_WITH_MD5": 64,
}

RESULT_LEVEL_FAIL = "fail"
RESULT_LEVEL_WARN = "warn"
RESULT_LEVEL_PASS = "pass"  # nosec
RESULT_LEVEL_INFO = "info"
RESULT_LEVEL_FAIL_DEFAULT = "FAIL!"
RESULT_LEVEL_WARN_DEFAULT = "WARN!"
RESULT_LEVEL_PASS_DEFAULT = "PASS!"  # nosec
RESULT_LEVEL_INFO_DEFAULT = "INFO!"
CLI_COLOR_PRIMARY = "cyan"
CLI_COLOR_FAIL = "light_coral"
CLI_COLOR_WARN = "khaki1"
CLI_COLOR_PASS = "dark_sea_green2"  # nosec
CLI_COLOR_INFO = "deep_sky_blue2"
CLI_COLOR_MAP = {
    RESULT_LEVEL_FAIL: CLI_COLOR_FAIL,
    RESULT_LEVEL_WARN: CLI_COLOR_WARN,
    RESULT_LEVEL_PASS: CLI_COLOR_PASS,
    RESULT_LEVEL_INFO: CLI_COLOR_INFO,
}
CLI_ICON_MAP = {
    RESULT_LEVEL_FAIL: ":cross_mark:",
    RESULT_LEVEL_WARN: ":bell:",
    RESULT_LEVEL_PASS: ":white_heavy_check_mark:",
    RESULT_LEVEL_INFO: ":speech_balloon:",
}
DEFAULT_MAP = {
    RESULT_LEVEL_FAIL: RESULT_LEVEL_FAIL_DEFAULT,
    RESULT_LEVEL_WARN: RESULT_LEVEL_WARN_DEFAULT,
    RESULT_LEVEL_PASS: RESULT_LEVEL_PASS_DEFAULT,
    RESULT_LEVEL_INFO: RESULT_LEVEL_INFO_DEFAULT,
}
