import datetime
import decimal
import enum
import uuid
from typing import Any, Callable, Iterable, Type, cast

import marshmallow as m
import marshmallow.validate

from .missing import MISSING, Missing

_MARSHMALLOW_VERSION_MAJOR = int(m.__version__.split(".")[0])


def str_field(
    *,
    required: bool,
    default: str | None | Missing = MISSING,
    name: str | None = None,
    **_: Any,
) -> m.fields.Field:
    if required:
        if default is not MISSING:
            raise ValueError("Default values is not supported for required fields")

        return m.fields.String(required=True, **data_key_fields(name))

    return m.fields.Str(
        allow_none=True,
        **default_fields(None if default is MISSING else default),
        **data_key_fields(name),
    )


def bool_field(
    *,
    required: bool,
    default: bool | None | Missing = MISSING,
    name: str | None = None,
    **_: Any,
) -> m.fields.Field:
    if required:
        if default is not MISSING:
            raise ValueError("Default values is not supported for required fields")

        return m.fields.Boolean(required=True, **data_key_fields(name))

    return m.fields.Bool(
        allow_none=True,
        **default_fields(None if default is MISSING else default),
        **data_key_fields(name),
    )


def decimal_field(
    *,
    required: bool,
    default: decimal.Decimal | None | Missing = MISSING,
    name: str | None = None,
    places: int = 2,
    as_string: bool = True,
    **_: Any,
) -> m.fields.Field:
    if required:
        if default is not MISSING:
            raise ValueError("Default values is not supported for required fields")
        return m.fields.Decimal(required=True, as_string=as_string, places=places, **data_key_fields(name))

    return m.fields.Decimal(
        allow_none=True,
        as_string=as_string,
        places=places,
        **default_fields(None if default is MISSING else default),
        **data_key_fields(name),
    )


def int_field(
    *,
    required: bool,
    default: int | None | Missing = MISSING,
    name: str | None = None,
    **_: Any,
) -> m.fields.Field:
    if required:
        if default is not MISSING:
            raise ValueError("Default values is not supported for required fields")
        return m.fields.Int(required=True, **data_key_fields(name))

    return m.fields.Int(
        allow_none=True,
        **default_fields(None if default is MISSING else default),
        **data_key_fields(name),
    )


def float_field(
    *,
    required: bool,
    default: float | None | Missing = MISSING,
    name: str | None = None,
    **_: Any,
) -> m.fields.Field:
    if required:
        if default is not MISSING:
            raise ValueError("Default values is not supported for required fields")
        return m.fields.Float(required=True, **data_key_fields(name))

    return m.fields.Float(
        allow_none=True,
        **default_fields(None if default is MISSING else default),
        **data_key_fields(name),
    )


def uuid_field(
    *,
    required: bool,
    default: uuid.UUID | None | Missing = MISSING,
    name: str | None = None,
    **_: Any,
) -> m.fields.Field:
    if required:
        if default is not MISSING:
            raise ValueError("Default values is not supported for required fields")
        return m.fields.UUID(required=True, **data_key_fields(name))

    return m.fields.UUID(
        allow_none=True,
        **default_fields(None if default is MISSING else default),
        **data_key_fields(name),
    )


def datetime_field(
    *,
    required: bool,
    default: datetime.datetime | None | Missing = MISSING,
    name: str | None = None,
    **_: Any,
) -> m.fields.Field:
    if required:
        if default is not MISSING:
            raise ValueError("Default values is not supported for required fields")
        return DateTimeField(required=True, **data_key_fields(name))

    return DateTimeField(
        allow_none=True,
        **default_fields(None if default is MISSING else default),
        **data_key_fields(name),
    )


def date_field(
    *,
    required: bool,
    default: datetime.date | None | Missing = MISSING,
    name: str | None = None,
    **_: Any,
) -> m.fields.Field:
    if required:
        if default is not MISSING:
            raise ValueError("Default values is not supported for required fields")
        return m.fields.Date(required=True, **data_key_fields(name))

    return m.fields.Date(
        allow_none=True,
        **default_fields(None if default is MISSING else default),
        **data_key_fields(name),
    )


def nested_field(
    nested_schema: Type[m.Schema],
    *,
    required: bool,
    default: Any | None | Missing = MISSING,
    name: str | None = None,
    **_: Any,
) -> m.fields.Field:
    if required:
        if default is not MISSING:
            raise ValueError("Default values is not supported for required fields")
        return m.fields.Nested(nested_schema, required=True, **data_key_fields(name))

    if default is not MISSING and default is not None:
        raise ValueError("Default values is not supported for required fields")
    return m.fields.Nested(
        nested_schema,
        allow_none=True,
        **default_fields(None),
        **data_key_fields(name),
    )


def list_field(
    field: m.fields.Field,
    *,
    required: bool,
    default: Any | None | Missing = MISSING,
    name: str | None = None,
    **_: Any,
) -> m.fields.Field:
    if required:
        if default is not MISSING:
            raise ValueError("Default values is not supported for required fields")
        return m.fields.List(field, required=True, **data_key_fields(name))

    if default is not MISSING and default is not None:
        raise ValueError("Default values is not supported for required fields")
    return m.fields.List(
        field,
        allow_none=True,
        **default_fields(None),
        **data_key_fields(name),
    )


def dict_field(
    *,
    required: bool,
    default: Any | None | Missing = MISSING,
    name: str | None = None,
    **_: Any,
) -> m.fields.Field:
    if required:
        if default is not MISSING:
            raise ValueError("Default values is not supported for required fields")
        return m.fields.Dict(required=True, **data_key_fields(name))

    if default is not MISSING and default is not None:
        raise ValueError("Default values is not supported for required fields")
    return m.fields.Dict(
        allow_none=True,
        **default_fields(None),
        **data_key_fields(name),
    )


def enum_field(
    enum_type: Type[enum.Enum],
    *,
    required: bool,
    name: str | None = None,
    default: Any | None | Missing = MISSING,
) -> marshmallow.fields.Field:
    if required:
        if default is not MISSING:
            raise ValueError("Default values is not supported for required fields")
        return EnumField(
            enum_type=enum_type,
            required=True,
            **data_key_fields(name),
        )

    return EnumField(
        enum_type=enum_type,
        allow_none=True,
        **default_fields(None if default is MISSING else default),
        **data_key_fields(name),
    )


DateTimeField: Type[m.fields.DateTime]
EnumField: Type[m.fields.String]

if _MARSHMALLOW_VERSION_MAJOR >= 3:

    def data_key_fields(name: str | None) -> dict[str, Any]:
        if name is None:
            return {}
        return dict(data_key=name)

    def default_fields(value: Any) -> dict[str, Any]:
        return dict(dump_default=value, load_default=value)

    class DateTimeFieldV3(m.fields.DateTime):
        def _deserialize(self, value: Any, attr: Any, data: Any, **kwargs: Any) -> Any:
            result = super()._deserialize(value, attr, data, **kwargs)
            if result.tzinfo is None:
                return result.replace(tzinfo=datetime.timezone.utc)
            return result.astimezone(datetime.timezone.utc)

        def _serialize(self, value: Any, attr: Any, obj: Any, **kwargs: Any) -> Any:
            if value is None:
                return None

            if value.tzinfo is None:
                value = value.replace(tzinfo=datetime.timezone.utc)

            return super()._serialize(value, attr, obj, **kwargs)

    DateTimeField = DateTimeFieldV3

    class EnumFieldV3(m.fields.String):
        default_error = "Not a valid choice: '{input}'. Allowed values: {choices}"

        def __init__(
            self,
            *args: Any,
            enum_type: Type[enum.Enum],
            error: str | None = None,
            extendable_default: Any = m.missing,
            **kwargs: Any,
        ):
            """
            :param enum_type: class inherited from Enum and string, where all values are different strings
            :param error: error string pattern with {input} and {choices}
            """
            allow_none = (
                kwargs.get("allow_none") is True
                or kwargs.get("allow_none") is None
                and kwargs.get("missing", m.missing) is None
            )

            self.enum_type = enum_type
            self._validate_enum(self.enum_type)

            self.error = error or EnumFieldV3.default_error
            self._validate_error(self.error)

            self.choices = [enum_instance.value for enum_instance in cast(Iterable[enum.Enum], enum_type)]
            self._validate_choices(self.choices)
            if allow_none:
                self.choices.append(None)

            self.extendable_default = extendable_default
            self._validate_default(self.enum_type, self.extendable_default, allow_none)
            if "default" in kwargs:
                self._validate_default(self.enum_type, kwargs["default"], allow_none)
            if "missing" in kwargs:
                self._validate_default(self.enum_type, kwargs["missing"], allow_none)

            enum_validator = m.validate.OneOf(self.choices, error=self.error)
            if "validate" in kwargs and kwargs["validate"] is not None:
                validators = kwargs["validate"]
                if not isinstance(validators, list):
                    validators = [validators]
                validators.append(enum_validator)
                kwargs["validate"] = validators
            else:
                kwargs["validate"] = enum_validator

            super().__init__(*args, **kwargs)

        def _serialize(self, value: Any, attr: Any, obj: Any, **kwargs: Any) -> Any:
            if value is None:
                return None
            if isinstance(value, self.enum_type):
                return cast(enum.Enum, value).value
            return super()._serialize(value, attr, obj)

        def _deserialize(self, value: Any, attr: Any, data: Any, **kwargs: Any) -> Any:
            if value is None:
                return None
            if isinstance(value, self.enum_type):
                return value
            string_value = super()._deserialize(value, attr, data)
            try:
                return cast(Callable[[str], enum.Enum], self.enum_type)(string_value)
            except ValueError:
                if self.extendable_default is m.missing:
                    raise m.ValidationError(self.default_error.format(input=value, choices=self.choices))
                return self.extendable_default
            except Exception:
                raise m.ValidationError(self.default_error.format(input=value, choices=self.choices))

        @staticmethod
        def _validate_enum(enum_type: Any) -> None:
            if not issubclass(enum_type, enum.Enum):
                raise ValueError(f"Enum type {enum_type} should be subtype of Enum")
            if not issubclass(enum_type, str):
                raise ValueError(f"Enum type {enum_type} should be subtype of str")

        @staticmethod
        def _validate_error(error: str) -> None:
            try:
                error.format(input="", choices="")
            except KeyError:
                raise ValueError("Error should contain only {{input}} and {{choices}}'")

        @staticmethod
        def _validate_choices(choices: list) -> None:
            for choice in choices:
                if not isinstance(choice, str):
                    raise ValueError(f"There is enum value, which is not a string: {choice}")

        @staticmethod
        def _validate_default(enum_type: Any, default: enum.Enum | None | Missing, allow_none: bool) -> None:
            if default is m.missing:
                return

            if allow_none and default is None:
                return

            if not isinstance(default, enum_type):
                raise ValueError(f"Default should be an instance of enum_type {enum_type}")

    EnumField = EnumFieldV3
else:
    dateutil_tz_utc_cls: Type[datetime.tzinfo] | None
    try:
        import dateutil.tz  # type: ignore

        dateutil_tz_utc_cls = dateutil.tz.tzutc
    except ImportError:
        dateutil_tz_utc_cls = None

    def data_key_fields(name: str | None) -> dict[str, Any]:
        if name is None:
            return {}
        return dict(dump_to=name, load_from=name)

    def default_fields(value: Any) -> dict[str, Any]:
        return dict(missing=value, default=value)

    class DateTimeFieldV2(m.fields.DateTime):
        def _deserialize(self, value: Any, attr: Any, data: Any, **_: Any) -> Any:
            result = super()._deserialize(value, attr, data)
            if result.tzinfo is None:
                return result.replace(tzinfo=datetime.timezone.utc)
            if dateutil_tz_utc_cls is not None and isinstance(result.tzinfo, dateutil_tz_utc_cls):
                return result.replace(tzinfo=datetime.timezone.utc)
            return result.astimezone(datetime.timezone.utc)

    DateTimeField = DateTimeFieldV2

    class EnumFieldV2(m.fields.String):
        default_error = "Not a valid choice: '{input}'. Allowed values: {choices}"

        def __init__(
            self,
            *args: Any,
            enum_type: Type[enum.Enum],
            error: str | None = None,
            extendable_default: Any = m.missing,
            **kwargs: Any,
        ):
            """
            :param enum_type: class inherited from Enum and string, where all values are different strings
            :param error: error string pattern with {input} and {choices}
            """
            allow_none = (
                kwargs.get("allow_none") is True
                or kwargs.get("allow_none") is None
                and kwargs.get("missing", m.missing) is None
            )

            self.enum_type = enum_type
            self._validate_enum(self.enum_type)

            self.error = error or EnumFieldV2.default_error
            self._validate_error(self.error)

            self.choices = [enum_instance.value for enum_instance in cast(Iterable[enum.Enum], enum_type)]
            self._validate_choices(self.choices)
            if allow_none:
                self.choices.append(None)

            self.extendable_default = extendable_default
            self._validate_default(self.enum_type, self.extendable_default, allow_none)
            if "default" in kwargs:
                self._validate_default(self.enum_type, kwargs["default"], allow_none)
            if "missing" in kwargs:
                self._validate_default(self.enum_type, kwargs["missing"], allow_none)

            enum_validator = m.validate.OneOf(self.choices, error=self.error)
            if "validate" in kwargs and kwargs["validate"] is not None:
                validators = kwargs["validate"]
                if not isinstance(validators, list):
                    validators = [validators]
                validators.append(enum_validator)
                kwargs["validate"] = validators
            else:
                kwargs["validate"] = enum_validator

            super().__init__(*args, **kwargs)

        def _serialize(self, value: Any, attr: Any, obj: Any, **kwargs: Any) -> Any:
            if value is None:
                return None
            if isinstance(value, self.enum_type):
                return cast(enum.Enum, value).value
            return super()._serialize(value, attr, obj)

        def _deserialize(self, value: Any, attr: Any, data: Any, **kwargs: Any) -> Any:
            if value is None:
                return None
            if isinstance(value, self.enum_type):
                return value
            string_value = super()._deserialize(value, attr, data)
            try:
                return cast(Callable[[str], enum.Enum], self.enum_type)(string_value)
            except ValueError:
                if self.extendable_default is m.missing:
                    raise m.ValidationError(self.default_error.format(input=value, choices=self.choices))
                return self.extendable_default
            except Exception:
                raise m.ValidationError(self.default_error.format(input=value, choices=self.choices))

        @staticmethod
        def _validate_enum(enum_type: Any) -> None:
            if not issubclass(enum_type, enum.Enum):
                raise ValueError(f"Enum type {enum_type} should be subtype of Enum")
            if not issubclass(enum_type, str):
                raise ValueError(f"Enum type {enum_type} should be subtype of str")

        @staticmethod
        def _validate_error(error: str) -> None:
            try:
                error.format(input="", choices="")
            except KeyError:
                raise ValueError("Error should contain only {{input}} and {{choices}}'")

        @staticmethod
        def _validate_choices(choices: list) -> None:
            for choice in choices:
                if not isinstance(choice, str):
                    raise ValueError(f"There is enum value, which is not a string: {choice}")

        @staticmethod
        def _validate_default(enum_type: Any, default: enum.Enum | None | Missing, allow_none: bool) -> None:
            if default is m.missing:
                return

            if allow_none and default is None:
                return

            if not isinstance(default, enum_type):
                raise ValueError(f"Default should be an instance of enum_type {enum_type}")

    EnumField = EnumFieldV2
