"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const sqs = require("aws-cdk-lib/aws-sqs");
const kms = require("aws-cdk-lib/aws-kms");
const defaults = require("@aws-solutions-constructs/core");
// --------------------------------------------------------------
// Pattern deployment with default props
// --------------------------------------------------------------
test('Pattern deployment with default props', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a default sqs queue
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testiotsqsEncryptionKey64EE64B1",
                "Arn"
            ]
        }
    });
    // Creates a dead letter queue
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    template.hasResourceProperties("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Creates an encryption key
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: true
    });
});
// --------------------------------------------------------------
// Testing with existing SQS Queue
// --------------------------------------------------------------
test('Pattern deployment with existing queue', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj'
    });
    const props = {
        existingQueueObj: queue,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a default sqs queue
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: "existing-queue-obj"
    });
});
// --------------------------------------------------------------
// Testing with passing queue and dead letter queue props
// --------------------------------------------------------------
test('Pattern deployment with queue and dead letter queue props', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        deadLetterQueueProps: {
            queueName: 'dlq-name'
        },
        queueProps: {
            queueName: 'queue-name'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a queue using the provided props
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: "queue-name",
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "testiotsqsdeadLetterQueue66A04E81",
                    "Arn",
                ],
            },
            maxReceiveCount: 15
        }
    });
    // Creates a dead letter queue using the provided props
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: "dlq-name"
    });
});
// --------------------------------------------------------------
// Testing with dead letter queue turned off
// --------------------------------------------------------------
test('Pattern deployment with dead letter queue turned off', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        deployDeadLetterQueue: false,
        queueProps: {
            queueName: 'queue-name'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a queue using the provided props
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: "queue-name"
    });
    // Does not create the default dead letter queue
    defaults.expectNonexistence(stack, "AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
});
// --------------------------------------------------------------
// Testing with custom maxReceiveCount
// --------------------------------------------------------------
test('Pattern deployment with custom maxReceiveCount', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        deadLetterQueueProps: {
            queueName: 'dlq-name'
        },
        deployDeadLetterQueue: true,
        maxReceiveCount: 1,
        queueProps: {
            queueName: 'queue-name'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a queue using the provided props
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: "queue-name",
        RedrivePolicy: {
            deadLetterTargetArn: {
                "Fn::GetAtt": [
                    "testiotsqsdeadLetterQueue66A04E81",
                    "Arn",
                ],
            },
            maxReceiveCount: 1
        },
    });
});
// --------------------------------------------------------------
// Testing without creating a KMS key
// --------------------------------------------------------------
test('Pattern deployment without creating a KMS key', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        enableEncryptionWithCustomerManagedKey: false,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a default sqs queue
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates a dead letter queue
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    template.hasResourceProperties("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Does not create an encryption key
    template.resourceCountIs("AWS::KMS::Key", 0);
});
// --------------------------------------------------------------
// Testing with existing KMS key
// --------------------------------------------------------------
test('Pattern deployment with existing KMS key', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const kmsKey = new kms.Key(stack, 'existing-key', {
        enableKeyRotation: false,
        alias: 'existing-key-alias'
    });
    const props = {
        encryptionKey: kmsKey,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a default sqs queue
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "existingkey205DFC01",
                "Arn"
            ]
        }
    });
    // Creates a dead letter queue
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    template.hasResourceProperties("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Uses the provided key
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
});
// ---------------------------------------------------------------
// Testing with existing KMS key on queueProps.encryptionMasterKey
// ---------------------------------------------------------------
test('Pattern deployment with existing KMS key', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const kmsKey = new kms.Key(stack, 'existing-key', {
        enableKeyRotation: false,
        alias: 'existing-key-alias'
    });
    const props = {
        queueProps: {
            encryptionMasterKey: kmsKey
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a default sqs queue
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "existingkey205DFC01",
                "Arn"
            ]
        }
    });
    // Creates a dead letter queue
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    template.hasResourceProperties("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Uses the provided key
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
});
// --------------------------------------------------------------
// Testing with passing KMS key props
// --------------------------------------------------------------
test('Pattern deployment passing KMS key props', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        encryptionKeyProps: {
            enableKeyRotation: false,
            alias: 'new-key-alias-from-props'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    // Creates a default sqs queue
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testiotsqsEncryptionKey64EE64B1",
                "Arn"
            ]
        }
    });
    // Creates a dead letter queue
    template.hasResourceProperties("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
    // Creates an IoT Topic Rule
    template.hasResourceProperties("AWS::IoT::TopicRule", {
        TopicRulePayload: {
            Actions: [
                {
                    Sqs: {
                        QueueUrl: { Ref: "testiotsqsqueue630B4C1F" },
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiotsqsiotactionsrole93B1D327",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing messages from IoT devices or factory machines",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/topic/#'"
        }
    });
    // Uses the props to create the key
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
    template.hasResourceProperties("AWS::KMS::Alias", {
        AliasName: "alias/new-key-alias-from-props",
        TargetKeyId: {
            "Fn::GetAtt": [
                "testiotsqsEncryptionKey64EE64B1",
                "Arn",
            ]
        }
    });
});
// --------------------------------------------------------------
// Testing with passing a FIFO queue (not supported by IoT)
// --------------------------------------------------------------
test('Pattern deployment with passing a FIFO queue (not supported by IoT)', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const queue = new sqs.Queue(stack, 'existing-fifo-queue-obj', {
        queueName: 'existing-queue.fifo',
        fifo: true
    });
    const props = {
        existingQueueObj: queue,
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing messages from IoT devices or factory machines",
                sql: "SELECT * FROM 'test/topic/#'",
                actions: []
            }
        }
    };
    expect.assertions(1);
    try {
        new lib_1.IotToSqs(stack, 'test-iot-sqs', props);
    }
    catch (err) {
        expect(err.message).toBe('The IoT SQS action doesn\'t support Amazon SQS FIFO (First-In-First-Out) queues');
    }
});
//# sourceMappingURL=data:application/json;base64,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