"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProwlerAudit = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// import { Stack, Duration, RemovalPolicy, CustomResource } from 'aws-cdk-lib';
// eslint-disable-next-line no-duplicate-imports
// import { aws_iam as iam, aws_logs as logs, aws_s3 as s3, aws_codebuild as codebuild, aws_lambda as lambda, custom_resources as cr } from 'aws-cdk-lib';
// import { Construct } from 'constructs';
const codebuild = require("@aws-cdk/aws-codebuild");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const statement = require("cdk-iam-floyd");
/**
 * Creates a CodeBuild project to audit an AWS account with Prowler and stores the html report in a S3 bucket.
 *
 * This will run onece at the beginning and on a schedule afterwards. Partial contribution from https://github.com/stevecjones
 *
 * @stability stable
 */
class ProwlerAudit extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(parent, id, props) {
        var _b, _c, _d, _e;
        super(parent, id);
        // defaults
        this.serviceName = (props === null || props === void 0 ? void 0 : props.serviceName) ? props.serviceName : 'prowler';
        this.logsRetentionInDays = (props === null || props === void 0 ? void 0 : props.logsRetentionInDays) ? props.logsRetentionInDays : logs.RetentionDays.THREE_DAYS;
        this.enableScheduler = (props === null || props === void 0 ? void 0 : props.enableScheduler) ? props.enableScheduler : false;
        this.prowlerScheduler = (props === null || props === void 0 ? void 0 : props.prowlerScheduler) ? props.prowlerScheduler : 'cron(0 22 * * ? *)';
        this.prowlerOptions = (props === null || props === void 0 ? void 0 : props.prowlerOptions) ? props.prowlerOptions : '-M text,junit-xml,html,csv,json';
        this.prowlerVersion = (props === null || props === void 0 ? void 0 : props.prowlerVersion) ? props.prowlerVersion : '2.5.0';
        const reportBucket = (_b = props === null || props === void 0 ? void 0 : props.reportBucket) !== null && _b !== void 0 ? _b : new s3.Bucket(this, 'ReportBucket', {
            //bucketName: `${'123456'}-prowler-reports`,
            autoDeleteObjects: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        const reportGroup = new codebuild.ReportGroup(this, 'reportGroup', { /**reportGroupName: 'testReportGroup', */ removalPolicy: core_1.RemovalPolicy.DESTROY });
        reportGroup;
        const prowlerBuild = new codebuild.Project(this, 'prowlerBuild', {
            description: 'Run Prowler assessment',
            timeout: core_1.Duration.hours(5),
            environment: {
                environmentVariables: {
                    BUCKET_REPORT: { value: reportBucket.bucketName || '' },
                    BUCKET_PREFIX: { value: (_c = props === null || props === void 0 ? void 0 : props.reportBucketPrefix) !== null && _c !== void 0 ? _c : '' },
                    PROWLER_OPTIONS: { value: this.prowlerOptions || '' },
                },
                buildImage: codebuild.LinuxBuildImage.fromCodeBuildImageId('aws/codebuild/amazonlinux2-x86_64-standard:3.0'),
            },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        'runtime-versions': {
                            python: 3.8,
                        },
                        'commands': [
                            'echo "Installing Prowler and dependencies..."',
                            'pip3 install detect-secrets',
                            'yum -y install jq',
                            'curl "https://awscli.amazonaws.com/awscli-exe-linux-x86_64.zip" -o "awscliv2.zip"',
                            'unzip awscliv2.zip',
                            './aws/install',
                            `git clone -b ${this.prowlerVersion} https://github.com/toniblyx/prowler`,
                        ],
                    },
                    build: {
                        commands: [
                            `echo "Running Prowler as ./prowler ${this.prowlerOptions} && echo OK || echo FAILED"`,
                            'cd prowler',
                            `./prowler ${this.prowlerOptions} && echo OK || echo FAILED`,
                        ],
                    },
                    post_build: {
                        commands: [
                            'echo "Uploading reports to S3..." ',
                            'aws s3 cp --sse AES256 output/ s3://$BUCKET_REPORT/$BUCKET_PREFIX --recursive',
                            'echo "Done!"',
                        ],
                    },
                },
                reports: {
                    [reportGroup.reportGroupName]: {
                        'files': ['**/*'],
                        'base-directory': 'prowler/junit-reports',
                        'file-format': 'JunitXml',
                    },
                },
            }),
        });
        (_d = prowlerBuild.role) === null || _d === void 0 ? void 0 : _d.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('SecurityAudit'));
        (_e = prowlerBuild.role) === null || _e === void 0 ? void 0 : _e.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('job-function/ViewOnlyAccess'));
        // prowlerBuild.addToRolePolicy(new statement.Dax().allow().to());
        prowlerBuild.addToRolePolicy(new statement.Ds().allow().toListAuthorizedApplications());
        prowlerBuild.addToRolePolicy(new statement.Ec2().allow().toGetEbsEncryptionByDefault());
        prowlerBuild.addToRolePolicy(new statement.Ecr().allow().toDescribeImageScanFindings().toDescribeImages().toDescribeRegistry());
        prowlerBuild.addToRolePolicy(new statement.Tag().allow().toGetTagKeys());
        prowlerBuild.addToRolePolicy(new statement.Lambda().allow().toGetFunction());
        prowlerBuild.addToRolePolicy(new statement.Glue().allow().toSearchTables().toGetConnections());
        prowlerBuild.addToRolePolicy(new statement.Apigateway().allow().toGET());
        prowlerBuild.addToRolePolicy(new iam.PolicyStatement({ actions: ['support:Describe*'], resources: ['*'] }));
        reportBucket.grantPut(prowlerBuild);
        const myRole = new iam.Role(this, 'MyRole', { assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com') });
        const prowlerStartBuildLambda = new lambda.Function(this, 'Lambda', {
            runtime: lambda.Runtime.PYTHON_3_6,
            timeout: core_1.Duration.seconds(120),
            handler: 'index.lambda_handler',
            code: lambda.Code.fromInline(`import boto3
import cfnresponse
from botocore.exceptions import ClientError
def lambda_handler(event,context):
  props = event['ResourceProperties']
  codebuild_client = boto3.client('codebuild')
  if (event['RequestType'] == 'Create' or event['RequestType'] == 'Update'):
    try:
        response = codebuild_client.start_build(projectName=props['Build'])
        print(response)
        print("Respond: SUCCESS")
        cfnresponse.send(event, context, cfnresponse.SUCCESS, {})
    except Exception as ex:
        print(ex.response['Error']['Message'])
        cfnresponse.send(event, context, cfnresponse.FAILED, ex.response)
      `),
        });
        prowlerStartBuildLambda.addToRolePolicy(new statement.Codebuild().allow().toStartBuild()); // onResource project ...
        const myProvider = new cr.Provider(this, 'MyProvider', {
            onEventHandler: prowlerStartBuildLambda,
            logRetention: this.logsRetentionInDays,
            role: myRole,
        });
        new core_1.CustomResource(this, 'Resource1', {
            serviceToken: myProvider.serviceToken,
            properties: {
                Build: prowlerBuild.projectName,
                RERUN_PROWLER: Boolean(this.node.tryGetContext('reRunProwler')) ? Date.now().toString() : '',
            },
        });
        if (this.enableScheduler) {
            const prowlerSchedulerLambda = new lambda.Function(this, 'ScheduleLambda', {
                runtime: lambda.Runtime.PYTHON_3_6,
                timeout: core_1.Duration.seconds(120),
                handler: 'index.lambda_handler',
                environment: {
                    buildName: prowlerBuild.projectName,
                },
                code: lambda.Code.fromInline(`import boto3
        import os
        def lambda_handler(event,context):
          codebuild_client = boto3.client('codebuild')
          print("Running Prowler scheduled!: " + os.environ['buildName'])
          project_name = os.environ['buildName']
          response = codebuild_client.start_build(projectName=project_name)
          print(response)
          print("Respond: SUCCESS")
        `),
            });
            new events.Rule(this, 'Rule', {
                description: 'A schedule for the Lambda function that triggers Prowler in CodeBuild.',
                targets: [new targets.LambdaFunction(prowlerSchedulerLambda)],
                schedule: events.Schedule.expression(this.prowlerScheduler || ''),
            });
        }
    }
}
exports.ProwlerAudit = ProwlerAudit;
_a = JSII_RTTI_SYMBOL_1;
ProwlerAudit[_a] = { fqn: "cdk-prowler.ProwlerAudit", version: "1.118.1" };
//# sourceMappingURL=data:application/json;base64,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