import google.protobuf.wrappers_pb2 as wrappers

from cognite.seismic._api.api import API
from cognite.seismic.data_classes.trace_list import Trace2DList
from cognite.seismic.protos.query_service_messages_pb2 import GeometrySliceQueryRequest, LineSliceQueryRequest
from cognite.seismic.protos.types_pb2 import CRS, Geometry, LineRange, LineSelect, Wkt


class SliceAPI(API):
    def __init__(self, query, metadata):
        super().__init__(query=query, metadata=metadata)

    @staticmethod
    def _build_range(r_from=None, r_to=None):
        """
        Returns a LineRange based on to and from values
        """
        if r_from or r_to:
            f = wrappers.Int32Value(value=r_from)
            t = wrappers.Int32Value(value=r_to)
            return LineRange(from_line=f, to_line=t)
        else:
            return None

    def _get_slice(self, line_select, file_identifier, include_headers, ort_range):
        request = LineSliceQueryRequest(
            file=file_identifier, line=line_select, include_trace_header=include_headers, range=ort_range
        )
        return Trace2DList([i for i in self.query.GetSliceByLine(request, metadata=self.metadata)])

    def get_inline(self, inline, file_id=None, file_name=None, include_headers=False, from_line=None, to_line=None):
        """
        Gets the traces of an inline in a given file.
        :param inline: inline number (required)
        :param file_id: File can be specified either by name or id (id will be used first if both are provided)
        :param file_name: File can be specified either by name or id (id will be used first if both are provided)
        :param include_headers: Whether or not to include the trace headers in the response
        :param from_line: Include only crosslines equal or greater to this in the slice
        :param to_line: Include only crosslines equal or less than this in the slice
        :return:  Series of traces where each contains its inline, xline and the values.
        The line can be converted to a 2D numpy array with just the values calling .to_array() on it
        """
        return self._get_slice(
            line_select=LineSelect(iline=inline),
            file_identifier=self.identify(file_id, file_name),
            include_headers=include_headers,
            ort_range=self._build_range(from_line, to_line),
        )

    def get_crossline(
        self, crossline, file_id=None, file_name=None, include_headers=False, from_line=None, to_line=None
    ):
        """
        Gets the traces of a crossline in a given file.
        :param crossline: crossline number (required)
        :param file_id: File can be specified either by name or id (id will be used first if both are provided)
        :param file_name: File can be specified either by name or id (id will be used first if both are provided)
        :param include_headers: Whether or not to include the trace headers in the response
        :param from_line: Include only inlines equal or greater to this in the slice
        :param to_line: Include only inlines equal or less than this in the slice
        :return:  Series of traces where each contains its inline, xline and the values.
        The line can be converted to a 2D numpy array with just the values calling .to_array() on it
        """
        return self._get_slice(
            line_select=LineSelect(xline=crossline),
            file_identifier=self.identify(file_id, file_name),
            include_headers=include_headers,
            ort_range=self._build_range(from_line, to_line),
        )

    def get_arbitrary_line(
        self, file_id=None, file_name=None, x0=None, y0=None, x1=None, y1=None, crs=None, interpolation_method=1
    ):
        """
        Returns a series of traces on an arbitrary line starting from (x0, yo) and ending in (x1, y1)
        Currently, all the traces are snapped to the grid inlines or crosslines
        :param file_id: File can be specified either by name or id (id will be used first if both are provided)
        :param file_name: File can be specified either by name or id (id will be used first if both are provided)
        :param crs: Specify the CRS in which the coordinates x0, y0, x1 and y1 are given (Ex.: "EPSG:23031")
        :param crs: Specify the interpolation_method. Currently possible are 0 for snapping the traces to the nearest
                    grid points (inlines or crosslines) or 1 for weighing the nearest traces on each side of the line
        :return: Series of traces composing the line
        The line can be converted to a 2D numpy array with just the values calling .to_array() on it
        """
        if not x0 or not y0 or not x1 or not y1 or not crs:
            raise Exception("x0, y0, x1, y1 and crs must be specified")
        p0 = str(x0) + " " + str(y0)
        p1 = str(x1) + " " + str(y1)
        linestring = "LINESTRING(" + p0 + ", " + p1 + ")"
        line = Geometry(crs=CRS(crs=crs), wkt=Wkt(geometry=linestring))
        file = self.identify(file_id, file_name)
        request = GeometrySliceQueryRequest(file=file, arbitrary_line=line, interpolation_method=interpolation_method)
        return Trace2DList([i for i in self.query.GetSliceByGeometry(request, metadata=self.metadata)])
