"""Mixed meshes.

There exists a preliminary support for mixed meshes, i.e. meshes combining
multiple element types such as triangles and quadrilaterals.  This is mostly
because the implementation of :func:`~skfem.assembly.asm` supports lists of
:class:`~skfem.assembly.basis.AbstractBasis` objects and the resulting matrices
of the multiple assembly operations is combined in the end.

The working principle is that a list of meshes is created by the user, one for
each element type.  Then the finite element basis is created separately for
each mesh.  Finally, the list of basis objects is passed on to
:func:`~skfem.assembly.asm`.

This example demonstrates how a mesh generated by Gmsh, consisting of triangles
and quadrilaterals, can be used to solve the Poisson problem with unit load.
Currently the user must verify that the finite element bases are compatible
with one another.  Moreover, the mixed meshes work properly only with finite
elements that have nodal degrees-of-freedom.  Support for higher-order elements
on mixed meshes is work-in-progress.

"""
from skfem import *
from skfem.models import laplace, unit_load

fname = 'docs/examples/meshes/mixedtriquad.msh'
out = ['cell_sets_dict']  # read boundary nodes from meshio
m = [
    Mesh.load(fname, force_meshio_type='triangle'),
    Mesh.load(fname, force_meshio_type='quad', out=out),
]
e = [ElementTriP1(), ElementQuad1()]
basis = list(map(Basis, m, e))

A = asm(laplace, basis)
f = asm(unit_load, basis)

y = solve(*condense(A, f, D=out[0]['boundary']['line']))

if __name__ == '__main__':
    from os.path import splitext
    from sys import argv
    from skfem.visuals.matplotlib import plot, draw, savefig
    ax = plot(basis[0], y, Nrefs=4)
    draw(basis[0], ax=ax)
    plot(basis[1], y, ax=ax, Nrefs=4)
    draw(basis[1], ax=ax)
    savefig(splitext(argv[0])[0] + '_solution.png')
