"""
Django settings for project project.

Generated by 'django-admin startproject' using Django 3.2.

For more information on this file, see
https://docs.djangoproject.com/en/3.2/topics/settings/

For the full list of settings and their values, see
https://docs.djangoproject.com/en/3.2/ref/settings/
"""

import logging
import os

import environ
import sentry_sdk
from sentry_sdk.integrations.celery import CeleryIntegration
from sentry_sdk.integrations.django import DjangoIntegration
from sentry_sdk.integrations.logging import LoggingIntegration

# Build paths inside the project like this: BASE_DIR / 'subdir'.
BASE_DIR = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
ROOT_DIR = environ.Path(BASE_DIR) - 1

env = environ.Env()
env.read_env(ROOT_DIR('.env.local'))

# Quick-start development settings - unsuitable for production
# See https://docs.djangoproject.com/en/3.2/howto/deployment/checklist/

# SECURITY WARNING: keep the secret key used in production secret!
SECRET_KEY = env("SECRET_KEY")

# PharmZakaz endpoint test
PHARMZAKAZENDPOINT = "https://demo.pharm-zakaz.ru/api/distributor/v1/"

# PharmZakaz endpoint prod
# PHARMZAKAZENDPOINT=https://pharm-zakaz.ru/distributor-api/v0/

# PharmZakaz token test
PHARMZAKAZTOKEN = env('PHARMZAKAZTOKEN')


# SECURITY WARNING: don't run with debug turned on in production!
DEBUG = env.bool("DEBUG", False)

ENV_TYPE = env('ENV_TYPE', default='stage')

DATA_BASE_ABBR = "PHZ"

ALLOWED_HOSTS = env.list("ALLOWED_HOSTS", default=["*"])


# Celery settings
CELERY_BROKER_URL = env("CELERY_BROKER_URL")
CELERY_ACCEPT_CONTENT = ["json"]
CELERY_TASK_SERIALIZER = "json"
CELERY_RESULT_BACKEND = env("CELERY_RESULT_BACKEND")
CELERY_RESULT_SERIALIZER = "json"

# Application definition

INSTALLED_APPS = [
    'django_celery_beat',
    'django.contrib.auth',
    'django.contrib.contenttypes',
    'django.contrib.sessions',
    'django.contrib.messages',
    'django.contrib.staticfiles',
    'django.contrib.admin',

    'rest_framework',
    'rest_framework.authtoken',

    'drf_yasg',

    'django_admin_shell',
    'django_python3_ldap',

    'rate_limit',

    'users',
    'core',
    'api',
]


# LDAP AUTHENTICATION

AUTHENTICATION_BACKENDS = [
    'core.auth.backends.ModelBackend',
    'django_python3_ldap.auth.LDAPBackend',
]

LDAP_AUTH_URL = env("LDAP_AUTH_URL")

LDAP_AUTH_USE_TLS = False

LDAP_AUTH_SEARCH_BASE = env("LDAP_AUTH_SEARCH_BASE")

LDAP_AUTH_OBJECT_CLASS = "user"

LDAP_AUTH_USER_FIELDS = {
    'username': 'sAMAccountName',
    'first_name': 'givenName',
    'last_name': 'sn',
    'email': 'mail',
    "usercoid": "extensionAttribute10"
}

LDAP_AUTH_USER_LOOKUP_FIELDS = ("username",)

LDAP_AUTH_CLEAN_USER_DATA = "django_python3_ldap.utils.clean_user_data"

LDAP_AUTH_SYNC_USER_RELATIONS = "django_python3_ldap.utils.sync_user_relations"

LDAP_AUTH_FORMAT_SEARCH_FILTERS = "django_python3_ldap.utils.format_search_filters"

LDAP_AUTH_FORMAT_USERNAME = "django_python3_ldap.utils.format_username_active_directory_principal"

LDAP_AUTH_ACTIVE_DIRECTORY_DOMAIN = env("LDAP_AUTH_ACTIVE_DIRECTORY_DOMAIN")

LDAP_AUTH_CONNECTION_USERNAME = env("LDAP_USERNAME")
LDAP_AUTH_CONNECTION_PASSWORD = env("LDAP_PASSWORD")

LDAP_AUTH_CONNECT_TIMEOUT = None
LDAP_AUTH_RECEIVE_TIMEOUT = None

# LDAP AUTHENTICATION

MIDDLEWARE = [
    'django.middleware.security.SecurityMiddleware',
    'django.contrib.sessions.middleware.SessionMiddleware',
    'django.middleware.common.CommonMiddleware',
    'django.middleware.csrf.CsrfViewMiddleware',
    'django.contrib.auth.middleware.AuthenticationMiddleware',
    'django.contrib.messages.middleware.MessageMiddleware',
    'django.middleware.clickjacking.XFrameOptionsMiddleware',
]

ROOT_URLCONF = 'project.urls'

TEMPLATES = [
    {
        'BACKEND': 'django.template.backends.django.DjangoTemplates',
        'DIRS': [],
        'APP_DIRS': True,
        'OPTIONS': {
            'context_processors': [
                'django.template.context_processors.debug',
                'django.template.context_processors.request',
                'django.contrib.auth.context_processors.auth',
                'django.contrib.messages.context_processors.messages',
            ],
        },
    },
]

WSGI_APPLICATION = 'project.wsgi.application'


# Database
# https://docs.djangoproject.com/en/3.2/ref/settings/#databases

DATABASES = {"default": env.db()}


# Password validation
# https://docs.djangoproject.com/en/3.2/ref/settings/#auth-password-validators

AUTH_PASSWORD_VALIDATORS = [
    {
        'NAME': 'django.contrib.auth.password_validation.UserAttributeSimilarityValidator',
    },
    {
        'NAME': 'django.contrib.auth.password_validation.MinimumLengthValidator',
    },
    {
        'NAME': 'django.contrib.auth.password_validation.CommonPasswordValidator',
    },
    {
        'NAME': 'django.contrib.auth.password_validation.NumericPasswordValidator',
    },
]


# Internationalization
# https://docs.djangoproject.com/en/3.2/topics/i18n/

LANGUAGE_CODE = 'ru-RU'

TIME_ZONE = 'Europe/Moscow'

USE_I18N = True

USE_L10N = True

USE_TZ = True


# Static files (CSS, JavaScript, Images)
# https://docs.djangoproject.com/en/3.2/howto/static-files/

STATIC_URL = '/static/'
STATIC_ROOT = os.path.join(BASE_DIR, '../static')

# Default primary key field type
# https://docs.djangoproject.com/en/3.2/ref/settings/#default-auto-field

DEFAULT_AUTO_FIELD = 'django.db.models.BigAutoField'

AUTH_USER_MODEL = 'users.AppUser'

API_RAISE_EXCEPTIONS = env.bool("API_RAISE_EXCEPTIONS", True)

CACHES = {
    "default": env.cache("CACHE_URL"),
}


RATELIMITER_REDIS_URL = env("RATELIMITER_REDIS_URL")

TARGET_TOPIC_TO_SEND = "ExchangeMSK_BRN7_test"

LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'formatters': {
        'verbose': {
            'format': "[%(asctime)s] %(levelname)s %(message)s",
        },
    },
    'handlers': {
        'console': {
            'class': 'logging.StreamHandler',
            'formatter': 'verbose',
        },
    },
    'root': {
        'handlers': ['console'],
        'level': env('CONSOLE_LOGGING_LEVEL', default='WARNING'),
    },
    'loggers': {
        'elasticapm': {
            'handlers': ['console'],
            'level': 'CRITICAL',
        },
        'core.api.client': {
            'handlers': ['console'],
            'level': 'INFO',
            'propagate': False,
        },
        "django_python3_ldap": {
            "handlers": ["console"],
            "level": "INFO",
        },
    }
}


SENTRY_DSN = env('SENTRY_DSN', default=None)
if SENTRY_DSN:
    versions = environ.Env()
    versions.read_env(os.path.join(BASE_DIR, '.versions'))
    commit_sha = versions('COMMIT_SHA', default='unknown_sha')
    commit_branch = versions('COMMIT_BRANCH', default='unknown_branch')

    sentry_logging = LoggingIntegration(
        level=logging.INFO,        # Capture info and above as breadcrumbs
        event_level=env('SENTRY_EVENT_LEVEL', default='ERROR'),  # Send errors as events
    )

    sentry_sdk.init(
        dsn=SENTRY_DSN,
        integrations=[sentry_logging, DjangoIntegration(), CeleryIntegration()],
        environment=ENV_TYPE,
        # Таймаут очистки очереди.
        shutdown_timeout=1,
        release=commit_sha
    )
    sentry_sdk.set_tag('commit_branch', commit_branch)


# APM settings
ELASTIC_APM_SERVER = env('ELASTIC_APM_SERVER', default=None)
ELASTIC_APM_SERVICE_NAME = env('ELASTIC_APM_SERVICE_NAME', default='pharm-zakaz')
ELASTIC_APM_SERVICE_VERSION = env('ELASTIC_APM_SERVICE_VERSION', default=None)
ELASTIC_APM_DISABLE_METRICS = env('ELASTIC_APM_DISABLE_METRICS', default=None)
ELASTIC_APM_INSTRUMENTATION_EXCLUDE = env.list('ELASTIC_APM_INSTRUMENTATION_EXCLUDE', default=None)
if ELASTIC_APM_SERVER:
    INSTALLED_APPS += ['elasticapm.contrib.django']
    ELASTIC_APM = {
        'SERVICE_NAME': ELASTIC_APM_SERVICE_NAME,
        'SERVICE_VERSION': ELASTIC_APM_SERVICE_VERSION,
        'ENVIRONMENT': ENV_TYPE,
        'SERVER_URL': ELASTIC_APM_SERVER,
        'CAPTURE_BODY': 'all',
        'DUPLICATE_HEADERS_TO_LABELS': True,
        'TRANSACTIONS_IGNORE_PATTERNS': ['^OPTIONS', 'django.contrib.admin'],
        'PROCESSORS': (
            'elasticapm.processors.sanitize_stacktrace_locals',
            'elasticapm.processors.sanitize_http_request_cookies',
            'elasticapm.processors.sanitize_http_wsgi_env',
        ),
        'DISABLE_METRICS': ELASTIC_APM_DISABLE_METRICS,
        'DEBUG': True,
    }

REST_FRAMEWORK = {
    'DEFAULT_AUTHENTICATION_CLASSES': ['rest_framework.authentication.TokenAuthentication', ],
    'DEFAULT_PERMISSION_CLASSES': ['rest_framework.permissions.IsAuthenticated', ],
    'DEFAULT_PAGINATION_CLASS': 'rest_framework.pagination.PageNumberPagination',
    'PAGE_SIZE': 99
}

KAFKA_SETTINGS = {  # TODO возможно вынести в админку стоит
    'BOOTSTRAP_SERVERS': ['broker.puls.local', ],
    'AUTO_COMMIT': False,
    'SLEEP_SECONDS': 5,
}
