"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bucket = void 0;
const s3_bucket_1 = require("@cdktf/provider-aws/lib/s3-bucket");
const s3_bucket_policy_1 = require("@cdktf/provider-aws/lib/s3-bucket-policy");
const s3_bucket_public_access_block_1 = require("@cdktf/provider-aws/lib/s3-bucket-public-access-block");
const s3_bucket_server_side_encryption_configuration_1 = require("@cdktf/provider-aws/lib/s3-bucket-server-side-encryption-configuration");
const s3_bucket_versioning_1 = require("@cdktf/provider-aws/lib/s3-bucket-versioning");
const constructs_1 = require("constructs");
/*----------------------------------------------------------------------------------
                                  Construct
  ----------------------------------------------------------------------------------*/
class Bucket extends constructs_1.Construct {
    constructor(scope, id, config) {
        super(scope, id);
        this.bucket = new s3_bucket_1.S3Bucket(this, 'resource', {
            bucketPrefix: `secure-bucket-${config.name}`,
        });
        this.id = this.bucket.bucket;
        if (config.sseEnabled ?? false) {
            this.enableSSE();
        }
        if (config.blockAllPublicAccess ?? true) {
            this.blockAllPublicAccess();
        }
        if (config.versioningEnabled ?? false) {
            this.enableVersioning();
        }
    }
    attachBucketPolicy(policy) {
        new s3_bucket_policy_1.S3BucketPolicy(this, 'policy', {
            bucket: this.bucket.id,
            policy: policy.json,
        });
    }
    enableSSE(kmsKeyId) {
        new s3_bucket_server_side_encryption_configuration_1.S3BucketServerSideEncryptionConfigurationA(this, 'sse-config', {
            bucket: this.bucket.id,
            rule: [
                {
                    bucketKeyEnabled: true,
                    applyServerSideEncryptionByDefault: {
                        kmsMasterKeyId: kmsKeyId,
                        sseAlgorithm: 'aws:kms',
                    },
                },
            ],
        });
    }
    blockAllPublicAccess() {
        new s3_bucket_public_access_block_1.S3BucketPublicAccessBlock(this, 'public-access-block', {
            bucket: this.bucket.id,
            blockPublicAcls: true,
            blockPublicPolicy: true,
            ignorePublicAcls: true,
            restrictPublicBuckets: true,
        });
    }
    enableVersioning() {
        new s3_bucket_versioning_1.S3BucketVersioningA(this, 'versioning', {
            bucket: this.bucket.id,
            versioningConfiguration: {
                status: 'Enabled',
            },
        });
    }
}
exports.Bucket = Bucket;
//# sourceMappingURL=data:application/json;base64,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