import * as codebuild from 'aws-cdk-lib/aws-codebuild';
import { Asset } from 'aws-cdk-lib/aws-s3-assets';
import { IResource, Resource } from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { IApp } from './app';
import { BasicAuth } from './basic-auth';
/**
 * A branch
 */
export interface IBranch extends IResource {
    /**
     * The name of the branch
     *
     * @attribute
     */
    readonly branchName: string;
}
/**
 * Options to add a branch to an application
 */
export interface BranchOptions {
    /**
     * The Basic Auth configuration. Use this to set password protection for
     * the branch
     *
     * @default - no password protection
     */
    readonly basicAuth?: BasicAuth;
    /**
     * The name of the branch
     *
     * @default - the construct's id
     */
    readonly branchName?: string;
    /**
     * BuildSpec for the branch
     *
     * @see https://docs.aws.amazon.com/amplify/latest/userguide/build-settings.html
     *
     * @default - no build spec
     */
    readonly buildSpec?: codebuild.BuildSpec;
    /**
     * A description for the branch
     *
     * @default - no description
     */
    readonly description?: string;
    /**
     * Whether to enable auto building for the branch
     *
     * @default true
     */
    readonly autoBuild?: boolean;
    /**
     * Whether to enable pull request preview for the branch.
     *
     * @default true
     */
    readonly pullRequestPreview?: boolean;
    /**
     * Environment variables for the branch.
     *
     * All environment variables that you add are encrypted to prevent rogue
     * access so you can use them to store secret information.
     *
     * @default - application environment variables
     */
    readonly environmentVariables?: {
        [name: string]: string;
    };
    /**
     * The dedicated backend environment for the pull request previews
     *
     * @default - automatically provision a temporary backend
     */
    readonly pullRequestEnvironmentName?: string;
    /**
     * Stage for the branch
     *
     * @default - no stage
     */
    readonly stage?: string;
    /**
     * Asset for deployment.
     *
     * The Amplify app must not have a sourceCodeProvider configured as this resource uses Amplify's
     * startDeployment API to initiate and deploy a S3 asset onto the App.
     *
     * @default - no asset
     */
    readonly asset?: Asset;
    /**
     * Enables performance mode for the branch.
     *
     * Performance mode optimizes for faster hosting performance by keeping content cached at the edge
     * for a longer interval. When performance mode is enabled, hosting configuration or code changes
     * can take up to 10 minutes to roll out.
     *
     * @default false
     */
    readonly performanceMode?: boolean;
}
/**
 * Properties for a Branch
 */
export interface BranchProps extends BranchOptions {
    /**
     * The application within which the branch must be created
     */
    readonly app: IApp;
}
/**
 * An Amplify Console branch
 */
export declare class Branch extends Resource implements IBranch {
    /**
     * Import an existing branch
     */
    static fromBranchName(scope: Construct, id: string, branchName: string): IBranch;
    /**
     * The ARN of the branch
     *
     * @attribute
     */
    readonly arn: string;
    readonly branchName: string;
    private readonly environmentVariables;
    constructor(scope: Construct, id: string, props: BranchProps);
    /**
     * Adds an environment variable to this branch.
     *
     * All environment variables that you add are encrypted to prevent rogue
     * access so you can use them to store secret information.
     */
    addEnvironment(name: string, value: string): this;
}
