"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const scalable_target_1 = require("./scalable-target");
/**
 * Represent an attribute for which autoscaling can be configured
 *
 * This class is basically a light wrapper around ScalableTarget, but with
 * all methods protected instead of public so they can be selectively
 * exposed and/or more specific versions of them can be exposed by derived
 * classes for individual services support autoscaling.
 *
 * Typical use cases:
 *
 * - Hide away the PredefinedMetric enum for target tracking policies.
 * - Don't expose all scaling methods (for example Dynamo tables don't support
 *   Step Scaling, so the Dynamo subclass won't expose this method).
 */
class BaseScalableAttribute extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.target = new scalable_target_1.ScalableTarget(this, 'Target', {
            serviceNamespace: this.props.serviceNamespace,
            scalableDimension: this.props.dimension,
            resourceId: this.props.resourceId,
            role: this.props.role,
            minCapacity: props.minCapacity !== undefined ? props.minCapacity : 1,
            maxCapacity: props.maxCapacity
        });
    }
    /**
     * Scale out or in based on time
     */
    doScaleOnSchedule(id, props) {
        this.target.scaleOnSchedule(id, props);
    }
    /**
     * Scale out or in based on a metric value
     */
    doScaleOnMetric(id, props) {
        this.target.scaleOnMetric(id, props);
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    doScaleToTrackMetric(id, props) {
        this.target.scaleToTrackMetric(id, props);
    }
}
exports.BaseScalableAttribute = BaseScalableAttribute;
//# sourceMappingURL=data:application/json;base64,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