"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
/**
 * Define a scalable target
 */
class ScalableTarget extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.actions = new Array();
        core_1.withResolved(props.maxCapacity, max => {
            if (max < 0) {
                throw new RangeError(`maxCapacity cannot be negative, got: ${props.maxCapacity}`);
            }
        });
        core_1.withResolved(props.minCapacity, min => {
            if (min < 0) {
                throw new RangeError(`minCapacity cannot be negative, got: ${props.minCapacity}`);
            }
        });
        core_1.withResolved(props.minCapacity, props.maxCapacity, (min, max) => {
            if (max < min) {
                throw new RangeError(`minCapacity (${props.minCapacity}) should be lower than maxCapacity (${props.maxCapacity})`);
            }
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('application-autoscaling.amazonaws.com')
        });
        const resource = new applicationautoscaling_generated_1.CfnScalableTarget(this, 'Resource', {
            maxCapacity: props.maxCapacity,
            minCapacity: props.minCapacity,
            resourceId: props.resourceId,
            roleArn: this.role.roleArn,
            scalableDimension: props.scalableDimension,
            scheduledActions: core_1.Lazy.anyValue({ produce: () => this.actions }, { omitEmptyArray: true }),
            serviceNamespace: props.serviceNamespace
        });
        this.scalableTargetId = resource.ref;
    }
    static fromScalableTargetId(scope, id, scalableTargetId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.scalableTargetId = scalableTargetId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a policy statement to the role's policy
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, action) {
        if (action.minCapacity === undefined && action.maxCapacity === undefined) {
            throw new Error(`You must supply at least one of minCapacity or maxCapacity, got ${JSON.stringify(action)}`);
        }
        this.actions.push({
            scheduledActionName: id,
            schedule: action.schedule.expressionString,
            startTime: action.startTime,
            endTime: action.endTime,
            scalableTargetAction: {
                maxCapacity: action.maxCapacity,
                minCapacity: action.minCapacity
            },
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, Object.assign(Object.assign({}, props), { scalingTarget: this }));
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, id, Object.assign(Object.assign({}, props), { scalingTarget: this }));
    }
}
exports.ScalableTarget = ScalableTarget;
/**
 * The service that supports Application AutoScaling
 */
var ServiceNamespace;
(function (ServiceNamespace) {
    /**
     * Elastic Container Service
     */
    ServiceNamespace["ECS"] = "ecs";
    /**
     * Elastic Map Reduce
     */
    ServiceNamespace["ELASTIC_MAP_REDUCE"] = "elasticmapreduce";
    /**
     * Elastic Compute Cloud
     */
    ServiceNamespace["EC2"] = "ec2";
    /**
     * App Stream
     */
    ServiceNamespace["APPSTREAM"] = "appstream";
    /**
     * Dynamo DB
     */
    ServiceNamespace["DYNAMODB"] = "dynamodb";
    /**
     * Relational Database Service
     */
    ServiceNamespace["RDS"] = "rds";
    /**
     * SageMaker
     */
    ServiceNamespace["SAGEMAKER"] = "sagemaker";
    /**
     * Custom Resource
     */
    ServiceNamespace["CUSTOM_RESOURCE"] = "custom-resource";
})(ServiceNamespace = exports.ServiceNamespace || (exports.ServiceNamespace = {}));
//# sourceMappingURL=data:application/json;base64,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