# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['mugmoment']

package_data = \
{'': ['*']}

install_requires = \
['httpx>=0.17.1,<0.18.0']

setup_kwargs = {
    'name': 'mugmoment',
    'version': '0.2.0',
    'description': 'Library to download chat logs from twitch VODs',
    'long_description': '# mugmoment\n\nmugmoment is a python3 library to download chat logs from twitch VODs.\n\nBy default, it exports in the default json format of twitch (referred to in this project as "raw"), but it can also export the chat to various other formats, such as a simpler json, txt and html.\n\n## Installation\n\n### PyPI\n\nThis package is available on PyPI as `mugmoment`.\n\n### Poetry\n\nThis project uses [poetry](https://python-poetry.org/), and the recommended way of installing it is running `poetry install` on the root of this repository, which will install it in a venv.\n\n## Usage\n\n### Get raw chat logs\n\n`mugmoment.fetch_raw(vod_id)` can be used to fetch raw VOD chat logs in the same format as twitch.\n\n```python\nimport mugmoment\n\nvod_id = 983321676\nchat_log = mugmoment.fetch_raw(vod_id)\nprint(chat_log[0])\n# {\'_id\': \'36007596-db82-4f7e-8d24-f19fc75dd5c9\', \'created_at\': \'2021-04-11T03:16:42.603Z\', \'updated_at\': \'2021-04-11T03:16:42.603Z\', \'channel_id\': \'28219022\', \'content_type\': \'video\', \'content_id\': \'983321676\', \'content_offset_seconds\': 7.9030000000000005, \'commenter\': {\'display_name\': \'Encode_NA\', \'_id\': \'156774711\', \'name\': \'encode_na\', \'type\': \'user\', \'bio\': "I don\'t know much about them, but I\'m sure twitch.tv is great.", \'created_at\': \'2017-05-13T19:10:06.262184Z\', \'updated_at\': \'2021-03-26T11:14:49.88742Z\', \'logo\': \'https://static-cdn.jtvnw.net/user-default-pictures-uv/294c98b5-e34d-42cd-a8f0-140b72fba9b0-profile_image-300x300.png\'}, \'source\': \'chat\', \'state\': \'published\', \'message\': {\'body\': \':( what a way to go\', \'emoticons\': [{\'_id\': \'555555558\', \'begin\': 0, \'end\': 1}], \'fragments\': [{\'text\': \':(\', \'emoticon\': {\'emoticon_id\': \'555555558\', \'emoticon_set_id\': \'\'}}, {\'text\': \' what a way to go\'}], \'is_action\': False, \'user_color\': \'#5F9EA0\', \'user_notice_params\': {}}}\n```\n\n### Get chat logs in other formats\n\n#### Simpler dictionary\n\n`mugmoment.fetch_simple(vod_id)` can be used to fetch VOD chat logs and convert them to a simpler format.\n\n```python\nimport mugmoment\n\nvod_id = 983321676\nchat_log = mugmoment.fetch_simple(vod_id)\nprint(chat_log[0])\n# {\'author\': \'Encode_NA\', \'body\': \':( what a way to go\', \'offset\': 7.9030000000000005, \'fancy_offset\': \'00:07\'}\n```\n\n**Note:** `fancy_offset` has the format hh:mm:ss. If the vod is shorter than an hour, then it\'ll be displayed in mm:ss format instead.\n\n#### txt\n\n`mugmoment.fetch_txt(vod_id)` can be used to fetch VOD chat logs and convert them to a txt format.\n\n```python\nimport mugmoment\n\nvod_id = 983321676\nchat_log = mugmoment.fetch_txt(vod_id)\nprint(chat_log)\n# 00:07 <Encode_NA> :( what a way to go\\r\\n[...]\n```\n\n**Note:** Offset on each line has the format hh:mm:ss. If the vod is shorter than an hour, then it\'ll be displayed in mm:ss format instead.\n\n#### html\n\n`mugmoment.fetch_html(vod_id)` can be used to fetch VOD chat logs and convert them to an html format. This is a WIP feature and may be developed further in the future.\n\n```python\nimport mugmoment\n\nvod_id = 983321676\nchat_log = mugmoment.fetch_html(vod_id)\nprint(chat_log)\n# <span class="timeoffset">00:07</span> <span class="author" style="color: #5F9EA0">Encode_NA</span>: <span class="message"><img alt=":(" class="emote emote-555555558" src="https://static-cdn.jtvnw.net/emoticons/v2/555555558/default/dark/1.0" srcset="https://static-cdn.jtvnw.net/emoticons/v2/555555558/default/dark/1.0 1x,https://static-cdn.jtvnw.net/emoticons/v2/555555558/default/dark/2.0 2x,https://static-cdn.jtvnw.net/emoticons/v2/555555558/default/dark/3.0 4x"> what a way to go</span><br>\\r\\n[...]\n```\n\nYou may also set the following arguments to render further parts of the chat:\n\n- `render_badges`: If set to true, badges will be rendered before the usernames. This involves 2 API calls to twitch.\n- `render_ext_emotes`: If set to true, third party emotes (BetterTTV, FrankerFaceZ) will be rendered. This involves 1 API call to twitch and 4 API calls to BetterTTV.\n\n**Note:** `timeoffset` has the format hh:mm:ss. If the vod is shorter than an hour, then it\'ll be displayed in mm:ss format instead.\n\n### Convert raw chat logs to other formats\n\nThe following converter functions are available:\n\n- `mugmoment.converters.ttv_raw_to_simple_format(raw_log)`\n- `mugmoment.converters.ttv_raw_to_txt(raw_log)`\n- `mugmoment.converters.ttv_raw_to_html(raw_log)`\n\nThese functions output same data as calling `mugmoment.fetch_formatname`.\n\n**Example:**\n\n```python\nimport mugmoment\n\nvod_id = 983321676\nraw_log = mugmoment.fetch_raw(vod_id)\nchat_log = mugmoment.converters.ttv_raw_to_simple_format(raw_log)\nprint(chat_log[0])\n# {\'author\': \'Encode_NA\', \'body\': \':( what a way to go\', \'offset\': 7.9030000000000005, \'fancy_offset\': \'00:07\'}\n```\n\n### Get badges\n\n`mugmoment.fetch_badges_by_vod_id(vod_id)` and `mugmoment.fetch_badges_by_channel_login(channel_login)` can be used to fetch raw badge list in the same format as twitch.\n\n```python\nimport mugmoment\n\nvod_id = 983321676\nbadges = mugmoment.fetch_badges_by_vod_id(vod_id)\nprint(badges[0])\n# {\'id\': \'YW1iYXNzYWRvcjsxOw==\', \'setID\': \'ambassador\', \'version\': \'1\', \'title\': \'Twitch Ambassador\', \'image1x\': \'https://static-cdn.jtvnw.net/badges/v1/2cbc339f-34f4-488a-ae51-efdf74f4e323/1\', \'image2x\': \'https://static-cdn.jtvnw.net/badges/v1/2cbc339f-34f4-488a-ae51-efdf74f4e323/2\', \'image4x\': \'https://static-cdn.jtvnw.net/badges/v1/2cbc339f-34f4-488a-ae51-efdf74f4e323/3\', \'clickAction\': \'VISIT_URL\', \'clickURL\': \'https://www.twitch.tv/team/ambassadors\', \'__typename\': \'Badge\'}\n\nchannel_login = "vargskelethor"\nbadges = mugmoment.fetch_badges_by_channel_login(channel_login)\nprint(badges[0])\n# {\'id\': \'YW1iYXNzYWRvcjsxOw==\', \'setID\': \'ambassador\', \'version\': \'1\', \'title\': \'Twitch Ambassador\', \'image1x\': \'https://static-cdn.jtvnw.net/badges/v1/2cbc339f-34f4-488a-ae51-efdf74f4e323/1\', \'image2x\': \'https://static-cdn.jtvnw.net/badges/v1/2cbc339f-34f4-488a-ae51-efdf74f4e323/2\', \'image4x\': \'https://static-cdn.jtvnw.net/badges/v1/2cbc339f-34f4-488a-ae51-efdf74f4e323/3\', \'clickAction\': \'VISIT_URL\', \'clickURL\': \'https://www.twitch.tv/team/ambassadors\', \'__typename\': \'Badge\'}\n```\n\n### Get third party (BetterTTV and FrankerFaceZ) emotes\n\n`mugmoment.fetch_third_party_emotes_by_channel_id(channel_id)` and `mugmoment.fetch_third_party_emotes_by_vod_id(vod_id)` can be used to fetch a dictionary of all BetterTTV and FrankerFaceZ emotes that can be used in this channel (this includes shared emotes and globals).\n\n```python\nimport mugmoment\n\nchannel_id = 28219022\nemotes = mugmoment.fetch_third_party_emotes_by_channel_id(channel_id)\nprint(emotes["vargD"])\n# {\'id\': \'6078c1f239b5010444d0008f\', \'alt\': \'BTTV channel emote\', \'src\': \'https://cdn.betterttv.net/emote/6078c1f239b5010444d0008f/1x\', \'srcset\': \'https://cdn.betterttv.net/emote/6078c1f239b5010444d0008f/1x 1x,https://cdn.betterttv.net/emote/6078c1f239b5010444d0008f/2x 2x,https://cdn.betterttv.net/emote/6078c1f239b5010444d0008f/3x 4x\', \'global\': False, \'source\': \'bttv\'}\n\nvod_id = 983321676\nemotes = mugmoment.fetch_third_party_emotes_by_vod_id(vod_id)\nprint(emotes["ALERTA"])\n# {\'id\': \'604447c0306b602acc59852a\', \'alt\': \'BTTV channel emote from Rakyz\', \'src\': \'https://cdn.betterttv.net/emote/604447c0306b602acc59852a/1x\', \'srcset\': \'https://cdn.betterttv.net/emote/604447c0306b602acc59852a/1x 1x,https://cdn.betterttv.net/emote/604447c0306b602acc59852a/2x 2x,https://cdn.betterttv.net/emote/604447c0306b602acc59852a/3x 4x\', \'global\': False, \'source\': \'bttv\'}\n```\n\n## Shoutouts\n\n- [RechatTool](https://github.com/jdpurcell/RechatTool)\'s codebase helped with figuring out how to do pagination in the API.\n- [BetterTTV](https://github.com/night/betterttv)\'s API is used heavily for emotes, and emote matching is based on that codebase.\n',
    'author': 'Ave',
    'author_email': 'ave@ave.zone',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://gitlab.com/a/mugmoment',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
