# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['shazamio']

package_data = \
{'': ['*']}

install_requires = \
['aiofiles>=0.8.0,<0.9.0',
 'aiohttp>=3.8.1,<4.0.0',
 'anyio>=3.6.1,<4.0.0',
 'dataclass-factory==2.16',
 'numpy>=1.22.1,<2.0.0',
 'pydub>=0.25.1,<0.26.0',
 'pytest-asyncio>=0.19.0,<0.20.0',
 'pytest>=7.1.2,<8.0.0']

setup_kwargs = {
    'name': 'shazamio',
    'version': '0.3.0.0',
    'description': 'Is a asynchronous framework from reverse engineered Shazam API written in Python 3.8+ with asyncio and aiohttp.',
    'long_description': '<p align="center">\n<img src="https://img.shields.io/lgtm/grade/python/github/dotX12/waio" alt="https://img.shields.io/lgtm/grade/python/github/dotX12/waio">\n<img src="https://scrutinizer-ci.com/g/dotX12/ShazamIO/badges/quality-score.png?b=master" alt="https://scrutinizer-ci.com/g/dotX12/ShazamIO/">\n<img src="https://scrutinizer-ci.com/g/dotX12/ShazamIO/badges/code-intelligence.svg?b=master" alt="https://scrutinizer-ci.com/g/dotX12/ShazamIO/">\n<img src="https://scrutinizer-ci.com/g/dotX12/ShazamIO/badges/build.png?b=master" alt="https://scrutinizer-ci.com/g/dotX12/ShazamIO/">\n<img src="https://badge.fury.io/py/shazamio.svg" alt="https://badge.fury.io/py/shazamio">\n<img src="https://pepy.tech/badge/shazamio" alt="https://pepy.tech/project/shazamio">\n<img src="https://pepy.tech/badge/shazamio/month" alt="https://pepy.tech/project/shazamio">\n<img src="https://img.shields.io/github/license/dotX12/shazamio.svg" alt="https://github.com/dotX12/ShazamIO/blob/master/LICENSE.txt">\n<br><br>\n  \n  <img width="1000" src="https://user-images.githubusercontent.com/64792903/109359596-ca561a00-7896-11eb-9c93-9cf1f283b1a5.png">\n  🎵 Is a FREE asynchronous library from reverse engineered Shazam API written in Python 3.8+ with asyncio and aiohttp. Includes all the methods that Shazam has, including searching for a song by file.\n \n-----\n</p>\n\n## 💿 Installation\n\n```\n💲 pip install shazamio\n```\n\n## 💻 Example\n\n\n<details> \n<summary>\n<i>🔎🎵 Recognize track</i>\n</summary>\n\nRecognize a track based on a file<br>\n\n  ```python3\nimport asyncio\nfrom shazamio import Shazam\n\n\nasync def main():\n    shazam = Shazam()\n    out = await shazam.recognize_song(\'dora.ogg\')\n    print(out)\n\nloop = asyncio.get_event_loop()\nloop.run_until_complete(main())\n  ```\n</details>\n\n<details> \n<summary>\n<i>👨\u200d🎤 About artist</i>\n</summary>\n\nRetrieving information from an artist profile<br>\n<a href="https://www.shazam.com/artist/43328183/nathan-evans">https://www.shazam.com/artist/43328183/nathan-evans</a>\n\n  ```python3\nimport asyncio\nfrom shazamio import Shazam, Serialize\n\n\nasync def main():\n    shazam = Shazam()\n    artist_id = 43328183\n    about_artist = await shazam.artist_about(artist_id)\n    serialized = Serialize.artist(about_artist)\n\n    print(about_artist)  # dict\n    print(serialized)  # serialized from dataclass factory\n\nloop = asyncio.get_event_loop()\nloop.run_until_complete(main())\n  ```\n</details>\n\n\n<details> \n<summary>\n<i>🎵📄 About track</i>\n</summary>\n\nGet track information<br>\n<a href="https://www.shazam.com/track/552406075/ale-jazz">https://www.shazam.com/track/552406075/ale-jazz</a>\n\n  ```python3\nimport asyncio\nfrom shazamio import Shazam, Serialize\n\n\nasync def main():\n    shazam = Shazam()\n    track_id = 552406075\n    about_track = await shazam.track_about(track_id=track_id)\n    serialized = Serialize.track(data=about_track)\n\n    print(about_track)  # dict\n    print(serialized)  # serialized from dataclass factory\n\nloop = asyncio.get_event_loop()\nloop.run_until_complete(main())\n  ```\n</details>\n\n<details> \n<summary>\n<i>🎵⌛ Track listenings count</i>\n</summary>\n\nReturns the number of times a particular song has been played<br>\n<a href="https://www.shazam.com/track/559284007/rampampam">https://www.shazam.com/track/559284007/rampampam</a>\n\n  ```python3\nimport asyncio\nfrom shazamio import Shazam\n\n\nasync def main():\n    # Example: https://www.shazam.com/track/559284007/rampampam\n\n    shazam = Shazam()\n    track_id = 559284007\n    count = await shazam.listening_counter(track_id=track_id)\n    print(count)\n\nloop = asyncio.get_event_loop()\nloop.run_until_complete(main())\n  ```\n</details>\n\n<details> \n<summary>\n<i>🎶💬 Similar songs</i>\n</summary>\n\nSimilar songs based song id<br>\n<a href="https://www.shazam.com/track/546891609/2-phu%CC%81t-ho%CC%9Bn-kaiz-remix">https://www.shazam.com/track/546891609/2-phu%CC%81t-ho%CC%9Bn-kaiz-remix</a>\n\n  ```python3\nimport asyncio\nfrom shazamio import Shazam\n\n\nasync def main():\n    shazam = Shazam()\n    track_id = 546891609\n    related = await shazam.related_tracks(track_id=track_id, limit=5, start_from=2)\n    # ONLY №3, №4 SONG\n    print(related)\n\nloop = asyncio.get_event_loop()\nloop.run_until_complete(main())\n  ```\n</details>\n\n<details> \n<summary>\n<i>🔎👨\u200d🎤 Search artists</i>\n</summary>\n\nSearch all artists by prefix<br>\n  ```python3\nimport asyncio\nfrom shazamio import Shazam, Serialize\n\n\nasync def main():\n    shazam = Shazam()\n    artists = await shazam.search_artist(query=\'Lil\', limit=5)\n    for artist in artists[\'artists\'][\'hits\']:\n        serialized = Serialize.artist(data=artist)\n        print(serialized)\n\nloop = asyncio.get_event_loop()\nloop.run_until_complete(main())\n\n  ```\n</details>\n\n<details> \n<summary>\n<i>🔎🎶 Search tracks</i>\n</summary>\n\nSearch all tracks by prefix<br>\n\n  ```python3\nimport asyncio\nfrom shazamio import Shazam\n\n\nasync def main():\n    shazam = Shazam()\n    tracks = await shazam.search_track(query=\'Lil\', limit=5)\n    print(tracks)\n\nloop = asyncio.get_event_loop()\nloop.run_until_complete(main())\n\n  ```\n</details>\n\n<details> \n<summary>\n<i>🔝🎶👨\u200d🎤 Top artist tracks</i>\n</summary>\n\nGet the top songs according to Shazam<br>\n<a href="https://www.shazam.com/artist/201896832/kizaru">https://www.shazam.com/artist/201896832/kizaru</a>\n\n  ```python3\nimport asyncio\nfrom shazamio import Shazam, Serialize\n\n\nasync def main():\n    shazam = Shazam()\n    artist_id = 201896832\n    top_three_artist_tracks = await shazam.artist_top_tracks(artist_id=artist_id, limit=3)\n    for track in top_three_artist_tracks[\'tracks\']:\n        serialized_track = Serialize.track(data=track)\n        print(serialized_track)\n\nloop = asyncio.get_event_loop()\nloop.run_until_complete(main())\n\n  ```\n</details>\n\n<details> \n<summary>\n<i>🔝🎶🏙️ Top tracks in city</i>\n</summary>\n\nRetrieving information from an artist profile<br>\n<a href="https://www.shazam.com/charts/top-50/russia/moscow">https://www.shazam.com/charts/top-50/russia/moscow</a>\n\n  ```python3\nimport asyncio\nfrom shazamio import Shazam, Serialize\n\n\nasync def main():\n    shazam = Shazam()\n    top_ten_moscow_tracks = await shazam.top_city_tracks(country_code=\'RU\', city_name=\'Moscow\', limit=10)\n    print(top_ten_moscow_tracks)\n    # ALL TRACKS DICT\n    for track in top_ten_moscow_tracks[\'tracks\']:\n        serialized = Serialize.track(data=track)\n        # SERIALIZE FROM DATACLASS FACTORY\n        print(serialized)\n\nloop = asyncio.get_event_loop()\nloop.run_until_complete(main())\n\n  ```\n</details>\n\n<details> \n<summary>\n<i>🔝🎶🏳️\u200d🌈 Top tracks in country</i>\n</summary>\n\nGet the best tracks by country code<br>\n<a href="https://www.shazam.com/charts/discovery/netherlands">https://www.shazam.com/charts/discovery/netherlands</a>\n\n  ```python3\nimport asyncio\nfrom shazamio import Shazam, Serialize\n\n\nasync def main():\n    shazam = Shazam()\n    top_five_track_from_amsterdam = await shazam.top_country_tracks(\'NL\', 5)\n    for track in top_five_track_from_amsterdam[\'tracks\']:\n        serialized = Serialize.track(data=track)\n        print(serialized)\n\nloop = asyncio.get_event_loop()\nloop.run_until_complete(main())\n  ```\n</details>\n\n<details> \n<summary>\n<i>🔝🎶🏳️\u200d🌈🎸 Top tracks in country by genre</i>\n</summary>\n\nThe best tracks by a genre in the country<br>\n<a href="https://www.shazam.com/charts/genre/spain/hip-hop-rap">https://www.shazam.com/charts/genre/spain/hip-hop-rap</a>\n\n  ```python3\nimport asyncio\nfrom shazamio import Shazam, GenreMusic\n\n\nasync def main():\n    shazam = Shazam()\n    top_spain_rap = await shazam.top_country_genre_tracks(\n        country_code=\'ES\',\n        genre=GenreMusic.HIP_HOP_RAP,\n        limit=4\n    )\n    print(top_spain_rap)\n\nloop = asyncio.get_event_loop()\nloop.run_until_complete(main())\n  ```\n</details>\n\n<details> \n<summary>\n<i>🔝🎶🌏🎸 Top tracks in word by genre</i>\n</summary>\n\nGet world tracks by certain genre<br>\n<a href="https://www.shazam.com/charts/genre/world/rock">https://www.shazam.com/charts/genre/world/rock</a>\n\n  ```python3\nimport asyncio\nfrom shazamio import Shazam, Serialize, GenreMusic\n\n\nasync def main():\n    shazam = Shazam()\n    top_rock_in_the_world = await shazam.top_world_genre_tracks(genre=GenreMusic.ROCK, limit=10)\n\n    for track in top_rock_in_the_world[\'tracks\']:\n        serialized_track = Serialize.track(data=track)\n        print(serialized_track.spotify_url)\n\n\nloop = asyncio.get_event_loop()\nloop.run_until_complete(main())\n  ```\n</details>\n\n<details> \n<summary>\n<i>🔝🎶🌏Top tracks in word</i>\n</summary>\n\nGet the best tracks from all over the world<br>\n<a href="https://www.shazam.com/charts/top-200/world">https://www.shazam.com/charts/top-200/world</a>\n\n  ```python3\nimport asyncio\nfrom shazamio import Shazam, Serialize\n\n\nasync def main():\n    shazam = Shazam()\n    top_world_tracks = await shazam.top_world_tracks(limit=10)\n    print(top_world_tracks)\n    for track in top_world_tracks[\'tracks\']:\n        serialized = Serialize.track(track)\n        print(serialized)\n\nloop = asyncio.get_event_loop()\nloop.run_until_complete(main())\n  ```\n</details>\n\n\n## How to use data serialization\n\n<details> \n<summary>\n<i>Open Code</i>\n</summary>\n\n  ```python3\nimport asyncio\nfrom shazamio import Shazam, Serialize\n\n\nasync def main():\n    shazam = Shazam()\n    top_five_track_from_amsterdam = await shazam.top_country_tracks(\'NL\', 5)\n    for track in top_five_track_from_amsterdam[\'tracks\']:\n        serialized = Serialize.track(data=track)\n        print(serialized.title)\n\nloop = asyncio.get_event_loop()\nloop.run_until_complete(main())\n  ```\n</details>\n\n<details> \n<summary>\n<i>Open photo: What song information looks like (Dict)</i>\n</summary>\n<img src="https://user-images.githubusercontent.com/64792903/109454521-75b4c980-7a65-11eb-917e-62da3abefb8a.png">\n\n</details>\n\n<details> \n<summary>\n<i>Open photo: what song information looks like (Custom serializer)</i>\n</summary>\n<img src="https://user-images.githubusercontent.com/64792903/109454465-57e76480-7a65-11eb-956c-1bcac41d7de5.png">\n\n</details>\n\nAgree, thanks to the serializer, you no longer need to manually select the necessary data from the dictionary. Now the serializer contains the most necessary information about an artist or a track.\n',
    'author': 'dotX12',
    'author_email': 'None',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/dotX12/ShazamIO',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
