"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const cdk = require("@aws-cdk/core");
const sns = require("../lib");
module.exports = {
    'create a subscription'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription', {
            Endpoint: 'endpoint',
            Protocol: 'lambda',
            TopicArn: {
                Ref: 'TopicBFC7AF6E',
            },
        }));
        test.done();
    },
    'create a subscription with DLQ when client provides DLQ'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const dlQueue = new aws_sqs_1.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MySubscription_DLQ',
            retentionPeriod: cdk.Duration.days(14),
        });
        // WHEN
        new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
            deadLetterQueue: dlQueue,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription', {
            Endpoint: 'endpoint',
            Protocol: 'lambda',
            TopicArn: {
                Ref: 'TopicBFC7AF6E',
            },
            RedrivePolicy: {
                deadLetterTargetArn: {
                    'Fn::GetAtt': [
                        'DeadLetterQueue9F481546',
                        'Arn',
                    ],
                },
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::Queue', {
            QueueName: 'MySubscription_DLQ',
            MessageRetentionPeriod: 1209600,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SQS::QueuePolicy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'sqs:SendMessage',
                        Condition: {
                            ArnEquals: {
                                'aws:SourceArn': {
                                    Ref: 'TopicBFC7AF6E',
                                },
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'sns.amazonaws.com',
                        },
                        Resource: {
                            'Fn::GetAtt': [
                                'DeadLetterQueue9F481546',
                                'Arn',
                            ],
                        },
                    },
                ],
                Version: '2012-10-17',
            },
            Queues: [
                {
                    Ref: 'DeadLetterQueue9F481546',
                },
            ],
        }));
        test.done();
    },
    'with filter policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            filterPolicy: {
                color: sns.SubscriptionFilter.stringFilter({
                    whitelist: ['red', 'green'],
                    blacklist: ['white', 'orange'],
                    matchPrefixes: ['bl', 'ye'],
                }),
                price: sns.SubscriptionFilter.numericFilter({
                    whitelist: [100, 200],
                    between: { start: 300, stop: 350 },
                    greaterThan: 500,
                    lessThan: 1000,
                    betweenStrict: { start: 2000, stop: 3000 },
                    greaterThanOrEqualTo: 1000,
                    lessThanOrEqualTo: -2,
                }),
            },
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription', {
            FilterPolicy: {
                color: [
                    'red',
                    'green',
                    { 'anything-but': ['white', 'orange'] },
                    { prefix: 'bl' },
                    { prefix: 'ye' },
                ],
                price: [
                    { numeric: ['=', 100] },
                    { numeric: ['=', 200] },
                    { numeric: ['>', 500] },
                    { numeric: ['>=', 1000] },
                    { numeric: ['<', 1000] },
                    { numeric: ['<=', -2] },
                    { numeric: ['>=', 300, '<=', 350] },
                    { numeric: ['>', 2000, '<', 3000] },
                ],
            },
        }));
        test.done();
    },
    'with existsFilter'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            filterPolicy: {
                size: sns.SubscriptionFilter.existsFilter(),
            },
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SNS::Subscription', {
            FilterPolicy: {
                size: [{ exists: true }],
            },
        }));
        test.done();
    },
    'throws with raw delivery for protocol other than http, https or sqs'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // THEN
        test.throws(() => new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
            rawMessageDelivery: true,
        }), /Raw message delivery/);
        test.done();
    },
    'throws with more than 5 attributes in a filter policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        const cond = { conditions: [] };
        // THEN
        test.throws(() => new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
            filterPolicy: {
                a: cond,
                b: cond,
                c: cond,
                d: cond,
                e: cond,
                f: cond,
            },
        }), /5 attribute names/);
        test.done();
    },
    'throws with more than 100 conditions in a filter policy'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // THEN
        test.throws(() => new sns.Subscription(stack, 'Subscription', {
            endpoint: 'endpoint',
            protocol: sns.SubscriptionProtocol.LAMBDA,
            topic,
            filterPolicy: {
                a: { conditions: [...Array.from(Array(2).keys())] },
                b: { conditions: [...Array.from(Array(10).keys())] },
                c: { conditions: [...Array.from(Array(6).keys())] },
            },
        }), /\(120\) must not exceed 100/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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