# -*- coding: utf-8 -*-
#
# Copyright 2017-2022 - Swiss Data Science Center (SDSC)
# A partnership between École Polytechnique Fédérale de Lausanne (EPFL) and
# Eidgenössische Technische Hochschule Zürich (ETHZ).
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Workflow file parser."""

from __future__ import annotations

from abc import ABCMeta, abstractmethod
from pathlib import Path
from typing import TYPE_CHECKING, Tuple, Union

if TYPE_CHECKING:
    from renku.core.workflow.model.workflow_file import WorkflowFile


class IWorkflowFileParser(metaclass=ABCMeta):
    """Abstract class for parsing a workflow file."""

    @abstractmethod
    def workflow_file_parser(self) -> Tuple[IWorkflowFileParser, str]:
        """Supported workflow description formats.

        Returns:
            Tuple[IWorkflowParser, str]: A tuple of the provider itself and the workflow parser name.
        """
        raise NotImplementedError

    @abstractmethod
    def parse(self, path: Union[Path, str]) -> "WorkflowFile":
        """Parse a given workflow file using the provider.

        Args:
            path(Union[Path, str]): Path to the workflow file to parse.

        Returns:
            WorkflowFile: A ``WorkflowFile`` generated by the provider.
        """
        raise NotImplementedError
