import { Store, useStoreState } from 'pullstate';
import { UIStore } from '../stores/UIStore';
import { COMM_NAME_KERNEL, COMM_NAME_FRONTEND, METADATA_KEY } from '../const';
import { actions } from '../utils/Actions';
import { InjectNotebookToolbar } from '../InjectNotebookToolbar';
const StatusStore = new Store({ status: {} });
export function useNotebookResolveStatusStore() {
    const resolveStatus = useStoreState(StatusStore, s => s.status);
    return resolveStatus;
}
export class NotebookListener {
    constructor(options) {
        this.kernelNotebookMapping = {};
        this.options = options;
        this.setup();
    }
    setup() {
        const { notebookTracker } = this.options;
        const instanceConfigurationChange = () => {
            console.log('[Listener] Instance config changed!');
            StatusStore.update(s => {
                s.status = {};
            });
            notebookTracker.forEach(p => this.reinject(p));
        };
        UIStore.subscribe(s => s.activeInstance, activeInstance => {
            if (activeInstance) {
                instanceConfigurationChange();
            }
        });
        notebookTracker.widgetAdded.connect(this.onNotebookOpened, this);
    }
    onNotebookOpened(sender, panel) {
        panel.sessionContext.statusChanged.connect((sender, status) => {
            if (status === 'restarting') {
                this.onKernelRestarted(panel, sender.session.kernel);
            }
        });
        panel.sessionContext.kernelChanged.connect((sender, changed) => {
            const oldKernel = changed.oldValue;
            if (oldKernel) {
                this.onKernelDetached(panel, oldKernel);
            }
            const newKernel = changed.newValue;
            if (newKernel) {
                panel.sessionContext.ready.then(() => {
                    this.onKernelAttached(panel, newKernel);
                });
            }
        });
        this.insertInjectButton(panel);
    }
    insertInjectButton(notebookPanel) {
        const onClick = () => {
            var _a, _b;
            if ((_b = (_a = notebookPanel.sessionContext) === null || _a === void 0 ? void 0 : _a.session) === null || _b === void 0 ? void 0 : _b.kernel) {
                this.reinject(notebookPanel);
            }
        };
        const injectNotebookButtonWidget = new InjectNotebookToolbar({ notebookPanel, onClick });
        notebookPanel.toolbar.insertAfter('spacer', 'InjectButton', injectNotebookButtonWidget);
    }
    onKernelRestarted(notebook, kernelConnection) {
        this.clearKernelResolverStatus(kernelConnection.id);
    }
    onKernelDetached(notebook, kernelConnection) {
        this.clearKernelResolverStatus(kernelConnection.id);
        this.deleteKernelNotebookMapping(kernelConnection.id);
    }
    onKernelAttached(notebook, kernelConnection) {
        this.setKernelNotebookMapping(kernelConnection.id, notebook.id);
        this.setupKernelReceiverComm(kernelConnection);
        const activeNotebookAttachments = this.getAttachmentsFromMetadata(notebook);
        this.injectAttachments(kernelConnection, activeNotebookAttachments);
    }
    clearKernelResolverStatus(kernelConnectionId) {
        const notebookId = this.getNotebookIdFromKernelConnectionId(kernelConnectionId);
        StatusStore.update(s => {
            s.status[notebookId] = {};
        });
    }
    setupKernelReceiverComm(kernelConnection) {
        kernelConnection.registerCommTarget(COMM_NAME_FRONTEND, (comm, openMsg) => {
            comm.onMsg = msg => {
                this.processIncomingMessage(kernelConnection, comm, msg);
            };
            this.processIncomingMessage(kernelConnection, comm, openMsg);
        });
    }
    processIncomingMessage(kernelConnection, comm, msg) {
        const data = msg.content.data;
        console.log('Incoming message', data);
        if (data.action === 'request-inject') {
            const { activeInstance } = UIStore.getRawState();
            const notebookId = this.getNotebookIdFromKernelConnectionId(kernelConnection.id);
            const { notebookTracker } = this.options;
            const notebook = notebookTracker.find(p => p.id === notebookId);
            const activeNotebookAttachments = this.getAttachmentsFromMetadata(notebook);
            if (!activeNotebookAttachments || !activeInstance) {
                return;
            }
            this.resolveAttachments(activeNotebookAttachments, kernelConnection.id).then(injections => {
                console.log('Replying request-inject', injections);
                return comm
                    .send({ action: 'inject', dids: injections })
                    .done.then(() => {
                    injections.forEach(injection => this.setResolveStatus(kernelConnection.id, injection.did, 'READY'));
                })
                    .catch(e => {
                    console.error(e);
                    injections.forEach(injection => this.setResolveStatus(kernelConnection.id, injection.did, 'FAILED'));
                });
            });
        }
    }
    injectUninjected(notebookPanel) {
        var _a, _b;
        const attachments = this.getAttachmentsFromMetadata(notebookPanel);
        const kernel = (_b = (_a = notebookPanel.sessionContext) === null || _a === void 0 ? void 0 : _a.session) === null || _b === void 0 ? void 0 : _b.kernel;
        this.injectUninjectedAttachments(kernel, attachments);
        this.removeNonExistentInjectedAttachments(kernel === null || kernel === void 0 ? void 0 : kernel.id, attachments);
    }
    reinject(notebookPanel) {
        var _a, _b;
        const attachments = this.getAttachmentsFromMetadata(notebookPanel);
        const kernel = (_b = (_a = notebookPanel.sessionContext) === null || _a === void 0 ? void 0 : _a.session) === null || _b === void 0 ? void 0 : _b.kernel;
        this.injectAttachments(kernel, attachments);
    }
    getAttachmentsFromMetadata(notebook) {
        const rucioDidAttachments = notebook.model.metadata.get(METADATA_KEY);
        const attachedDIDs = rucioDidAttachments;
        return attachedDIDs;
    }
    injectUninjectedAttachments(kernel, attachments) {
        const kernelConnectionId = kernel === null || kernel === void 0 ? void 0 : kernel.id;
        const notebookId = this.getNotebookIdFromKernelConnectionId(kernelConnectionId);
        const notebookStatus = StatusStore.getRawState().status[notebookId] || {};
        const uninjectedAttachments = attachments.filter(a => !notebookStatus[a.did]);
        this.injectAttachments(kernel, uninjectedAttachments);
    }
    removeNonExistentInjectedAttachments(kernelConnectionId, attachments) {
        const notebookId = this.getNotebookIdFromKernelConnectionId(kernelConnectionId);
        StatusStore.update(s => {
            const notebookStatus = s.status[notebookId];
            const injectedDIDs = Object.keys(notebookStatus);
            const nonExistentDIDs = injectedDIDs.filter(did => !attachments.find(a => a.did === did));
            nonExistentDIDs.forEach(nd => delete s.status[notebookId][nd]);
        });
    }
    injectAttachments(kernel, attachments) {
        if (!this.isExtensionProperlySetup()) {
            return;
        }
        this.resolveAttachments(attachments, kernel.id)
            .then(injections => {
            return this.injectVariables(kernel, injections);
        })
            .then(() => {
            attachments.forEach(attachment => this.setResolveStatus(kernel.id, attachment.did, 'READY'));
        })
            .catch(e => {
            console.error(e);
            attachments.forEach(attachment => this.setResolveStatus(kernel.id, attachment.did, 'FAILED'));
        });
    }
    injectVariables(kernel, injections) {
        if (injections.length === 0) {
            return;
        }
        console.log('Injecting variables', injections);
        const comm = kernel.createComm(COMM_NAME_KERNEL);
        return comm
            .open()
            .done.then(() => comm.send({ action: 'inject', dids: injections }).done)
            .then(() => comm.close().done);
    }
    async resolveAttachments(attachments, kernelConnectionId) {
        const promises = attachments.map(a => this.resolveAttachment(a, kernelConnectionId));
        return Promise.all(promises);
    }
    async resolveAttachment(attachment, kernelConnectionId) {
        const { variableName, type, did } = attachment;
        this.setResolveStatus(kernelConnectionId, did, 'RESOLVING');
        try {
            if (type === 'container') {
                const didDetails = await this.resolveContainerDIDDetails(did);
                const path = this.getContainerDIDPaths(didDetails);
                this.setResolveStatus(kernelConnectionId, did, 'PENDING_INJECTION');
                return { variableName, path, did };
            }
            else {
                const didDetails = await this.resolveFileDIDDetails(did);
                const path = this.getFileDIDPaths(didDetails);
                this.setResolveStatus(kernelConnectionId, did, 'PENDING_INJECTION');
                return { variableName, path, did };
            }
        }
        catch (e) {
            this.setResolveStatus(kernelConnectionId, did, 'FAILED');
            throw e;
        }
    }
    setResolveStatus(kernelConnectionId, did, status) {
        const notebookId = this.getNotebookIdFromKernelConnectionId(kernelConnectionId);
        StatusStore.update(s => {
            if (!s.status[notebookId]) {
                s.status[notebookId] = {};
            }
            s.status[notebookId][did] = status;
        });
    }
    getContainerDIDPaths(didDetails) {
        return didDetails.map(d => d.path).filter(p => !!p);
    }
    getFileDIDPaths(didDetails) {
        return didDetails.path;
    }
    async resolveFileDIDDetails(did) {
        const { activeInstance } = UIStore.getRawState();
        return actions.getFileDIDDetails(activeInstance.name, did);
    }
    async resolveContainerDIDDetails(did) {
        const { activeInstance } = UIStore.getRawState();
        return actions.getContainerDIDDetails(activeInstance.name, did);
    }
    getNotebookIdFromKernelConnectionId(kernelConnectionId) {
        return this.kernelNotebookMapping[kernelConnectionId];
    }
    setKernelNotebookMapping(kernelConnectionId, notebookId) {
        this.kernelNotebookMapping[kernelConnectionId] = notebookId;
    }
    deleteKernelNotebookMapping(kernelConnectionId) {
        delete this.kernelNotebookMapping[kernelConnectionId];
    }
    isExtensionProperlySetup() {
        const { activeInstance } = UIStore.getRawState();
        return !!activeInstance;
    }
}
