"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProxyResource = exports.Resource = exports.ResourceBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const cors_1 = require("./cors");
const integrations_1 = require("./integrations");
const method_1 = require("./method");
/**
 * @stability stable
 */
class ResourceBase extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id) {
        super(scope, id);
        this.children = {};
    }
    /**
     * Defines a new child resource where this resource is the parent.
     *
     * @stability stable
     */
    addResource(pathPart, options) {
        return new Resource(this, pathPart, { parent: this, pathPart, ...options });
    }
    /**
     * Defines a new method for this resource.
     *
     * @stability stable
     */
    addMethod(httpMethod, integration, options) {
        return new method_1.Method(this, httpMethod, { resource: this, httpMethod, integration, options });
    }
    /**
     * Adds a greedy proxy resource ("{proxy+}") and an ANY method to this route.
     *
     * @stability stable
     */
    addProxy(options) {
        return new ProxyResource(this, '{proxy+}', { parent: this, ...options });
    }
    /**
     * Adds an OPTIONS method to this resource which responds to Cross-Origin Resource Sharing (CORS) preflight requests.
     *
     * Cross-Origin Resource Sharing (CORS) is a mechanism that uses additional
     * HTTP headers to tell browsers to give a web application running at one
     * origin, access to selected resources from a different origin. A web
     * application executes a cross-origin HTTP request when it requests a
     * resource that has a different origin (domain, protocol, or port) from its
     * own.
     *
     * @stability stable
     */
    addCorsPreflight(options) {
        var _d;
        const headers = {};
        //
        // Access-Control-Allow-Headers
        const allowHeaders = options.allowHeaders || cors_1.Cors.DEFAULT_HEADERS;
        headers['Access-Control-Allow-Headers'] = `'${allowHeaders.join(',')}'`;
        //
        // Access-Control-Allow-Origin
        if (options.allowOrigins.length === 0) {
            throw new Error('allowOrigins must contain at least one origin');
        }
        if (options.allowOrigins.includes('*') && options.allowOrigins.length > 1) {
            throw new Error(`Invalid "allowOrigins" - cannot mix "*" with specific origins: ${options.allowOrigins.join(',')}`);
        }
        // we use the first origin here and if there are more origins in the list, we
        // will match against them in the response velocity template
        const initialOrigin = options.allowOrigins[0];
        headers['Access-Control-Allow-Origin'] = `'${initialOrigin}'`;
        // the "Vary" header is required if we allow a specific origin
        // https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin#CORS_and_caching
        if (initialOrigin !== '*') {
            headers.Vary = '\'Origin\'';
        }
        //
        // Access-Control-Allow-Methods
        let allowMethods = options.allowMethods || cors_1.Cors.ALL_METHODS;
        if (allowMethods.includes('ANY')) {
            if (allowMethods.length > 1) {
                throw new Error(`ANY cannot be used with any other method. Received: ${allowMethods.join(',')}`);
            }
            allowMethods = cors_1.Cors.ALL_METHODS;
        }
        headers['Access-Control-Allow-Methods'] = `'${allowMethods.join(',')}'`;
        //
        // Access-Control-Allow-Credentials
        if (options.allowCredentials) {
            headers['Access-Control-Allow-Credentials'] = '\'true\'';
        }
        //
        // Access-Control-Max-Age
        let maxAgeSeconds;
        if (options.maxAge && options.disableCache) {
            throw new Error('The options "maxAge" and "disableCache" are mutually exclusive');
        }
        if (options.maxAge) {
            maxAgeSeconds = options.maxAge.toSeconds();
        }
        if (options.disableCache) {
            maxAgeSeconds = -1;
        }
        if (maxAgeSeconds) {
            headers['Access-Control-Max-Age'] = `'${maxAgeSeconds}'`;
        }
        //
        // Access-Control-Expose-Headers
        //
        if (options.exposeHeaders) {
            headers['Access-Control-Expose-Headers'] = `'${options.exposeHeaders.join(',')}'`;
        }
        //
        // statusCode
        const statusCode = (_d = options.statusCode) !== null && _d !== void 0 ? _d : 204;
        //
        // prepare responseParams
        const integrationResponseParams = {};
        const methodResponseParams = {};
        for (const [name, value] of Object.entries(headers)) {
            const key = `method.response.header.${name}`;
            integrationResponseParams[key] = value;
            methodResponseParams[key] = true;
        }
        return this.addMethod('OPTIONS', new integrations_1.MockIntegration({
            requestTemplates: { 'application/json': '{ statusCode: 200 }' },
            integrationResponses: [
                { statusCode: `${statusCode}`, responseParameters: integrationResponseParams, responseTemplates: renderResponseTemplate() },
            ],
        }), {
            methodResponses: [
                { statusCode: `${statusCode}`, responseParameters: methodResponseParams },
            ],
        });
        // renders the response template to match all possible origins (if we have more than one)
        function renderResponseTemplate() {
            const origins = options.allowOrigins.slice(1);
            if (origins.length === 0) {
                return undefined;
            }
            const template = new Array();
            template.push('#set($origin = $input.params("Origin"))');
            template.push('#if($origin == "") #set($origin = $input.params("origin")) #end');
            const condition = origins.map(o => `$origin.matches("${o}")`).join(' || ');
            template.push(`#if(${condition})`);
            template.push('  #set($context.responseOverride.header.Access-Control-Allow-Origin = $origin)');
            template.push('#end');
            return {
                'application/json': template.join('\n'),
            };
        }
    }
    /**
     * Retrieves a child resource by path part.
     *
     * @stability stable
     */
    getResource(pathPart) {
        return this.children[pathPart];
    }
    /**
     * @internal
     */
    _trackChild(pathPart, resource) {
        this.children[pathPart] = resource;
    }
    /**
     * Gets or create all resources leading up to the specified path.
     *
     * - Path may only start with "/" if this method is called on the root resource.
     * - All resources are created using default options.
     *
     * @stability stable
     */
    resourceForPath(path) {
        if (!path) {
            return this;
        }
        if (path.startsWith('/')) {
            if (this.path !== '/') {
                throw new Error(`Path may start with "/" only for the resource, but we are at: ${this.path}`);
            }
            // trim trailing "/"
            return this.resourceForPath(path.substr(1));
        }
        const parts = path.split('/');
        const next = parts.shift();
        if (!next || next === '') {
            throw new Error('resourceForPath cannot be called with an empty path');
        }
        let resource = this.getResource(next);
        if (!resource) {
            resource = this.addResource(next);
        }
        return resource.resourceForPath(parts.join('/'));
    }
    /**
     * @deprecated - Throws error in some use cases that have been enabled since this deprecation notice. Use `RestApi.urlForPath()` instead.
     */
    get url() {
        return this.restApi.urlForPath(this.path);
    }
}
exports.ResourceBase = ResourceBase;
_a = JSII_RTTI_SYMBOL_1;
ResourceBase[_a] = { fqn: "@aws-cdk/aws-apigateway.ResourceBase", version: "1.110.0" };
/**
 * @stability stable
 */
class Resource extends ResourceBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        validateResourcePathPart(props.pathPart);
        this.parentResource = props.parent;
        if (props.parent instanceof ResourceBase) {
            props.parent._trackChild(props.pathPart, this);
        }
        const resourceProps = {
            restApiId: props.parent.api.restApiId,
            parentId: props.parent.resourceId,
            pathPart: props.pathPart,
        };
        const resource = new apigateway_generated_1.CfnResource(this, 'Resource', resourceProps);
        this.resourceId = resource.ref;
        this.api = props.parent.api;
        // render resource path (special case for root)
        this.path = props.parent.path;
        if (!this.path.endsWith('/')) {
            this.path += '/';
        }
        this.path += props.pathPart;
        const deployment = props.parent.api.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ resource: resourceProps });
        }
        // setup defaults based on properties and inherit from parent. method defaults
        // are inherited per property, so children can override piecemeal.
        this.defaultIntegration = props.defaultIntegration || props.parent.defaultIntegration;
        this.defaultMethodOptions = {
            ...props.parent.defaultMethodOptions,
            ...props.defaultMethodOptions,
        };
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions || props.parent.defaultCorsPreflightOptions;
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
    /**
     * Import an existing resource.
     *
     * @stability stable
     */
    static fromResourceAttributes(scope, id, attrs) {
        class Import extends ResourceBase {
            constructor() {
                super(...arguments);
                this.api = attrs.restApi;
                this.resourceId = attrs.resourceId;
                this.path = attrs.path;
                this.defaultIntegration = undefined;
                this.defaultMethodOptions = undefined;
                this.defaultCorsPreflightOptions = undefined;
            }
            get parentResource() {
                throw new Error('parentResource is not configured for imported resource.');
            }
            get restApi() {
                throw new Error('restApi is not configured for imported resource.');
            }
        }
        return new Import(scope, id);
    }
    /**
     * (deprecated) The RestApi associated with this Resource.
     *
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        if (!this.parentResource) {
            throw new Error('parentResource was unexpectedly not defined');
        }
        return this.parentResource.restApi;
    }
}
exports.Resource = Resource;
_b = JSII_RTTI_SYMBOL_1;
Resource[_b] = { fqn: "@aws-cdk/aws-apigateway.Resource", version: "1.110.0" };
/**
 * Defines a {proxy+} greedy resource and an ANY method on a route.
 *
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-set-up-simple-proxy.html
 * @stability stable
 */
class ProxyResource extends Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d;
        super(scope, id, {
            parent: props.parent,
            pathPart: '{proxy+}',
            defaultIntegration: props.defaultIntegration,
            defaultMethodOptions: props.defaultMethodOptions,
        });
        const anyMethod = (_d = props.anyMethod) !== null && _d !== void 0 ? _d : true;
        if (anyMethod) {
            this.anyMethod = this.addMethod('ANY');
        }
    }
    /**
     * Defines a new method for this resource.
     *
     * @stability stable
     */
    addMethod(httpMethod, integration, options) {
        // In case this proxy is mounted under the root, also add this method to
        // the root so that empty paths are proxied as well.
        if (this.parentResource && this.parentResource.path === '/') {
            // skip if the root resource already has this method defined
            if (!(this.parentResource.node.tryFindChild(httpMethod) instanceof method_1.Method)) {
                this.parentResource.addMethod(httpMethod, integration, options);
            }
        }
        return super.addMethod(httpMethod, integration, options);
    }
}
exports.ProxyResource = ProxyResource;
_c = JSII_RTTI_SYMBOL_1;
ProxyResource[_c] = { fqn: "@aws-cdk/aws-apigateway.ProxyResource", version: "1.110.0" };
function validateResourcePathPart(part) {
    // strip {} which indicate this is a parameter
    if (part.startsWith('{') && part.endsWith('}')) {
        part = part.substr(1, part.length - 2);
        // proxy resources are allowed to end with a '+'
        if (part.endsWith('+')) {
            part = part.substr(0, part.length - 1);
        }
    }
    if (!/^[a-zA-Z0-9\.\_\-]+$/.test(part)) {
        throw new Error(`Resource's path part only allow [a-zA-Z0-9._-], an optional trailing '+'
      and curly braces at the beginning and the end: ${part}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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