"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LowCostECS = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const lib = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const ecs = require("aws-cdk-lib/aws-ecs");
const aws_efs_1 = require("aws-cdk-lib/aws-efs");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const route53 = require("aws-cdk-lib/aws-route53");
const aws_sns_1 = require("aws-cdk-lib/aws-sns");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const sfn_tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
;
class LowCostECS extends lib.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.vpc =
            props.vpc ??
                new ec2.Vpc(this, 'Vpc', {
                    natGateways: 0,
                    subnetConfiguration: [
                        {
                            name: 'PublicSubnet',
                            subnetType: ec2.SubnetType.PUBLIC,
                        },
                    ],
                });
        this.cluster = new ecs.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            containerInsights: props.containerInsights,
        });
        this.hostAutoScalingGroup = this.cluster.addCapacity('HostInstanceCapacity', {
            machineImage: ecs.EcsOptimizedImage.amazonLinux2(ecs.AmiHardwareType.STANDARD, {
                cachedInContext: true,
            }),
            instanceType: new ec2.InstanceType(props.hostInstanceType ?? 't2.micro'),
            spotPrice: props.hostInstanceSpotPrice,
            vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC },
            associatePublicIpAddress: true,
            minCapacity: 1,
            maxCapacity: 1,
        });
        if (props.securityGroup) {
            this.hostAutoScalingGroup.addSecurityGroup(props.securityGroup);
        }
        else {
            this.hostAutoScalingGroup.connections.allowFromAnyIpv4(ec2.Port.tcp(80));
            this.hostAutoScalingGroup.connections.allowFromAnyIpv4(ec2.Port.tcp(443));
            this.hostAutoScalingGroup.connections.allowFrom(ec2.Peer.anyIpv6(), ec2.Port.tcp(80));
            this.hostAutoScalingGroup.connections.allowFrom(ec2.Peer.anyIpv6(), ec2.Port.tcp(443));
        }
        /**
         * Add managed policy to allow ssh through ssm manager
         */
        this.hostAutoScalingGroup.role.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        /**
         * Add policy to associate elastic ip on startup
         */
        this.hostAutoScalingGroup.role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['ec2:DescribeAddresses', 'ec2:AssociateAddress'],
            resources: ['*'],
        }));
        const hostInstanceIp = new ec2.CfnEIP(this, 'HostInstanceIp');
        const tagUniqueId = lib.Names.uniqueId(hostInstanceIp);
        hostInstanceIp.tags.setTag('Name', tagUniqueId);
        const awsCliTag = props.awsCliDockerTag ?? 'latest';
        this.hostAutoScalingGroup.addUserData('INSTANCE_ID=$(curl --silent http://169.254.169.254/latest/meta-data/instance-id)', `ALLOCATION_ID=$(docker run --net=host amazon/aws-cli:${awsCliTag} ec2 describe-addresses --region ${this.hostAutoScalingGroup.env.region} --filter Name=tag:Name,Values=${tagUniqueId} --query 'Addresses[].AllocationId' --output text | head)`, `docker run --net=host amazon/aws-cli:${awsCliTag} ec2 associate-address --region ${this.hostAutoScalingGroup.env.region} --instance-id "$INSTANCE_ID" --allocation-id "$ALLOCATION_ID" --allow-reassociation`);
        this.certFileSystem = new aws_efs_1.FileSystem(this, 'FileSystem', {
            vpc: this.vpc,
            encrypted: true,
            securityGroup: new ec2.SecurityGroup(this, 'FileSystemSecurityGroup', {
                vpc: this.vpc,
                allowAllOutbound: false,
            }),
            removalPolicy: props.removalPolicy ?? lib.RemovalPolicy.DESTROY,
        });
        this.certFileSystem.connections.allowDefaultPortTo(this.hostAutoScalingGroup);
        this.certFileSystem.connections.allowDefaultPortFrom(this.hostAutoScalingGroup);
        /**
         * ARecord to Elastic ip
         */
        const hostedZone = route53.HostedZone.fromLookup(this, 'HostedZone', {
            domainName: props.hostedZoneDomain,
        });
        const records = props.recordDomainNames ?? [hostedZone.zoneName];
        records.forEach((record) => new route53.ARecord(this, `ARecord${record}`, {
            zone: hostedZone,
            recordName: record,
            target: route53.RecordTarget.fromIpAddresses(hostInstanceIp.ref),
        }));
        /**
         * Certbot Task Definition
         * Mounts generated certificate to EFS
         */
        const logGroup = props.logGroup ??
            new aws_logs_1.LogGroup(this, 'LogGroup', {
                retention: aws_logs_1.RetentionDays.TWO_YEARS,
                removalPolicy: props.removalPolicy ?? lib.RemovalPolicy.DESTROY,
            });
        const certbotTaskDefinition = new ecs.Ec2TaskDefinition(this, 'CertbotTaskDefinition');
        certbotTaskDefinition.addToTaskRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['route53:ListHostedZones', 'route53:GetChange'],
            resources: ['*'],
        }));
        certbotTaskDefinition.addToTaskRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['route53:ChangeResourceRecordSets'],
            resources: [hostedZone.hostedZoneArn],
        }));
        const certbotTag = props.certbotDockerTag ?? 'v1.29.0';
        const certbotContainer = certbotTaskDefinition.addContainer('CertbotContainer', {
            image: ecs.ContainerImage.fromRegistry(`certbot/dns-route53:${certbotTag}`),
            containerName: 'certbot',
            memoryReservationMiB: 64,
            command: [
                'certonly',
                '--verbose',
                '--preferred-challenges=dns-01',
                '--dns-route53',
                '--dns-route53-propagation-seconds=300',
                '--non-interactive',
                '--agree-tos',
                '--expand',
                '-m',
                props.email,
                '--cert-name',
                records[0],
                ...records.flatMap((domain) => ['-d', domain]),
            ],
            logging: ecs.LogDriver.awsLogs({
                logGroup,
                streamPrefix: certbotTag,
            }),
        });
        this.certFileSystem.grant(certbotTaskDefinition.taskRole, 'elasticfilesystem:ClientWrite');
        certbotTaskDefinition.addVolume({
            name: 'certVolume',
            efsVolumeConfiguration: {
                fileSystemId: this.certFileSystem.fileSystemId,
            },
        });
        certbotContainer.addMountPoints({
            sourceVolume: 'certVolume',
            containerPath: '/etc/letsencrypt',
            readOnly: false,
        });
        /**
         * Schedule Certbot certificate create/renew on Step Functions
         * Sends email notification on certbot failure
         */
        const topic = new aws_sns_1.Topic(this, 'Topic');
        new aws_sns_1.Subscription(this, 'EmailSubscription', {
            topic: topic,
            protocol: aws_sns_1.SubscriptionProtocol.EMAIL,
            endpoint: props.email,
        });
        const certbotRunTask = new sfn_tasks.EcsRunTask(this, 'CreateCertificate', {
            cluster: this.cluster,
            taskDefinition: certbotTaskDefinition,
            launchTarget: new sfn_tasks.EcsEc2LaunchTarget(),
            integrationPattern: sfn.IntegrationPattern.RUN_JOB,
        });
        certbotRunTask.addCatch(new sfn_tasks.SnsPublish(this, 'SendEmailOnFailure', {
            topic: topic,
            message: sfn.TaskInput.fromJsonPathAt('$'),
        }).next(new sfn.Fail(this, 'Fail')));
        certbotRunTask.addRetry({
            interval: lib.Duration.seconds(20),
        });
        const certbotStateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: certbotRunTask,
        });
        new aws_events_1.Rule(this, 'CertbotScheduleRule', {
            schedule: aws_events_1.Schedule.rate(lib.Duration.days(props.certbotScheduleInterval ?? 60)),
            targets: [new aws_events_targets_1.SfnStateMachine(certbotStateMachine)],
        });
        /**
         * Server ECS task
         */
        const serverTaskDefinition = props.serverTaskDefinition ?? this.sampleSeverTask(records, logGroup);
        this.certFileSystem.grant(serverTaskDefinition.taskRole, 'elasticfilesystem:ClientMount');
        serverTaskDefinition.addVolume({
            name: 'certVolume',
            efsVolumeConfiguration: {
                fileSystemId: this.certFileSystem.fileSystemId,
            },
        });
        serverTaskDefinition.defaultContainer?.addMountPoints({
            sourceVolume: 'certVolume',
            containerPath: '/etc/letsencrypt',
            readOnly: true,
        });
        /**
         * AWS cli container to execute certbot sfn before the default container startup.
         */
        serverTaskDefinition.defaultContainer?.addContainerDependencies({
            container: serverTaskDefinition.addContainer('AWSCliContainer', {
                image: ecs.ContainerImage.fromRegistry(`amazon/aws-cli:${awsCliTag}`),
                containerName: 'aws-cli',
                memoryReservationMiB: 64,
                entryPoint: ['/bin/bash', '-c'],
                command: [
                    `set -eux
          aws configure set region ${certbotStateMachine.env.region} && \\
          aws configure set output text && \\
          EXECUTION_ARN=$(aws stepfunctions start-execution --state-machine-arn ${certbotStateMachine.stateMachineArn} --query executionArn) && \\
          until [ $(aws stepfunctions describe-execution --execution-arn "$EXECUTION_ARN" --query status) != RUNNING ];
          do
            echo "Waiting for $EXECUTION_ARN"
            sleep 10
          done`,
                ],
                essential: false,
                logging: ecs.LogDriver.awsLogs({
                    logGroup: logGroup,
                    streamPrefix: awsCliTag,
                }),
            }),
            condition: ecs.ContainerDependencyCondition.COMPLETE,
        });
        certbotStateMachine.grantExecution(serverTaskDefinition.taskRole, 'states:DescribeExecution');
        certbotStateMachine.grantStartExecution(serverTaskDefinition.taskRole);
        this.service = new ecs.Ec2Service(this, 'Service', {
            cluster: this.cluster,
            taskDefinition: serverTaskDefinition,
            desiredCount: 1,
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
            circuitBreaker: {
                rollback: true,
            },
            enableExecuteCommand: true,
        });
        new lib.CfnOutput(this, 'PublicIpAddress', { value: hostInstanceIp.ref });
        new lib.CfnOutput(this, 'certbotStateMachineName', { value: certbotStateMachine.stateMachineName });
        new lib.CfnOutput(this, 'ClusterName', { value: this.cluster.clusterName });
        new lib.CfnOutput(this, 'ServiceName', { value: this.service.serviceName });
    }
    sampleSeverTask(records, logGroup) {
        const nginxTaskDefinition = new ecs.Ec2TaskDefinition(this, 'NginxTaskDefinition');
        const nginxContainer = nginxTaskDefinition.addContainer('NginxContainer', {
            image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../examples/containers/nginx')),
            containerName: 'nginx',
            memoryReservationMiB: 64,
            essential: true,
            environment: {
                SERVER_NAME: records.join(' '),
                CERT_NAME: records[0],
            },
            logging: ecs.LogDrivers.awsLogs({
                logGroup: logGroup,
                streamPrefix: 'nginx-proxy',
            }),
        });
        nginxContainer.addPortMappings({
            hostPort: 80,
            containerPort: 80,
            protocol: ecs.Protocol.TCP,
        }, {
            hostPort: 443,
            containerPort: 443,
            protocol: ecs.Protocol.TCP,
        });
        return nginxTaskDefinition;
    }
}
exports.LowCostECS = LowCostECS;
_a = JSII_RTTI_SYMBOL_1;
LowCostECS[_a] = { fqn: "low-cost-ecs.LowCostECS", version: "0.0.13" };
//# sourceMappingURL=data:application/json;base64,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