"use strict";
const assert_1 = require("@aws-cdk/assert");
const cfn = require("@aws-cdk/aws-cloudformation");
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const cpactions = require("@aws-cdk/aws-codepipeline-actions");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cdk = require("@aws-cdk/core");
const fc = require("fast-check");
const nodeunit = require("nodeunit");
const pipeline_deploy_stack_action_1 = require("../lib/pipeline-deploy-stack-action");
const accountId = fc.array(fc.integer(0, 9), 12, 12).map(arr => arr.join());
class FakeAction {
    constructor(actionName) {
        this.actionProperties = {
            actionName,
            artifactBounds: { minInputs: 0, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            category: codepipeline.ActionCategory.TEST,
            provider: 'Test',
        };
        this.outputArtifact = new codepipeline.Artifact('OutputArtifact');
    }
    bind(_scope, _stage, _options) {
        return {};
    }
    onStateChange(_name, _target, _options) {
        throw new Error('onStateChange() is not available on FakeAction');
    }
}
function getTestStack() {
    return new cdk.Stack(undefined, 'TestStack', { env: { account: '123456789012', region: 'us-east-1' } });
}
function createSelfUpdatingStack(pipelineStack) {
    const pipeline = new codepipeline.Pipeline(pipelineStack, 'CodePipeline', {
        restartExecutionOnUpdate: true,
    });
    // simple source
    const bucket = s3.Bucket.fromBucketArn(pipeline, 'PatternBucket', 'arn:aws:s3:::totally-fake-bucket');
    const sourceOutput = new codepipeline.Artifact('SourceOutput');
    const sourceAction = new cpactions.S3SourceAction({
        actionName: 'S3Source',
        bucket,
        bucketKey: 'the-great-key',
        output: sourceOutput,
    });
    pipeline.addStage({
        stageName: 'source',
        actions: [sourceAction],
    });
    const project = new codebuild.PipelineProject(pipelineStack, 'CodeBuild');
    const buildOutput = new codepipeline.Artifact('BuildOutput');
    const buildAction = new cpactions.CodeBuildAction({
        actionName: 'CodeBuild',
        project,
        input: sourceOutput,
        outputs: [buildOutput],
    });
    pipeline.addStage({
        stageName: 'build',
        actions: [buildAction],
    });
    return { synthesizedApp: buildOutput, pipeline };
}
function hasPipelineAction(expectedAction) {
    return (props) => {
        for (const stage of props.Stages) {
            for (const action of stage.Actions) {
                if (assert_1.isSuperObject(action, expectedAction, [], true)) {
                    return true;
                }
            }
        }
        return false;
    };
}
module.exports = nodeunit.testCase({
    'rejects cross-environment deployment'(test) {
        fc.assert(fc.property(accountId, accountId, (pipelineAccount, stackAccount) => {
            fc.pre(pipelineAccount !== stackAccount);
            test.throws(() => {
                const app = new cdk.App();
                const stack = new cdk.Stack(app, 'Test', { env: { account: pipelineAccount } });
                const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
                const fakeAction = new FakeAction('Fake');
                pipeline.addStage({
                    stageName: 'FakeStage',
                    actions: [fakeAction],
                });
                const deployStage = pipeline.addStage({ stageName: 'DeployStage' });
                deployStage.addAction(new pipeline_deploy_stack_action_1.PipelineDeployStackAction({
                    changeSetName: 'ChangeSet',
                    input: fakeAction.outputArtifact,
                    stack: new cdk.Stack(app, 'DeployedStack', { env: { account: stackAccount } }),
                    adminPermissions: false,
                }));
            }, 'Cross-environment deployment is not supported');
        }));
        test.done();
    },
    'rejects createRunOrder >= executeRunOrder'(test) {
        fc.assert(fc.property(fc.integer(1, 999), fc.integer(1, 999), (createRunOrder, executeRunOrder) => {
            fc.pre(createRunOrder >= executeRunOrder);
            test.throws(() => {
                const app = new cdk.App();
                const stack = new cdk.Stack(app, 'Test');
                const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
                const fakeAction = new FakeAction('Fake');
                pipeline.addStage({
                    stageName: 'FakeStage',
                    actions: [fakeAction],
                });
                const deployStage = pipeline.addStage({ stageName: 'DeployStage' });
                deployStage.addAction(new pipeline_deploy_stack_action_1.PipelineDeployStackAction({
                    changeSetName: 'ChangeSet',
                    createChangeSetRunOrder: createRunOrder,
                    executeChangeSetRunOrder: executeRunOrder,
                    input: fakeAction.outputArtifact,
                    stack: new cdk.Stack(app, 'DeployedStack'),
                    adminPermissions: false,
                }));
            }, 'createChangeSetRunOrder must be < executeChangeSetRunOrder');
        }));
        test.done();
    },
    'users can supply CloudFormation capabilities'(test) {
        const pipelineStack = getTestStack();
        const stackWithNoCapability = new cdk.Stack(undefined, 'NoCapStack', { env: { account: '123456789012', region: 'us-east-1' } });
        const stackWithAnonymousCapability = new cdk.Stack(undefined, 'AnonymousIAM', { env: { account: '123456789012', region: 'us-east-1' } });
        const stackWithAutoExpandCapability = new cdk.Stack(undefined, 'AutoExpand', { env: { account: '123456789012', region: 'us-east-1' } });
        const stackWithAnonymousAndAutoExpandCapability = new cdk.Stack(undefined, 'AnonymousIAMAndAutoExpand', { env: { account: '123456789012', region: 'us-east-1' } });
        const selfUpdatingStack = createSelfUpdatingStack(pipelineStack);
        const pipeline = selfUpdatingStack.pipeline;
        const selfUpdateStage1 = pipeline.addStage({ stageName: 'SelfUpdate1' });
        const selfUpdateStage2 = pipeline.addStage({ stageName: 'SelfUpdate2' });
        const selfUpdateStage3 = pipeline.addStage({ stageName: 'SelfUpdate3' });
        const selfUpdateStage4 = pipeline.addStage({ stageName: 'SelfUpdate4' });
        const selfUpdateStage5 = pipeline.addStage({ stageName: 'SelfUpdate5' });
        selfUpdateStage1.addAction(new pipeline_deploy_stack_action_1.PipelineDeployStackAction({
            stack: pipelineStack,
            input: selfUpdatingStack.synthesizedApp,
            capabilities: [cfn.CloudFormationCapabilities.NAMED_IAM],
            adminPermissions: false,
        }));
        selfUpdateStage2.addAction(new pipeline_deploy_stack_action_1.PipelineDeployStackAction({
            stack: stackWithNoCapability,
            input: selfUpdatingStack.synthesizedApp,
            capabilities: [cfn.CloudFormationCapabilities.NONE],
            adminPermissions: false,
        }));
        selfUpdateStage3.addAction(new pipeline_deploy_stack_action_1.PipelineDeployStackAction({
            stack: stackWithAnonymousCapability,
            input: selfUpdatingStack.synthesizedApp,
            capabilities: [cfn.CloudFormationCapabilities.ANONYMOUS_IAM],
            adminPermissions: false,
        }));
        selfUpdateStage4.addAction(new pipeline_deploy_stack_action_1.PipelineDeployStackAction({
            stack: stackWithAutoExpandCapability,
            input: selfUpdatingStack.synthesizedApp,
            capabilities: [cfn.CloudFormationCapabilities.AUTO_EXPAND],
            adminPermissions: false,
        }));
        selfUpdateStage5.addAction(new pipeline_deploy_stack_action_1.PipelineDeployStackAction({
            stack: stackWithAnonymousAndAutoExpandCapability,
            input: selfUpdatingStack.synthesizedApp,
            capabilities: [cfn.CloudFormationCapabilities.ANONYMOUS_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND],
            adminPermissions: false,
        }));
        assert_1.expect(pipelineStack).to(assert_1.haveResource('AWS::CodePipeline::Pipeline', hasPipelineAction({
            Configuration: {
                StackName: 'TestStack',
                ActionMode: 'CHANGE_SET_REPLACE',
                Capabilities: 'CAPABILITY_NAMED_IAM',
            },
        })));
        assert_1.expect(pipelineStack).to(assert_1.haveResource('AWS::CodePipeline::Pipeline', hasPipelineAction({
            Configuration: {
                StackName: 'AnonymousIAM',
                ActionMode: 'CHANGE_SET_REPLACE',
                Capabilities: 'CAPABILITY_IAM',
            },
        })));
        assert_1.expect(pipelineStack).notTo(assert_1.haveResource('AWS::CodePipeline::Pipeline', hasPipelineAction({
            Configuration: {
                StackName: 'NoCapStack',
                ActionMode: 'CHANGE_SET_REPLACE',
                Capabilities: 'CAPABILITY_NAMED_IAM',
            },
        })));
        assert_1.expect(pipelineStack).notTo(assert_1.haveResource('AWS::CodePipeline::Pipeline', hasPipelineAction({
            Configuration: {
                StackName: 'NoCapStack',
                ActionMode: 'CHANGE_SET_REPLACE',
                Capabilities: 'CAPABILITY_IAM',
            },
        })));
        assert_1.expect(pipelineStack).to(assert_1.haveResource('AWS::CodePipeline::Pipeline', hasPipelineAction({
            Configuration: {
                StackName: 'NoCapStack',
                ActionMode: 'CHANGE_SET_REPLACE',
            },
        })));
        assert_1.expect(pipelineStack).to(assert_1.haveResource('AWS::CodePipeline::Pipeline', hasPipelineAction({
            Configuration: {
                StackName: 'AutoExpand',
                ActionMode: 'CHANGE_SET_REPLACE',
                Capabilities: 'CAPABILITY_AUTO_EXPAND',
            },
        })));
        assert_1.expect(pipelineStack).to(assert_1.haveResource('AWS::CodePipeline::Pipeline', hasPipelineAction({
            Configuration: {
                StackName: 'AnonymousIAMAndAutoExpand',
                ActionMode: 'CHANGE_SET_REPLACE',
                Capabilities: 'CAPABILITY_IAM,CAPABILITY_AUTO_EXPAND',
            },
        })));
        test.done();
    },
    'users can use admin permissions'(test) {
        const pipelineStack = getTestStack();
        const selfUpdatingStack = createSelfUpdatingStack(pipelineStack);
        const pipeline = selfUpdatingStack.pipeline;
        const selfUpdateStage = pipeline.addStage({ stageName: 'SelfUpdate' });
        selfUpdateStage.addAction(new pipeline_deploy_stack_action_1.PipelineDeployStackAction({
            stack: pipelineStack,
            input: selfUpdatingStack.synthesizedApp,
            adminPermissions: true,
        }));
        assert_1.expect(pipelineStack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::GetAtt': [
                                    'CodePipelineArtifactsBucketF1E925CF',
                                    'Arn',
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'CodePipelineArtifactsBucketF1E925CF',
                                                'Arn',
                                            ],
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                    {
                        Action: [
                            'kms:Decrypt',
                            'kms:DescribeKey',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'CodePipelineArtifactsBucketEncryptionKey85407CB4',
                                'Arn',
                            ],
                        },
                    },
                    {
                        Action: '*',
                        Effect: 'Allow',
                        Resource: '*',
                    },
                ],
            },
        }));
        assert_1.expect(pipelineStack).to(assert_1.haveResource('AWS::CodePipeline::Pipeline', hasPipelineAction({
            Configuration: {
                StackName: 'TestStack',
                ActionMode: 'CHANGE_SET_REPLACE',
                Capabilities: 'CAPABILITY_NAMED_IAM,CAPABILITY_AUTO_EXPAND',
            },
        })));
        test.done();
    },
    'users can supply a role for deploy action'(test) {
        const pipelineStack = getTestStack();
        const selfUpdatingStack = createSelfUpdatingStack(pipelineStack);
        const role = new iam.Role(pipelineStack, 'MyRole', {
            assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
        });
        const pipeline = selfUpdatingStack.pipeline;
        const selfUpdateStage = pipeline.addStage({ stageName: 'SelfUpdate' });
        const deployAction = new pipeline_deploy_stack_action_1.PipelineDeployStackAction({
            stack: pipelineStack,
            input: selfUpdatingStack.synthesizedApp,
            adminPermissions: false,
            role,
        });
        selfUpdateStage.addAction(deployAction);
        test.same(deployAction.deploymentRole, role);
        test.done();
    },
    'users can specify IAM permissions for the deploy action'(test) {
        // GIVEN //
        const pipelineStack = getTestStack();
        // the fake stack to deploy
        const emptyStack = getTestStack();
        const selfUpdatingStack = createSelfUpdatingStack(pipelineStack);
        const pipeline = selfUpdatingStack.pipeline;
        // WHEN //
        // this our app/service/infra to deploy
        const deployStage = pipeline.addStage({ stageName: 'Deploy' });
        const deployAction = new pipeline_deploy_stack_action_1.PipelineDeployStackAction({
            stack: emptyStack,
            input: selfUpdatingStack.synthesizedApp,
            adminPermissions: false,
        });
        deployStage.addAction(deployAction);
        // we might need to add permissions
        deployAction.addToDeploymentRolePolicy(new iam.PolicyStatement({
            actions: [
                'ec2:AuthorizeSecurityGroupEgress',
                'ec2:AuthorizeSecurityGroupIngress',
                'ec2:DeleteSecurityGroup',
                'ec2:DescribeSecurityGroups',
                'ec2:CreateSecurityGroup',
                'ec2:RevokeSecurityGroupEgress',
                'ec2:RevokeSecurityGroupIngress',
            ],
            resources: ['*'],
        }));
        // THEN //
        assert_1.expect(pipelineStack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::GetAtt': [
                                    'CodePipelineArtifactsBucketF1E925CF',
                                    'Arn',
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'CodePipelineArtifactsBucketF1E925CF',
                                                'Arn',
                                            ],
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                    {
                        Action: [
                            'kms:Decrypt',
                            'kms:DescribeKey',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'CodePipelineArtifactsBucketEncryptionKey85407CB4',
                                'Arn',
                            ],
                        },
                    },
                    {
                        Action: [
                            'ec2:AuthorizeSecurityGroupEgress',
                            'ec2:AuthorizeSecurityGroupIngress',
                            'ec2:DeleteSecurityGroup',
                            'ec2:DescribeSecurityGroups',
                            'ec2:CreateSecurityGroup',
                            'ec2:RevokeSecurityGroupEgress',
                            'ec2:RevokeSecurityGroupIngress',
                        ],
                        Effect: 'Allow',
                        Resource: '*',
                    },
                ],
            },
            Roles: [
                {
                    Ref: 'CodePipelineDeployChangeSetRoleF9F2B343',
                },
            ],
        }));
        test.done();
    },
    'rejects stacks with assets'(test) {
        fc.assert(fc.property(fc.integer(1, 5), (assetCount) => {
            const app = new cdk.App();
            const deployedStack = new cdk.Stack(app, 'DeployedStack');
            for (let i = 0; i < assetCount; i++) {
                deployedStack.node.addMetadata(cxschema.ArtifactMetadataEntryType.ASSET, {});
            }
            test.throws(() => {
                new pipeline_deploy_stack_action_1.PipelineDeployStackAction({
                    changeSetName: 'ChangeSet',
                    input: new codepipeline.Artifact(),
                    stack: deployedStack,
                    adminPermissions: false,
                });
            }, /Cannot deploy the stack DeployedStack because it references/);
        }));
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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