"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Job = exports.MetricType = exports.JobState = exports.WorkerType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const events = require("aws-cdk-lib/aws-events");
const iam = require("aws-cdk-lib/aws-iam");
const s3 = require("aws-cdk-lib/aws-s3");
const cdk = require("aws-cdk-lib");
const _1 = require(".");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
/**
 * The type of predefined worker that is allocated when a job runs.
 *
 * If you need to use a WorkerType that doesn't exist as a static member, you
 * can instantiate a `WorkerType` object, e.g: `WorkerType.of('other type')`.
 */
class WorkerType {
    constructor(name) {
        this.name = name;
    }
    /**
     * Custom worker type
     * @param workerType custom worker type
     */
    static of(workerType) {
        return new WorkerType(workerType);
    }
}
exports.WorkerType = WorkerType;
_a = JSII_RTTI_SYMBOL_1;
WorkerType[_a] = { fqn: "@aws-cdk/aws-glue-alpha.WorkerType", version: "2.28.1-alpha.0" };
/**
 * Each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2 executors per worker.
 */
WorkerType.STANDARD = new WorkerType('Standard');
/**
 * Each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk), and provides 1 executor per worker. Suitable for memory-intensive jobs.
 */
WorkerType.G_1X = new WorkerType('G.1X');
/**
 * Each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk), and provides 1 executor per worker. Suitable for memory-intensive jobs.
 */
WorkerType.G_2X = new WorkerType('G.2X');
/**
 * Job states emitted by Glue to CloudWatch Events.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types for more information.
 */
var JobState;
(function (JobState) {
    /**
     * State indicating job run succeeded
     */
    JobState["SUCCEEDED"] = "SUCCEEDED";
    /**
     * State indicating job run failed
     */
    JobState["FAILED"] = "FAILED";
    /**
     * State indicating job run timed out
     */
    JobState["TIMEOUT"] = "TIMEOUT";
    /**
     * State indicating job is starting
     */
    JobState["STARTING"] = "STARTING";
    /**
     * State indicating job is running
     */
    JobState["RUNNING"] = "RUNNING";
    /**
     * State indicating job is stopping
     */
    JobState["STOPPING"] = "STOPPING";
    /**
     * State indicating job stopped
     */
    JobState["STOPPED"] = "STOPPED";
})(JobState = exports.JobState || (exports.JobState = {}));
/**
 * The Glue CloudWatch metric type.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
 */
var MetricType;
(function (MetricType) {
    /**
     * A value at a point in time.
     */
    MetricType["GAUGE"] = "gauge";
    /**
     * An aggregate number.
     */
    MetricType["COUNT"] = "count";
})(MetricType = exports.MetricType || (exports.MetricType = {}));
class JobBase extends cdk.Resource {
    /**
     * Create a CloudWatch Event Rule for this Glue Job when it's in a given state
     *
     * @param id construct id
     * @param options event options. Note that some values are overridden if provided, these are
     *  - eventPattern.source = ['aws.glue']
     *  - eventPattern.detailType = ['Glue Job State Change', 'Glue Job Run Status']
     *  - eventPattern.detail.jobName = [this.jobName]
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.glue'],
            detailType: ['Glue Job State Change', 'Glue Job Run Status'],
            detail: {
                jobName: [this.jobName],
            },
        });
        return rule;
    }
    /**
     * Create a CloudWatch Event Rule for the transition into the input jobState.
     *
     * @param id construct id.
     * @param jobState the job state.
     * @param options optional event options.
     */
    onStateChange(id, jobState, options = {}) {
        const rule = this.onEvent(id, {
            description: `Rule triggered when Glue job ${this.jobName} is in ${jobState} state`,
            ...options,
        });
        rule.addEventPattern({
            detail: {
                state: [jobState],
            },
        });
        return rule;
    }
    /**
     * Create a CloudWatch Event Rule matching JobState.SUCCEEDED.
     *
     * @param id construct id.
     * @param options optional event options. default is {}.
     */
    onSuccess(id, options = {}) {
        return this.onStateChange(id, JobState.SUCCEEDED, options);
    }
    /**
     * Return a CloudWatch Event Rule matching FAILED state.
     *
     * @param id construct id.
     * @param options optional event options. default is {}.
     */
    onFailure(id, options = {}) {
        return this.onStateChange(id, JobState.FAILED, options);
    }
    /**
     * Return a CloudWatch Event Rule matching TIMEOUT state.
     *
     * @param id construct id.
     * @param options optional event options. default is {}.
     */
    onTimeout(id, options = {}) {
        return this.onStateChange(id, JobState.TIMEOUT, options);
    }
    /**
     * Create a CloudWatch metric.
     *
     * @param metricName name of the metric typically prefixed with `glue.driver.`, `glue.<executorId>.` or `glue.ALL.`.
     * @param type the metric type.
     * @param props metric options.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
     */
    metric(metricName, type, props) {
        return new cloudwatch.Metric({
            metricName,
            namespace: 'Glue',
            dimensionsMap: {
                JobName: this.jobName,
                JobRunId: 'ALL',
                Type: type,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * Return a CloudWatch Metric indicating job success.
     *
     * This metric is based on the Rule returned by no-args onSuccess() call.
     */
    metricSuccess(props) {
        return metricRule(this.metricJobStateRule('SuccessMetricRule', JobState.SUCCEEDED), props);
    }
    /**
     * Return a CloudWatch Metric indicating job failure.
     *
     * This metric is based on the Rule returned by no-args onFailure() call.
     */
    metricFailure(props) {
        return metricRule(this.metricJobStateRule('FailureMetricRule', JobState.FAILED), props);
    }
    /**
     * Return a CloudWatch Metric indicating job timeout.
     *
     * This metric is based on the Rule returned by no-args onTimeout() call.
     */
    metricTimeout(props) {
        return metricRule(this.metricJobStateRule('TimeoutMetricRule', JobState.TIMEOUT), props);
    }
    /**
     * Creates or retrieves a singleton event rule for the input job state for use with the metric JobState methods.
     *
     * @param id construct id.
     * @param jobState the job state.
     * @private
     */
    metricJobStateRule(id, jobState) {
        return this.node.tryFindChild(id) ?? this.onStateChange(id, jobState);
    }
}
/**
 * A Glue Job.
 */
class Job extends JobBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_JobProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Job);
            }
            throw error;
        }
        const executable = props.executable.bind();
        this.role = props.role ?? new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('glue.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole')],
        });
        this.grantPrincipal = this.role;
        const sparkUI = props.sparkUI?.enabled ? this.setupSparkUI(executable, this.role, props.sparkUI) : undefined;
        this.sparkUILoggingLocation = sparkUI?.location;
        const continuousLoggingArgs = props.continuousLogging?.enabled ? this.setupContinuousLogging(this.role, props.continuousLogging) : {};
        const profilingMetricsArgs = props.enableProfilingMetrics ? { '--enable-metrics': '' } : {};
        const defaultArguments = {
            ...this.executableArguments(executable),
            ...continuousLoggingArgs,
            ...profilingMetricsArgs,
            ...sparkUI?.args,
            ...this.checkNoReservedArgs(props.defaultArguments),
        };
        const jobResource = new aws_glue_1.CfnJob(this, 'Resource', {
            name: props.jobName,
            description: props.description,
            role: this.role.roleArn,
            command: {
                name: executable.type.name,
                scriptLocation: this.codeS3ObjectUrl(executable.script),
                pythonVersion: executable.pythonVersion,
            },
            glueVersion: executable.glueVersion.name,
            workerType: props.workerType?.name,
            numberOfWorkers: props.workerCount,
            maxCapacity: props.maxCapacity,
            maxRetries: props.maxRetries,
            executionProperty: props.maxConcurrentRuns ? { maxConcurrentRuns: props.maxConcurrentRuns } : undefined,
            notificationProperty: props.notifyDelayAfter ? { notifyDelayAfter: props.notifyDelayAfter.toMinutes() } : undefined,
            timeout: props.timeout?.toMinutes(),
            connections: props.connections ? { connections: props.connections.map((connection) => connection.connectionName) } : undefined,
            securityConfiguration: props.securityConfiguration?.securityConfigurationName,
            tags: props.tags,
            defaultArguments,
        });
        const resourceName = this.getResourceNameAttribute(jobResource.ref);
        this.jobArn = jobArn(this, resourceName);
        this.jobName = resourceName;
    }
    /**
     * Creates a Glue Job
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes
     */
    static fromJobAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_JobAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromJobAttributes);
            }
            throw error;
        }
        class Import extends JobBase {
            constructor() {
                super(...arguments);
                this.jobName = attrs.jobName;
                this.jobArn = jobArn(scope, attrs.jobName);
                this.grantPrincipal = attrs.role ?? new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Check no usage of reserved arguments.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
     */
    checkNoReservedArgs(defaultArguments) {
        if (defaultArguments) {
            const reservedArgs = new Set(['--conf', '--debug', '--mode', '--JOB_NAME']);
            Object.keys(defaultArguments).forEach((arg) => {
                if (reservedArgs.has(arg)) {
                    throw new Error(`The ${arg} argument is reserved by Glue. Don't set it`);
                }
            });
        }
        return defaultArguments;
    }
    executableArguments(config) {
        const args = {};
        args['--job-language'] = config.language;
        if (config.className) {
            args['--class'] = config.className;
        }
        if (config.extraJars && config.extraJars?.length > 0) {
            args['--extra-jars'] = config.extraJars.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraPythonFiles && config.extraPythonFiles.length > 0) {
            args['--extra-py-files'] = config.extraPythonFiles.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraFiles && config.extraFiles.length > 0) {
            args['--extra-files'] = config.extraFiles.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraJarsFirst) {
            args['--user-jars-first'] = 'true';
        }
        return args;
    }
    setupSparkUI(executable, role, props) {
        if (_1.JobType.PYTHON_SHELL === executable.type) {
            throw new Error('Spark UI is not available for JobType.PYTHON_SHELL jobs');
        }
        const bucket = props.bucket ?? new s3.Bucket(this, 'SparkUIBucket');
        bucket.grantReadWrite(role);
        const args = {
            '--enable-spark-ui': 'true',
            '--spark-event-logs-path': bucket.s3UrlForObject(props.prefix),
        };
        return {
            location: {
                prefix: props.prefix,
                bucket,
            },
            args,
        };
    }
    setupContinuousLogging(role, props) {
        const args = {
            '--enable-continuous-cloudwatch-log': 'true',
            '--enable-continuous-log-filter': (props.quiet ?? true).toString(),
        };
        if (props.logGroup) {
            args['--continuous-log-logGroup'] = props.logGroup.logGroupName;
            props.logGroup.grantWrite(role);
        }
        if (props.logStreamPrefix) {
            args['--continuous-log-logStreamPrefix'] = props.logStreamPrefix;
        }
        if (props.conversionPattern) {
            args['--continuous-log-conversionPattern'] = props.conversionPattern;
        }
        return args;
    }
    codeS3ObjectUrl(code) {
        const s3Location = code.bind(this, this.role).s3Location;
        return `s3://${s3Location.bucketName}/${s3Location.objectKey}`;
    }
}
exports.Job = Job;
_b = JSII_RTTI_SYMBOL_1;
Job[_b] = { fqn: "@aws-cdk/aws-glue-alpha.Job", version: "2.28.1-alpha.0" };
/**
 * Create a CloudWatch Metric that's based on Glue Job events
 * {@see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types}
 * The metric has namespace = 'AWS/Events', metricName = 'TriggeredRules' and RuleName = rule.ruleName dimension.
 *
 * @param rule for use in setting RuleName dimension value
 * @param props metric properties
 */
function metricRule(rule, props) {
    return new cloudwatch.Metric({
        namespace: 'AWS/Events',
        metricName: 'TriggeredRules',
        dimensionsMap: { RuleName: rule.ruleName },
        statistic: cloudwatch.Statistic.SUM,
        ...props,
    }).attachTo(rule);
}
/**
 * Returns the job arn
 * @param scope
 * @param jobName
 */
function jobArn(scope, jobName) {
    return cdk.Stack.of(scope).formatArn({
        service: 'glue',
        resource: 'job',
        resourceName: jobName,
    });
}
//# sourceMappingURL=data:application/json;base64,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