"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseAwsUrl = exports.AWS_URL_PATTERN = exports.normalizeComparisonString = exports.normalizeIdentifier = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
/**
 * Normalize string value to standard identifier form.
 *
 * Removes extraneous prefixes (Amazon, AWS), replaces all non-alphanumerics with underscore (_),
 * and converts to lower-case.
 * @param value - Value to normalize
 * @returns Returns normalized identifier string.
 */
function normalizeIdentifier(value) {
    value = value.replace(/^(Amazon|AWS)-?/i, ""); // remove prefix
    value = value.replace(/[^A-Za-z0-9]/g, "_"); // only alphanum + _
    value = value.replace(/_+/g, "_"); // de-dupe _
    return value.toLowerCase();
}
exports.normalizeIdentifier = normalizeIdentifier;
const ComparisonDictionary = {
    acl: "access_control_list",
    eni: "elastic_network_interface",
    eip: "elastic_ip_address",
};
/**
 * Normalizes string value for consistent comparison with variants between systems.
 * @param value - The value to normalize for comparison
 */
function normalizeComparisonString(value) {
    value = value.replace(/\.(png|svg)$/, "");
    value = normalizeIdentifier(value);
    Object.entries(ComparisonDictionary).forEach(([_key, _value]) => {
        _value = _value.replace(new RegExp(_key, "ig"), _value);
    });
    value = value.replace(/[_-]+(amazon|aws)[_-]+/gi, "");
    value = value.replace(/_(for|on|and|the|of|&)_/gi, "");
    value = value.replace(/v\d+/g, "");
    value = value.replace(/_/g, "");
    return value;
}
exports.normalizeComparisonString = normalizeComparisonString;
/**
 * RegEx pattern for https://aws.amazon.com based urls.
 */
exports.AWS_URL_PATTERN = /^https:\/\/(?<domain>(?:(?:docs)\.)?aws\.amazon\.com)\/(?<service>[^\/]+)(?:\/(?<resource>[^#]+))?\/?(?:#(?<dataType>.+))?$/i;
/** Parses https://aws.amazon.com based url into common definition. */
function parseAwsUrl(url) {
    url = url.replace(/\/(index\.html.*)?$/, "");
    const groups = url.match(exports.AWS_URL_PATTERN)
        ?.groups;
    if (groups) {
        const code = normalizeIdentifier(`${groups.service}_${groups.resource || ""}_${groups.dataType || ""}`);
        return {
            ...groups,
            code,
        };
    }
    throw new Error(`Failed to parse aws url: ${url}`);
}
exports.parseAwsUrl = parseAwsUrl;
//# sourceMappingURL=data:application/json;base64,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