"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DependencyLink = exports.ReferenceLink = exports.ChildLink = exports.Link = exports.Edge = exports.BaseEdge = void 0;
const Dot = require("ts-graphviz");
const subgraphs_1 = require("./subgraphs");
const theme_1 = require("../theme");
/**
 * BaseEdge class is the base class for defining a {@link Dot.Edge}.
 * @internal
 */
class BaseEdge extends Dot.Edge {
    /** Indicates if edge is considered **extraneous** */
    get isExtraneous() {
        return this.isSynthetic;
    }
    /** Indicates if edge is considered **verbose** */
    get isVerbose() {
        return this.isExtraneous || this.isClosedLoop;
    }
    /** Indicates if edge is a *closed loop*, meaning its *leaf* and *head* are the same entity */
    get isClosedLoop() {
        const [from, to] = this.targets;
        if (from.id === to.id)
            return true;
        if (from.id === this.attributes.get("lhead"))
            return true;
        if (to.id === this.attributes.get("ltail"))
            return true;
        return false;
    }
    /** @internal */
    constructor(from, to // TODO: support edge chain (need to handle subgraph case) // ...rest: EdgeTarget[]
    ) {
        const _attributes = {};
        let isCompound = false;
        let isSynthetic = false;
        // [compound](https://graphviz.org/docs/attrs/compound/) support - referencing container rather than node
        let _from;
        if (from instanceof subgraphs_1.Container) {
            isCompound = true;
            _from = from.nodes[0];
            if (_from == null) {
                isSynthetic = true;
                _from = {
                    id: `synthetic_${from.id}`,
                    style: "none",
                    shape: "none",
                    label: "",
                };
            }
            // https://graphviz.org/docs/attrs/ltail/
            _attributes.ltail = from.id;
        }
        else {
            _from = from;
        }
        let _to;
        if (to instanceof subgraphs_1.Container) {
            _to = to.nodes[0];
            isCompound = true;
            if (_to == null) {
                isSynthetic = true;
                _to = {
                    id: `synthetic_${to.id}`,
                    style: "none",
                    shape: "none",
                    label: "",
                };
            }
            // https://graphviz.org/docs/attrs/lhead/
            _attributes.lhead = to.id;
        }
        else {
            _to = to;
        }
        super([_from, _to], _attributes);
        this.from = from;
        this.to = to;
        this._extraneous = false;
        this.isCompound = isCompound;
        this.isSynthetic = isSynthetic;
        // Experimental: attempting to improve layout control of subgraphs (disabled by default)
        if (theme_1.GraphTheme.instance.rendering.unconstrainedCrossClusterEdges) {
            // Prevent cross-cluster edges from mangling the cluster layout
            if (this.isClosedLoop ||
                this.isCompound ||
                this.isSynthetic ||
                this.from.graphNode.findAncestor({
                    filter: (node) => node.isCluster,
                }) !==
                    this.to.graphNode.findAncestor({ filter: (node) => node.isCluster })) {
                this.attributes.set("constraint", false);
                if (!this.isClosedLoop &&
                    this.from instanceof Dot.Node &&
                    this.to instanceof Dot.Node) {
                    const group = "group_" +
                        this.from.graphNode.getNearestAncestor(this.to.graphNode).uuid;
                    this.from.attributes.set("group", group);
                    this.to.attributes.set("group", group);
                }
            }
        }
    }
}
exports.BaseEdge = BaseEdge;
/**
 * Edge class is the base class for {@link Graph.Edge} based {@link Dot.Edge} entities
 * @internal
 */
class Edge extends BaseEdge {
    /** @internal */
    constructor(edge, from, to
    // TODO: support edge chain (need to handle subgraph case)
    // Need to have specific use case before implementing this, but Dot.Edge supports chaining
    // ...rest: EdgeTarget[]
    ) {
        super(from, to);
        this.attributes.set("id", edge.uuid);
        this.graphEdge = edge;
        this._extraneous = edge.isExtraneous;
    }
}
exports.Edge = Edge;
/**
 * Link class defines a {@link Graph.Edge} defined by a {@link Graph.Node}
 * @internal
 */
class Link extends Edge {
    /** @internal */
    constructor(edge, from, to) {
        super(edge, from, to);
    }
}
exports.Link = Link;
/**
 * ChildLink class defines a {@link Dot.Edge} for a {@link Graph.Edge} that describes a parent-child {@link Graph.Node} relationship
 * @internal
 */
class ChildLink extends BaseEdge {
    /** @internal */
    constructor(from, to) {
        super(from, to);
        this.attributes.apply(theme_1.GraphTheme.instance.childLink);
    }
}
exports.ChildLink = ChildLink;
/**
 * ReferenceLink class defines a {@link Dot.Edge} for a {@link Graph.Reference} edge
 * @internal
 */
class ReferenceLink extends Link {
    /** @internal */
    constructor(edge, from, to) {
        super(edge, from, to);
        this.attributes.apply(theme_1.GraphTheme.instance.referenceLink);
    }
}
exports.ReferenceLink = ReferenceLink;
/**
 * DependencyLink class defines a {@link Dot.Edge} for a {@link Graph.Dependency} edge
 * @internal
 */
class DependencyLink extends Link {
    /** @internal */
    constructor(edge, from, to) {
        super(edge, from, to);
        this.attributes.apply(theme_1.GraphTheme.instance.dependencyLink);
    }
}
exports.DependencyLink = DependencyLink;
//# sourceMappingURL=data:application/json;base64,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