"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResourceNode = exports.ResourceNode = exports.CfnResourceNode = exports.ImageNode = exports.Node = void 0;
const startCase = require("lodash.startcase"); // eslint-disable-line @typescript-eslint/no-require-imports
const words = require("lodash.words"); // eslint-disable-line @typescript-eslint/no-require-imports
const Dot = require("ts-graphviz");
const wordWrap = require("word-wrap"); // eslint-disable-line @typescript-eslint/no-require-imports
const resource_images_1 = require("../../utils/resource-images");
const theme_1 = require("../theme");
/** Diagram label line height */
const LABEL_LINE_HEIGHT = 0.23;
/** Diagram label line max chars */
const LABEL_LINE_MAX_CHARS = 15;
/** Diagram label max number of lines */
const LABEL_MAX_LINES = 5;
/** Marshalls a label to contain length, output multi-line, etc for better rendering */
function marshallLabelForRendering(original) {
    let label = words(original).join(" ");
    label = wordWrap(label, {
        width: LABEL_LINE_MAX_CHARS,
        trim: true,
        indent: "",
    });
    const splitLabel = label.split("\n");
    const lines = splitLabel.slice(0, LABEL_MAX_LINES);
    // Ellipse last line if dropped lines
    if (splitLabel.length > lines.length) {
        lines[lines.length - 1] = lines[lines.length - 1] + "...";
    }
    label = lines
        .map((line) => {
        line = startCase(line).replace(/ /g, "");
        if (line.length > LABEL_LINE_MAX_CHARS) {
            return line.substring(0, LABEL_LINE_MAX_CHARS) + "...";
        }
        return line;
    })
        .join("\n");
    return { original, label, lines: lines.length };
}
/**
 * Node class defines a {@link Graph.Node} based diagram {@link Dot.Node}
 * @internal
 */
class Node extends Dot.Node {
    /** Get the label attribute for this node */
    get label() {
        return this.attributes.get("label");
    }
    /** @internal */
    constructor(node) {
        super(`node_${node.uuid}`);
        this.graphNode = node;
        this.attributes.set("label", marshallLabelForRendering(node.id).label);
        this.attributes.set("comment", `nodeType:${node.nodeType}` + (node.cfnType ? `(${node.cfnType})` : ""));
    }
}
exports.Node = Node;
/**
 * ImageNode class extends {@link Node} with support for rendering diagram images.
 * @internal
 */
class ImageNode extends Node {
    /** @internal */
    constructor(node, image) {
        super(node);
        // If image not defined, treat as regular node
        if (image) {
            this.attributes.apply(theme_1.GraphTheme.instance.imageNode);
            this.attributes.set("image", image);
            this.resize();
        }
    }
    /** Get `image` attribute */
    get image() {
        return this.attributes.get("image");
    }
    /** Resizes the node based on image and label dimensions */
    resize(baseHeight) {
        if (baseHeight == null) {
            baseHeight = (this.attributes.get("height") || 1);
        }
        const image = this.image;
        if (image) {
            const labelLines = this.label.split("\n").length;
            this.attributes.set("labelloc", "b");
            this.attributes.set("height", baseHeight + labelLines * LABEL_LINE_HEIGHT);
        }
        else {
            this.attributes.set("labelloc", "c");
            this.attributes.set("penwidth", 0.25);
            this.attributes.set("height", baseHeight);
        }
    }
}
exports.ImageNode = ImageNode;
/**
 * CfnResourceNode class defines a {@link Dot.Node} based on a {@link Graph.CfnResourceNode}
 * @internal
 */
class CfnResourceNode extends ImageNode {
    /** @internal */
    constructor(node) {
        super(node, (0, resource_images_1.resolveCfnResourceImage)(node));
        this.attributes.apply(theme_1.GraphTheme.instance.cfnResourceNode);
        this.resize(theme_1.GraphTheme.instance.cfnResourceNode.height === ""
            ? undefined
            : theme_1.GraphTheme.instance.cfnResourceNode.height);
        if (node.isImport) {
            this.attributes.apply({
                style: "filled,dotted",
                penwidth: 1,
                fontcolor: (theme_1.GraphTheme.instance.awsTheme?.text.tertiary ||
                    "#55555"),
                color: ((theme_1.GraphTheme.instance.awsTheme?.text.tertiary || "#55555") +
                    "33"), // 20%
                fillcolor: ((theme_1.GraphTheme.instance.awsTheme?.text.tertiary || "#55555") +
                    "1A"), // 10%
            });
        }
    }
}
exports.CfnResourceNode = CfnResourceNode;
/**
 * ResourceNode class defines a {@link Dot.Node} based on a {@link Graph.ResourceNode}
 * @internal
 */
class ResourceNode extends ImageNode {
    /** @internal */
    constructor(node) {
        const image = (0, resource_images_1.resolveResourceImage)(node);
        super(node, image);
        this.attributes.apply(theme_1.GraphTheme.instance.resourceNode);
        this.resize(theme_1.GraphTheme.instance.resourceNode.height === ""
            ? undefined
            : theme_1.GraphTheme.instance.resourceNode.height);
    }
}
exports.ResourceNode = ResourceNode;
/**
 * CustomResourceNode class defines a {@link Dot.Node} based on a {@link Graph.Node} for a *custom resource*
 * @internal
 */
class CustomResourceNode extends ImageNode {
    /** @internal */
    constructor(node) {
        super(node, (0, resource_images_1.resolveCustomResourceImage)(node));
    }
}
exports.CustomResourceNode = CustomResourceNode;
//# sourceMappingURL=data:application/json;base64,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