"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getBaseTheme = exports.GraphTheme = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_arch_1 = require("../../../../aws-arch");
const cloneDeep = require("lodash.clonedeep"); // eslint-disable-line @typescript-eslint/no-require-imports
const theme_1 = require("./theme");
const fonts_1 = require("../../fonts");
const DEFAULT_RENDERING = {
    resourceIconMin: theme_1.GraphThemeRenderingIconTarget.DATA,
    resourceIconMax: theme_1.GraphThemeRenderingIconTarget.CATEGORY,
    cfnResourceIconMin: theme_1.GraphThemeRenderingIconTarget.DATA,
    cfnResourceIconMax: theme_1.GraphThemeRenderingIconTarget.SERVICE,
};
/** GraphTheme class is the implementation of diagram graph based theme */
class GraphTheme {
    /** Initializes the theme based on {@link aws_arch.Theme} definition */
    static init(config) {
        if (config == null || typeof config === "string") {
            this._instance = new GraphTheme(generateGraphThemeFromAwsTheme(aws_arch_1.aws_arch.resolveTheme(config)));
        }
        else if (Object.keys(config).length <= 2) {
            const { theme, rendering } = config;
            this._instance = new GraphTheme(generateGraphThemeFromAwsTheme(aws_arch_1.aws_arch.resolveTheme(theme), rendering));
        }
        else {
            this._instance = new GraphTheme(config);
        }
        return this.instance;
    }
    /** Get the current singleton instance for the theme being utilized for the diagram */
    static get instance() {
        if (this._instance == null) {
            throw new Error("Must init GraphTheme before requesting instance");
        }
        return this._instance;
    }
    /** @internal */
    constructor(theme) {
        this.rendering = theme.rendering;
        this.graph = theme.graph;
        this.subgraph = theme.subgraph;
        this.cluster = theme.cluster;
        this.node = theme.node;
        this.edge = theme.edge;
        this.cloud = theme.cloud;
        this.stage = theme.stage;
        this.stack = theme.stack;
        this.nestedStack = theme.nestedStack;
        this.imageNode = theme.imageNode;
        this.cfnResourceNode = theme.cfnResourceNode;
        this.resourceNode = theme.resourceNode;
        this.childLink = theme.childLink;
        this.referenceLink = theme.referenceLink;
        this.dependencyLink = theme.dependencyLink;
        this.awsTheme = theme.awsTheme;
    }
}
exports.GraphTheme = GraphTheme;
/** Get the base theme */
function getBaseTheme(rendering) {
    return cloneDeep({
        rendering: {
            ...DEFAULT_RENDERING,
            ...rendering,
        },
        graph: GRAPH_ATTRIBUTES,
        subgraph: SUBGRAPH_ATTRIBUTES,
        cluster: CLUSTER_ATTRIBUTES,
        node: NODE_ATTRIBUTES,
        edge: EDGE_ATTRIBUTES,
        cloud: CLOUD_ATTRIBUTES,
        stage: STAGE_ATTRIBUTES,
        stack: STACK_ATTRIBUTES,
        nestedStack: NESTED_STACK_ATTRIBUTES,
        imageNode: IMAGE_NODE_ATTRIBUTES,
        cfnResourceNode: CFN_RESOURCE_NODE_ATTRIBUTES,
        resourceNode: RESOURCE_NODE_ATTRIBUTES,
        childLink: CHILD_LINK_ATTRIBUTES,
        referenceLink: REFERENCE_LINK_ATTRIBUTES,
        dependencyLink: DEPENDENCY_LINK_ATTRIBUTES,
    });
}
exports.getBaseTheme = getBaseTheme;
/** Generate {@link IGraphTheme} from {@link aws_arch.Theme} */
function generateGraphThemeFromAwsTheme(awsTheme, rendering) {
    const theme = getBaseTheme(rendering);
    Object.assign(theme, {
        awsTheme,
    });
    Object.assign(theme.graph, {
        bgcolor: awsTheme.backgrounds.base,
        fontcolor: awsTheme.text.default,
    });
    Object.assign(theme.node, {
        fontcolor: awsTheme.text.primary,
    });
    Object.assign(theme.edge, awsArrowToEdgeAtts(awsTheme.arrows.default));
    // containers
    Object.assign(theme.cloud, awsGroupToSubgraphAtts(awsTheme.groups.cloud));
    Object.assign(theme.stage, awsGroupToSubgraphAtts(awsTheme.groups.awsAccount, true));
    Object.assign(theme.stack, awsGroupToSubgraphAtts(awsTheme.groups.genericAlt, true));
    Object.assign(theme.nestedStack, awsGroupToSubgraphAtts(awsTheme.groups.genericAlt, true));
    // nodes
    Object.assign(theme.cfnResourceNode, {
        color: awsTheme.text.secondary,
        fontcolor: awsTheme.text.secondary,
    });
    Object.assign(theme.resourceNode, {
        color: awsTheme.text.primary,
        fontcolor: awsTheme.text.primary,
    });
    Object.assign(theme.childLink, awsArrowToEdgeAtts(awsTheme.arrows.child));
    Object.assign(theme.referenceLink, awsArrowToEdgeAtts(awsTheme.arrows.reference));
    Object.assign(theme.dependencyLink, awsArrowToEdgeAtts(awsTheme.arrows.dependency));
    return theme;
}
/** Base graph attributes */
const GRAPH_ATTRIBUTES = {
    ...fonts_1.GraphFonts.REGULAR,
    center: true,
    compound: true,
    concentrate: true,
    dpi: 300,
    fontcolor: "#222222",
    fontnames: "ps",
    fontsize: 14,
    forcelabels: true,
    labelloc: "tc",
    nodesep: 0.8,
    pad: "%2,%1",
    // @ts-ignore
    rankdir: "TB",
    ranksep: 0.75,
    ratio: "compress",
    remincross: true,
    size: "%1024,%1024!",
    splines: "ortho",
    stylesheet: fonts_1.FONT_STYLESHEET,
};
/** Base subgraph attributes */
const SUBGRAPH_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT_ITALIC,
    labelloc: "tc",
    fontsize: 12,
    style: "rounded,solid",
};
/** Base cluster attributes */
const CLUSTER_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT,
    labelloc: "tc",
    fontsize: 12,
    style: "rounded,dashed",
};
/** Base cloud container attributes */
const CLOUD_ATTRIBUTES = {
    ...fonts_1.GraphFonts.BOLD,
    style: "solid",
};
/** Base stage attributes */
const STAGE_ATTRIBUTES = {
    ...fonts_1.GraphFonts.BOLD_ITALIC,
    style: "dashed",
    margin: 6,
    rank: "same",
};
/** Base stack attributes */
const STACK_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT,
    style: "solid,bold,filled",
    fillcolor: "#5A6B861A", // 10%
    margin: 10,
};
/** Base nested stack attributes */
const NESTED_STACK_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT_ITALIC,
    style: "solid,filled",
    fillcolor: "#5A6B861A", // 10%
    margin: 6,
};
/** Base node attributes */
const NODE_ATTRIBUTES = {
    ...fonts_1.GraphFonts.REGULAR,
    shape: "box",
    style: "solid",
    fixedsize: false,
    width: 0.25,
    height: 0.25,
    labelloc: "c",
    imagescale: true,
    fontsize: 10,
    penwidth: 0,
};
/** Base image based node attributes */
const IMAGE_NODE_ATTRIBUTES = {
    shape: "box",
    style: "solid,rounded",
    fixedsize: true,
    width: 1,
    height: 1,
    labelloc: "b",
    imagescale: true,
    imagepos: "tc",
    fillcolor: "transparent",
};
/** Base cfn resource node attributes */
const CFN_RESOURCE_NODE_ATTRIBUTES = {
    width: 1,
    height: 1,
    fixedsize: true,
    imagescale: true,
    style: "solid,rounded",
    color: "#999999",
    fontcolor: "#999999",
};
/** Base resource node attributes */
const RESOURCE_NODE_ATTRIBUTES = {
    width: 1,
    height: 1,
    fixedsize: true,
    imagescale: true,
    color: "#666666",
    fontcolor: "#666666",
};
/** Base edge attributes */
const EDGE_ATTRIBUTES = {
    ...fonts_1.GraphFonts.LIGHT_ITALIC,
    dir: "both",
    color: "#545B64",
    penwidth: 0.75,
    arrowhead: "dot",
    arrowtail: "dot",
    arrowsize: 0.5,
    fontsize: 9,
    style: "solid",
};
/** Base child link attributes */
const CHILD_LINK_ATTRIBUTES = {
    penwidth: 1,
    arrowhead: "none",
    arrowtail: "normal",
    arrowsize: 1,
    style: "solid",
};
/** Base reference link attributes */
const REFERENCE_LINK_ATTRIBUTES = {
    penwidth: 1,
    arrowhead: "none",
    arrowtail: "normal",
    arrowsize: 0.75,
    style: "solid",
};
/** Base dependency link attributes */
const DEPENDENCY_LINK_ATTRIBUTES = {
    penwidth: 0.75,
    arrowhead: "dot",
    arrowtail: "odot",
    arrowsize: 0.75,
    style: "dotted",
};
/** Convert {@link aws_arch.GroupFormat} to {@link Dot.SubgraphAttributesObject} */
function awsGroupToSubgraphAtts(group, ignoreStyle = false) {
    const attributes = {};
    if (!ignoreStyle) {
        if (group.borderStyle === "none") {
            attributes.style = "solid";
            attributes.penwidth = 0;
        }
        else {
            attributes.style = group.borderStyle;
        }
    }
    attributes.color = group.color || "transparent";
    attributes.bgcolor = group.bgcolor || "transparent";
    attributes.fillcolor = group.bgcolor || "transparent";
    attributes.pencolor = (group.borderColor || "transparent");
    if (group.color && group.color !== "transparent") {
        attributes.fontcolor = group.color;
    }
    return attributes;
}
/** Convert {@link aws_arch.ArrowFormat} to {@link Dot.EdgeAttributesObject} */
function awsArrowToEdgeAtts(arrow) {
    return {
        color: arrow.color,
        fontcolor: arrow.color,
        arrowtail: arrow.tail,
        arrowhead: arrow.head,
        penwidth: arrow.width,
        style: arrow.style,
    };
}
//# sourceMappingURL=data:application/json;base64,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