"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkGraphDiagramPlugin = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const child_process_1 = require("child_process");
const path = require("path");
const cdk_graph_1 = require("../cdk-graph");
const chalk = require("chalk"); // eslint-disable-line @typescript-eslint/no-require-imports
const fs = require("fs-extra");
const adapter_1 = require("ts-graphviz/adapter");
const config_1 = require("./config");
const debug_1 = require("./internal/debug");
const diagram_1 = require("./internal/graphviz/diagram");
const svg_1 = require("./internal/utils/svg");
/**
 * CdkGraphDiagramPlugin is a {@link ICdkGraphPlugin CdkGraph Plugin} implementation for generating
 * diagram artifacts from the {@link CdkGraph} framework.
 */
class CdkGraphDiagramPlugin {
    /** Get standardized artifact id for diagram artifacts */
    static artifactId(name, format) {
        if (name === config_1.DEFAULT_DIAGRAM_NAME) {
            return `${this.ARTIFACT_NS}_${format.toUpperCase()}`;
        }
        return `${this.ARTIFACT_NS}_${name.toUpperCase()}_${format.toUpperCase()}`;
    }
    /** Get standardized artifact file name for diagram artifacts */
    static artifactFilename(name, format) {
        if (name === config_1.DEFAULT_DIAGRAM_NAME) {
            return `${this.ARTIFACT_NS.toLowerCase()}.${format}`;
        }
        return `${this.ARTIFACT_NS.toLowerCase()}.${name}.${format}`;
    }
    /** @inheritdoc */
    get id() {
        return CdkGraphDiagramPlugin.ID;
    }
    /** @inheritdoc */
    get version() {
        return CdkGraphDiagramPlugin.VERSION;
    }
    /** Get default dot artifact */
    get defaultDotArtifact() {
        try {
            return this.getDiagramArtifact(config_1.DEFAULT_DIAGRAM_NAME, config_1.DiagramFormat.DOT);
        }
        catch {
            return;
        }
    }
    /** Get default PNG artifact */
    get defaultPngArtifact() {
        try {
            return this.getDiagramArtifact(config_1.DEFAULT_DIAGRAM_NAME, config_1.DiagramFormat.PNG);
        }
        catch {
            return;
        }
    }
    /** Get diagram plugin config */
    get config() {
        if (this._config != null) {
            return this._config;
        }
        throw new Error("Plugin config has not been instantiated, ensure bind is called");
    }
    constructor(config) {
        /** @inheritdoc */
        this.dependencies = [];
        /** @inheritdoc */
        this.bind = (graph) => {
            this._graph = graph;
            const rc = (graph.config[CdkGraphDiagramPlugin.ID] ||
                {});
            let diagrams = [
                ...(rc.diagrams || []),
                ...(this._config?.diagrams || []),
            ];
            if (diagrams.length === 0) {
                diagrams = [config_1.DEFAULT_DIAGRAM];
            }
            this._config = {
                defaults: {
                    ...config_1.CONFIG_DEFAULTS,
                    ...rc.defaults,
                    ...this._config?.defaults,
                },
                diagrams,
            };
        };
        this.streamToBuffer = (stream) => {
            const chunks = [];
            return new Promise((resolve, reject) => {
                stream.on("data", (chunk) => chunks.push(Buffer.from(chunk)));
                stream.on("error", (err) => reject(err));
                stream.on("end", () => resolve(Buffer.concat(chunks)));
            });
        };
        this.isGraphvizInstalledGlobally = () => {
            try {
                (0, child_process_1.execSync)("dot -V", { stdio: "ignore" });
                return true;
            }
            catch (e) {
                console.warn(chalk.yellowBright("SVG and PNG are not supported as graphviz is not installed. Please install graphviz (https://graphviz.org/download/) globally and re-try."));
                return false;
            }
        };
        /** @inheritdoc */
        this.report = async (context) => {
            const pluginConfig = this.config;
            for (const diagramConfig of pluginConfig.diagrams) {
                const config = {
                    ...(diagramConfig.ignoreDefaults ? {} : pluginConfig.defaults),
                    ...diagramConfig,
                };
                let formats = Array.isArray(config.format)
                    ? config.format
                    : [config.format || config_1.DiagramFormat.PNG];
                if (!formats.length) {
                    throw new Error(`Diagram config specifies empty list of formats; must provide at least 1 or undefined to use default.`);
                }
                // each diagram is destructive, so we need a clone
                const store = context.store.clone();
                if (config.filterPlan) {
                    (0, cdk_graph_1.performGraphFilterPlan)(store, config.filterPlan);
                }
                const generatePng = formats.includes(config_1.DiagramFormat.PNG);
                const generateSvg = generatePng || formats.includes(config_1.DiagramFormat.SVG);
                const generateDot = generateSvg || formats.includes(config_1.DiagramFormat.DOT);
                debug_1.IS_DEBUG &&
                    context.writeArtifact(this, "filtered.graph." + config.name + "", "debug/filtered-graph/" + config.name + ".json", JSON.stringify(store.serialize(), null, 2), "DEBUG");
                const diagram = (0, diagram_1.buildDiagram)(store, {
                    title: config.title,
                    preset: config.filterPlan?.preset,
                    theme: config.theme,
                });
                const dot = diagram.toDot();
                if (generateDot) {
                    // Graphviz provider
                    context.writeArtifact(this, CdkGraphDiagramPlugin.artifactId(config.name, config_1.DiagramFormat.DOT), CdkGraphDiagramPlugin.artifactFilename(config.name, config_1.DiagramFormat.DOT), dot, `Diagram generated "dot" file for ${config.name} - "${config.title}"`);
                    if (generateSvg && this.isGraphvizInstalledGlobally()) {
                        const svgFile = path.join(context.outdir, CdkGraphDiagramPlugin.artifactFilename(config.name, config_1.DiagramFormat.SVG));
                        const svg = await this.streamToBuffer(await (0, adapter_1.toStream)(dot, { format: "svg" }));
                        const resolvedSvg = await (0, svg_1.resolveSvgAwsArchAssetImagesInline)(svg.toString());
                        fs.ensureDirSync(path.dirname(svgFile));
                        fs.writeFileSync(svgFile, resolvedSvg);
                        context.logArtifact(this, CdkGraphDiagramPlugin.artifactId(config.name, config_1.DiagramFormat.SVG), svgFile, `Diagram generated "svg" file for ${config.name} - "${config.title}"`);
                        if (generatePng && this.isGraphvizInstalledGlobally()) {
                            const pngFile = path.join(context.outdir, CdkGraphDiagramPlugin.artifactFilename(config.name, config_1.DiagramFormat.PNG));
                            try {
                                // SVG's don't render correctly in non-svg formats. Replace with png equivalent.
                                const pngifiedDot = dot.replace(/\.svg/g, ".png");
                                const png = await this.streamToBuffer(await (0, adapter_1.toStream)(pngifiedDot, {
                                    format: "png",
                                }));
                                fs.ensureDirSync(path.dirname(pngFile));
                                fs.writeFileSync(pngFile, png);
                                context.logArtifact(this, CdkGraphDiagramPlugin.artifactId(config.name, config_1.DiagramFormat.PNG), pngFile, `Diagram generated "png" file for ${config.name} - "${config.title}"`);
                            }
                            catch (error) {
                                console.error(error);
                                throw new Error(`Failed to generate PNG diagram for ${config.name} at "${pngFile}" - ${String(error)}`);
                            }
                        }
                    }
                }
                // NB: add drawio provider support here
            }
        };
        this._config = config;
    }
    /** Get diagram artifact for a given name and format */
    getDiagramArtifact(name, format) {
        return this._graph?.graphContext?.getArtifact(CdkGraphDiagramPlugin.artifactId(name, format));
    }
}
exports.CdkGraphDiagramPlugin = CdkGraphDiagramPlugin;
_a = JSII_RTTI_SYMBOL_1;
CdkGraphDiagramPlugin[_a] = { fqn: "@aws/pdk.cdk_graph_plugin_diagram.CdkGraphDiagramPlugin", version: "0.23.61" };
/** Namespace for artifacts of the diagram plugin */
CdkGraphDiagramPlugin.ARTIFACT_NS = "DIAGRAM";
/** Fixed id of the diagram plugin */
CdkGraphDiagramPlugin.ID = "diagram";
/** Current semantic version of the diagram plugin */
CdkGraphDiagramPlugin.VERSION = "0.0.0";
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGx1Z2luLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicGx1Z2luLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUE7c0NBQ3NDO0FBQ3RDLGlEQUF5QztBQUN6Qyw2QkFBNkI7QUFDN0IsOENBUXdCO0FBQ3hCLCtCQUFnQyxDQUFDLDREQUE0RDtBQUM3RiwrQkFBK0I7QUFDL0IsaURBQStDO0FBQy9DLHFDQU9rQjtBQUNsQiw0Q0FBNEM7QUFDNUMseURBQTJEO0FBQzNELDhDQUEwRTtBQUUxRTs7O0dBR0c7QUFDSCxNQUFhLHFCQUFxQjtJQVFoQyx5REFBeUQ7SUFDekQsTUFBTSxDQUFDLFVBQVUsQ0FBQyxJQUFZLEVBQUUsTUFBcUI7UUFDbkQsSUFBSSxJQUFJLEtBQUssNkJBQW9CLEVBQUUsQ0FBQztZQUNsQyxPQUFPLEdBQUcsSUFBSSxDQUFDLFdBQVcsSUFBSSxNQUFNLENBQUMsV0FBVyxFQUFFLEVBQUUsQ0FBQztRQUN2RCxDQUFDO1FBQ0QsT0FBTyxHQUFHLElBQUksQ0FBQyxXQUFXLElBQUksSUFBSSxDQUFDLFdBQVcsRUFBRSxJQUFJLE1BQU0sQ0FBQyxXQUFXLEVBQUUsRUFBRSxDQUFDO0lBQzdFLENBQUM7SUFDRCxnRUFBZ0U7SUFDaEUsTUFBTSxDQUFDLGdCQUFnQixDQUFDLElBQVksRUFBRSxNQUFxQjtRQUN6RCxJQUFJLElBQUksS0FBSyw2QkFBb0IsRUFBRSxDQUFDO1lBQ2xDLE9BQU8sR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLFdBQVcsRUFBRSxJQUFJLE1BQU0sRUFBRSxDQUFDO1FBQ3ZELENBQUM7UUFDRCxPQUFPLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxXQUFXLEVBQUUsSUFBSSxJQUFJLElBQUksTUFBTSxFQUFFLENBQUM7SUFDL0QsQ0FBQztJQUVELGtCQUFrQjtJQUNsQixJQUFJLEVBQUU7UUFDSixPQUFPLHFCQUFxQixDQUFDLEVBQUUsQ0FBQztJQUNsQyxDQUFDO0lBQ0Qsa0JBQWtCO0lBQ2xCLElBQUksT0FBTztRQUNULE9BQU8scUJBQXFCLENBQUMsT0FBTyxDQUFDO0lBQ3ZDLENBQUM7SUFXRCwrQkFBK0I7SUFDL0IsSUFBSSxrQkFBa0I7UUFDcEIsSUFBSSxDQUFDO1lBQ0gsT0FBTyxJQUFJLENBQUMsa0JBQWtCLENBQUMsNkJBQW9CLEVBQUUsc0JBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUMxRSxDQUFDO1FBQUMsTUFBTSxDQUFDO1lBQ1AsT0FBTztRQUNULENBQUM7SUFDSCxDQUFDO0lBRUQsK0JBQStCO0lBQy9CLElBQUksa0JBQWtCO1FBQ3BCLElBQUksQ0FBQztZQUNILE9BQU8sSUFBSSxDQUFDLGtCQUFrQixDQUFDLDZCQUFvQixFQUFFLHNCQUFhLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDMUUsQ0FBQztRQUFDLE1BQU0sQ0FBQztZQUNQLE9BQU87UUFDVCxDQUFDO0lBQ0gsQ0FBQztJQUVELGdDQUFnQztJQUNoQyxJQUFJLE1BQU07UUFDUixJQUFJLElBQUksQ0FBQyxPQUFPLElBQUksSUFBSSxFQUFFLENBQUM7WUFDekIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDO1FBQ3RCLENBQUM7UUFDRCxNQUFNLElBQUksS0FBSyxDQUNiLGdFQUFnRSxDQUNqRSxDQUFDO0lBQ0osQ0FBQztJQUVELFlBQVksTUFBc0I7UUFyQ2xDLGtCQUFrQjtRQUNULGlCQUFZLEdBQWMsRUFBRSxDQUFDO1FBa0R0QyxrQkFBa0I7UUFDbEIsU0FBSSxHQUE2QixDQUFDLEtBQWUsRUFBUSxFQUFFO1lBQ3pELElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDO1lBQ3BCLE1BQU0sRUFBRSxHQUFHLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxFQUFFLENBQUM7Z0JBQ2hELEVBQUUsQ0FBMkIsQ0FBQztZQUNoQyxJQUFJLFFBQVEsR0FBcUI7Z0JBQy9CLEdBQUcsQ0FBQyxFQUFFLENBQUMsUUFBUSxJQUFJLEVBQUUsQ0FBQztnQkFDdEIsR0FBRyxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsUUFBUSxJQUFJLEVBQUUsQ0FBQzthQUNsQyxDQUFDO1lBQ0YsSUFBSSxRQUFRLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRSxDQUFDO2dCQUMxQixRQUFRLEdBQUcsQ0FBQyx3QkFBZSxDQUFDLENBQUM7WUFDL0IsQ0FBQztZQUNELElBQUksQ0FBQyxPQUFPLEdBQUc7Z0JBQ2IsUUFBUSxFQUFFO29CQUNSLEdBQUcsd0JBQWU7b0JBQ2xCLEdBQUcsRUFBRSxDQUFDLFFBQVE7b0JBQ2QsR0FBRyxJQUFJLENBQUMsT0FBTyxFQUFFLFFBQVE7aUJBQzFCO2dCQUNELFFBQVE7YUFDVCxDQUFDO1FBQ0osQ0FBQyxDQUFDO1FBRU0sbUJBQWMsR0FBRyxDQUFDLE1BQTZCLEVBQW1CLEVBQUU7WUFDMUUsTUFBTSxNQUFNLEdBQWEsRUFBRSxDQUFDO1lBQzVCLE9BQU8sSUFBSSxPQUFPLENBQUMsQ0FBQyxPQUFPLEVBQUUsTUFBTSxFQUFFLEVBQUU7Z0JBQ3JDLE1BQU0sQ0FBQyxFQUFFLENBQUMsTUFBTSxFQUFFLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUM5RCxNQUFNLENBQUMsRUFBRSxDQUFDLE9BQU8sRUFBRSxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7Z0JBQ3pDLE1BQU0sQ0FBQyxFQUFFLENBQUMsS0FBSyxFQUFFLEdBQUcsRUFBRSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUN6RCxDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQztRQUVNLGdDQUEyQixHQUFHLEdBQUcsRUFBRTtZQUN6QyxJQUFJLENBQUM7Z0JBQ0gsSUFBQSx3QkFBUSxFQUFDLFFBQVEsRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDO2dCQUN4QyxPQUFPLElBQUksQ0FBQztZQUNkLENBQUM7WUFBQyxPQUFPLENBQUMsRUFBRSxDQUFDO2dCQUNYLE9BQU8sQ0FBQyxJQUFJLENBQ1YsS0FBSyxDQUFDLFlBQVksQ0FDaEIsMklBQTJJLENBQzVJLENBQ0YsQ0FBQztnQkFDRixPQUFPLEtBQUssQ0FBQztZQUNmLENBQUM7UUFDSCxDQUFDLENBQUM7UUFFRixrQkFBa0I7UUFDbEIsV0FBTSxHQUEwQixLQUFLLEVBQ25DLE9BQXdCLEVBQ1QsRUFBRTtZQUNqQixNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsTUFBaUMsQ0FBQztZQUU1RCxLQUFLLE1BQU0sYUFBYSxJQUFJLFlBQVksQ0FBQyxRQUFRLEVBQUUsQ0FBQztnQkFDbEQsTUFBTSxNQUFNLEdBQW1CO29CQUM3QixHQUFHLENBQUMsYUFBYSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDO29CQUM5RCxHQUFHLGFBQWE7aUJBQ2pCLENBQUM7Z0JBRUYsSUFBSSxPQUFPLEdBQW9CLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQztvQkFDekQsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNO29CQUNmLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLElBQUksc0JBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDekMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFNLEVBQUUsQ0FBQztvQkFDcEIsTUFBTSxJQUFJLEtBQUssQ0FDYixzR0FBc0csQ0FDdkcsQ0FBQztnQkFDSixDQUFDO2dCQUVELGtEQUFrRDtnQkFDbEQsTUFBTSxLQUFLLEdBQUcsT0FBTyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsQ0FBQztnQkFFcEMsSUFBSSxNQUFNLENBQUMsVUFBVSxFQUFFLENBQUM7b0JBQ3RCLElBQUEsa0NBQXNCLEVBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQztnQkFDbkQsQ0FBQztnQkFFRCxNQUFNLFdBQVcsR0FBRyxPQUFPLENBQUMsUUFBUSxDQUFDLHNCQUFhLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQ3hELE1BQU0sV0FBVyxHQUFHLFdBQVcsSUFBSSxPQUFPLENBQUMsUUFBUSxDQUFDLHNCQUFhLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQ3ZFLE1BQU0sV0FBVyxHQUFHLFdBQVcsSUFBSSxPQUFPLENBQUMsUUFBUSxDQUFDLHNCQUFhLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBRXZFLGdCQUFRO29CQUNOLE9BQU8sQ0FBQyxhQUFhLENBQ25CLElBQUksRUFDSixpQkFBaUIsR0FBRyxNQUFNLENBQUMsSUFBSSxHQUFHLEVBQUUsRUFDcEMsdUJBQXVCLEdBQUcsTUFBTSxDQUFDLElBQUksR0FBRyxPQUFPLEVBQy9DLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsRUFDMUMsT0FBTyxDQUNSLENBQUM7Z0JBRUosTUFBTSxPQUFPLEdBQUcsSUFBQSxzQkFBWSxFQUFDLEtBQUssRUFBRTtvQkFDbEMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxLQUFLO29CQUNuQixNQUFNLEVBQUUsTUFBTSxDQUFDLFVBQVUsRUFBRSxNQUFNO29CQUNqQyxLQUFLLEVBQUUsTUFBTSxDQUFDLEtBQUs7aUJBQ3BCLENBQUMsQ0FBQztnQkFFSCxNQUFNLEdBQUcsR0FBRyxPQUFPLENBQUMsS0FBSyxFQUFFLENBQUM7Z0JBRTVCLElBQUksV0FBVyxFQUFFLENBQUM7b0JBQ2hCLG9CQUFvQjtvQkFDcEIsT0FBTyxDQUFDLGFBQWEsQ0FDbkIsSUFBSSxFQUNKLHFCQUFxQixDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLHNCQUFhLENBQUMsR0FBRyxDQUFDLEVBQ2hFLHFCQUFxQixDQUFDLGdCQUFnQixDQUNwQyxNQUFNLENBQUMsSUFBSSxFQUNYLHNCQUFhLENBQUMsR0FBRyxDQUNsQixFQUNELEdBQUcsRUFDSCxvQ0FBb0MsTUFBTSxDQUFDLElBQUksT0FBTyxNQUFNLENBQUMsS0FBSyxHQUFHLENBQ3RFLENBQUM7b0JBRUYsSUFBSSxXQUFXLElBQUksSUFBSSxDQUFDLDJCQUEyQixFQUFFLEVBQUUsQ0FBQzt3QkFDdEQsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FDdkIsT0FBTyxDQUFDLE1BQU0sRUFDZCxxQkFBcUIsQ0FBQyxnQkFBZ0IsQ0FDcEMsTUFBTSxDQUFDLElBQUksRUFDWCxzQkFBYSxDQUFDLEdBQUcsQ0FDbEIsQ0FDRixDQUFDO3dCQUVGLE1BQU0sR0FBRyxHQUFHLE1BQU0sSUFBSSxDQUFDLGNBQWMsQ0FDbkMsTUFBTSxJQUFBLGtCQUFRLEVBQUMsR0FBRyxFQUFFLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQ3ZDLENBQUM7d0JBQ0YsTUFBTSxXQUFXLEdBQUcsTUFBTSxJQUFBLHdDQUFrQyxFQUMxRCxHQUFHLENBQUMsUUFBUSxFQUFFLENBQ2YsQ0FBQzt3QkFFRixFQUFFLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQzt3QkFDeEMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxPQUFPLEVBQUUsV0FBVyxDQUFDLENBQUM7d0JBRXZDLE9BQU8sQ0FBQyxXQUFXLENBQ2pCLElBQUksRUFDSixxQkFBcUIsQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxzQkFBYSxDQUFDLEdBQUcsQ0FBQyxFQUNoRSxPQUFPLEVBQ1Asb0NBQW9DLE1BQU0sQ0FBQyxJQUFJLE9BQU8sTUFBTSxDQUFDLEtBQUssR0FBRyxDQUN0RSxDQUFDO3dCQUVGLElBQUksV0FBVyxJQUFJLElBQUksQ0FBQywyQkFBMkIsRUFBRSxFQUFFLENBQUM7NEJBQ3RELE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQ3ZCLE9BQU8sQ0FBQyxNQUFNLEVBQ2QscUJBQXFCLENBQUMsZ0JBQWdCLENBQ3BDLE1BQU0sQ0FBQyxJQUFJLEVBQ1gsc0JBQWEsQ0FBQyxHQUFHLENBQ2xCLENBQ0YsQ0FBQzs0QkFFRixJQUFJLENBQUM7Z0NBQ0gsZ0ZBQWdGO2dDQUNoRixNQUFNLFdBQVcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRSxNQUFNLENBQUMsQ0FBQztnQ0FDbEQsTUFBTSxHQUFHLEdBQUcsTUFBTSxJQUFJLENBQUMsY0FBYyxDQUNuQyxNQUFNLElBQUEsa0JBQVEsRUFBQyxXQUFXLEVBQUU7b0NBQzFCLE1BQU0sRUFBRSxLQUFLO2lDQUNkLENBQUMsQ0FDSCxDQUFDO2dDQUVGLEVBQUUsQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO2dDQUN4QyxFQUFFLENBQUMsYUFBYSxDQUFDLE9BQU8sRUFBRSxHQUFHLENBQUMsQ0FBQztnQ0FFL0IsT0FBTyxDQUFDLFdBQVcsQ0FDakIsSUFBSSxFQUNKLHFCQUFxQixDQUFDLFVBQVUsQ0FDOUIsTUFBTSxDQUFDLElBQUksRUFDWCxzQkFBYSxDQUFDLEdBQUcsQ0FDbEIsRUFDRCxPQUFPLEVBQ1Asb0NBQW9DLE1BQU0sQ0FBQyxJQUFJLE9BQU8sTUFBTSxDQUFDLEtBQUssR0FBRyxDQUN0RSxDQUFDOzRCQUNKLENBQUM7NEJBQUMsT0FBTyxLQUFLLEVBQUUsQ0FBQztnQ0FDZixPQUFPLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO2dDQUNyQixNQUFNLElBQUksS0FBSyxDQUNiLHNDQUNFLE1BQU0sQ0FBQyxJQUNULFFBQVEsT0FBTyxPQUFPLE1BQU0sQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUN0QyxDQUFDOzRCQUNKLENBQUM7d0JBQ0gsQ0FBQztvQkFDSCxDQUFDO2dCQUNILENBQUM7Z0JBRUQsdUNBQXVDO1lBQ3pDLENBQUM7UUFDSCxDQUFDLENBQUM7UUE5TEEsSUFBSSxDQUFDLE9BQU8sR0FBRyxNQUFNLENBQUM7SUFDeEIsQ0FBQztJQUVELHVEQUF1RDtJQUN2RCxrQkFBa0IsQ0FDaEIsSUFBWSxFQUNaLE1BQXFCO1FBRXJCLE9BQU8sSUFBSSxDQUFDLE1BQU0sRUFBRSxZQUFZLEVBQUUsV0FBVyxDQUMzQyxxQkFBcUIsQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxDQUMvQyxDQUFDO0lBQ0osQ0FBQzs7QUFqRkgsc0RBcVFDOzs7QUFwUUMsb0RBQW9EO0FBQ3BDLGlDQUFXLEdBQUcsU0FBUyxBQUFaLENBQWE7QUFDeEMscUNBQXFDO0FBQ3JCLHdCQUFFLEdBQUcsU0FBUyxBQUFaLENBQWE7QUFDL0IscURBQXFEO0FBQ3JDLDZCQUFPLEdBQUcsT0FBTyxBQUFWLENBQVciLCJzb3VyY2VzQ29udGVudCI6WyIvKiEgQ29weXJpZ2h0IFtBbWF6b24uY29tXShodHRwOi8vYW1hem9uLmNvbS8pLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjAgKi9cbmltcG9ydCB7IGV4ZWNTeW5jIH0gZnJvbSBcImNoaWxkX3Byb2Nlc3NcIjtcbmltcG9ydCAqIGFzIHBhdGggZnJvbSBcInBhdGhcIjtcbmltcG9ydCB7XG4gIENka0dyYXBoLFxuICBDZGtHcmFwaEFydGlmYWN0LFxuICBDZGtHcmFwaENvbnRleHQsXG4gIElDZGtHcmFwaFBsdWdpbixcbiAgSUdyYXBoUGx1Z2luQmluZENhbGxiYWNrLFxuICBJR3JhcGhSZXBvcnRDYWxsYmFjayxcbiAgcGVyZm9ybUdyYXBoRmlsdGVyUGxhbixcbn0gZnJvbSBcIkBhd3MvY2RrLWdyYXBoXCI7XG5pbXBvcnQgY2hhbGsgPSByZXF1aXJlKFwiY2hhbGtcIik7IC8vIGVzbGludC1kaXNhYmxlLWxpbmUgQHR5cGVzY3JpcHQtZXNsaW50L25vLXJlcXVpcmUtaW1wb3J0c1xuaW1wb3J0ICogYXMgZnMgZnJvbSBcImZzLWV4dHJhXCI7XG5pbXBvcnQgeyB0b1N0cmVhbSB9IGZyb20gXCJ0cy1ncmFwaHZpei9hZGFwdGVyXCI7XG5pbXBvcnQge1xuICBDT05GSUdfREVGQVVMVFMsXG4gIERFRkFVTFRfRElBR1JBTSxcbiAgREVGQVVMVF9ESUFHUkFNX05BTUUsXG4gIERpYWdyYW1Gb3JtYXQsXG4gIElEaWFncmFtQ29uZmlnLFxuICBJUGx1Z2luQ29uZmlnLFxufSBmcm9tIFwiLi9jb25maWdcIjtcbmltcG9ydCB7IElTX0RFQlVHIH0gZnJvbSBcIi4vaW50ZXJuYWwvZGVidWdcIjtcbmltcG9ydCB7IGJ1aWxkRGlhZ3JhbSB9IGZyb20gXCIuL2ludGVybmFsL2dyYXBodml6L2RpYWdyYW1cIjtcbmltcG9ydCB7IHJlc29sdmVTdmdBd3NBcmNoQXNzZXRJbWFnZXNJbmxpbmUgfSBmcm9tIFwiLi9pbnRlcm5hbC91dGlscy9zdmdcIjtcblxuLyoqXG4gKiBDZGtHcmFwaERpYWdyYW1QbHVnaW4gaXMgYSB7QGxpbmsgSUNka0dyYXBoUGx1Z2luIENka0dyYXBoIFBsdWdpbn0gaW1wbGVtZW50YXRpb24gZm9yIGdlbmVyYXRpbmdcbiAqIGRpYWdyYW0gYXJ0aWZhY3RzIGZyb20gdGhlIHtAbGluayBDZGtHcmFwaH0gZnJhbWV3b3JrLlxuICovXG5leHBvcnQgY2xhc3MgQ2RrR3JhcGhEaWFncmFtUGx1Z2luIGltcGxlbWVudHMgSUNka0dyYXBoUGx1Z2luIHtcbiAgLyoqIE5hbWVzcGFjZSBmb3IgYXJ0aWZhY3RzIG9mIHRoZSBkaWFncmFtIHBsdWdpbiAqL1xuICBzdGF0aWMgcmVhZG9ubHkgQVJUSUZBQ1RfTlMgPSBcIkRJQUdSQU1cIjtcbiAgLyoqIEZpeGVkIGlkIG9mIHRoZSBkaWFncmFtIHBsdWdpbiAqL1xuICBzdGF0aWMgcmVhZG9ubHkgSUQgPSBcImRpYWdyYW1cIjtcbiAgLyoqIEN1cnJlbnQgc2VtYW50aWMgdmVyc2lvbiBvZiB0aGUgZGlhZ3JhbSBwbHVnaW4gKi9cbiAgc3RhdGljIHJlYWRvbmx5IFZFUlNJT04gPSBcIjAuMC4wXCI7XG5cbiAgLyoqIEdldCBzdGFuZGFyZGl6ZWQgYXJ0aWZhY3QgaWQgZm9yIGRpYWdyYW0gYXJ0aWZhY3RzICovXG4gIHN0YXRpYyBhcnRpZmFjdElkKG5hbWU6IHN0cmluZywgZm9ybWF0OiBEaWFncmFtRm9ybWF0KTogc3RyaW5nIHtcbiAgICBpZiAobmFtZSA9PT0gREVGQVVMVF9ESUFHUkFNX05BTUUpIHtcbiAgICAgIHJldHVybiBgJHt0aGlzLkFSVElGQUNUX05TfV8ke2Zvcm1hdC50b1VwcGVyQ2FzZSgpfWA7XG4gICAgfVxuICAgIHJldHVybiBgJHt0aGlzLkFSVElGQUNUX05TfV8ke25hbWUudG9VcHBlckNhc2UoKX1fJHtmb3JtYXQudG9VcHBlckNhc2UoKX1gO1xuICB9XG4gIC8qKiBHZXQgc3RhbmRhcmRpemVkIGFydGlmYWN0IGZpbGUgbmFtZSBmb3IgZGlhZ3JhbSBhcnRpZmFjdHMgKi9cbiAgc3RhdGljIGFydGlmYWN0RmlsZW5hbWUobmFtZTogc3RyaW5nLCBmb3JtYXQ6IERpYWdyYW1Gb3JtYXQpOiBzdHJpbmcge1xuICAgIGlmIChuYW1lID09PSBERUZBVUxUX0RJQUdSQU1fTkFNRSkge1xuICAgICAgcmV0dXJuIGAke3RoaXMuQVJUSUZBQ1RfTlMudG9Mb3dlckNhc2UoKX0uJHtmb3JtYXR9YDtcbiAgICB9XG4gICAgcmV0dXJuIGAke3RoaXMuQVJUSUZBQ1RfTlMudG9Mb3dlckNhc2UoKX0uJHtuYW1lfS4ke2Zvcm1hdH1gO1xuICB9XG5cbiAgLyoqIEBpbmhlcml0ZG9jICovXG4gIGdldCBpZCgpOiBzdHJpbmcge1xuICAgIHJldHVybiBDZGtHcmFwaERpYWdyYW1QbHVnaW4uSUQ7XG4gIH1cbiAgLyoqIEBpbmhlcml0ZG9jICovXG4gIGdldCB2ZXJzaW9uKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIENka0dyYXBoRGlhZ3JhbVBsdWdpbi5WRVJTSU9OO1xuICB9XG5cbiAgLyoqIEBpbmhlcml0ZG9jICovXG4gIHJlYWRvbmx5IGRlcGVuZGVuY2llcz86IHN0cmluZ1tdID0gW107XG5cbiAgLyoqIEBpbnRlcm5hbCAqL1xuICBwcml2YXRlIF9ncmFwaD86IENka0dyYXBoO1xuXG4gIC8qKiBAaW50ZXJuYWwgKi9cbiAgcHJpdmF0ZSBfY29uZmlnPzogSVBsdWdpbkNvbmZpZztcblxuICAvKiogR2V0IGRlZmF1bHQgZG90IGFydGlmYWN0ICovXG4gIGdldCBkZWZhdWx0RG90QXJ0aWZhY3QoKTogQ2RrR3JhcGhBcnRpZmFjdCB8IHVuZGVmaW5lZCB7XG4gICAgdHJ5IHtcbiAgICAgIHJldHVybiB0aGlzLmdldERpYWdyYW1BcnRpZmFjdChERUZBVUxUX0RJQUdSQU1fTkFNRSwgRGlhZ3JhbUZvcm1hdC5ET1QpO1xuICAgIH0gY2F0Y2gge1xuICAgICAgcmV0dXJuO1xuICAgIH1cbiAgfVxuXG4gIC8qKiBHZXQgZGVmYXVsdCBQTkcgYXJ0aWZhY3QgKi9cbiAgZ2V0IGRlZmF1bHRQbmdBcnRpZmFjdCgpOiBDZGtHcmFwaEFydGlmYWN0IHwgdW5kZWZpbmVkIHtcbiAgICB0cnkge1xuICAgICAgcmV0dXJuIHRoaXMuZ2V0RGlhZ3JhbUFydGlmYWN0KERFRkFVTFRfRElBR1JBTV9OQU1FLCBEaWFncmFtRm9ybWF0LlBORyk7XG4gICAgfSBjYXRjaCB7XG4gICAgICByZXR1cm47XG4gICAgfVxuICB9XG5cbiAgLyoqIEdldCBkaWFncmFtIHBsdWdpbiBjb25maWcgKi9cbiAgZ2V0IGNvbmZpZygpOiBJUGx1Z2luQ29uZmlnIHtcbiAgICBpZiAodGhpcy5fY29uZmlnICE9IG51bGwpIHtcbiAgICAgIHJldHVybiB0aGlzLl9jb25maWc7XG4gICAgfVxuICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgIFwiUGx1Z2luIGNvbmZpZyBoYXMgbm90IGJlZW4gaW5zdGFudGlhdGVkLCBlbnN1cmUgYmluZCBpcyBjYWxsZWRcIlxuICAgICk7XG4gIH1cblxuICBjb25zdHJ1Y3Rvcihjb25maWc/OiBJUGx1Z2luQ29uZmlnKSB7XG4gICAgdGhpcy5fY29uZmlnID0gY29uZmlnO1xuICB9XG5cbiAgLyoqIEdldCBkaWFncmFtIGFydGlmYWN0IGZvciBhIGdpdmVuIG5hbWUgYW5kIGZvcm1hdCAqL1xuICBnZXREaWFncmFtQXJ0aWZhY3QoXG4gICAgbmFtZTogc3RyaW5nLFxuICAgIGZvcm1hdDogRGlhZ3JhbUZvcm1hdFxuICApOiBDZGtHcmFwaEFydGlmYWN0IHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy5fZ3JhcGg/LmdyYXBoQ29udGV4dD8uZ2V0QXJ0aWZhY3QoXG4gICAgICBDZGtHcmFwaERpYWdyYW1QbHVnaW4uYXJ0aWZhY3RJZChuYW1lLCBmb3JtYXQpXG4gICAgKTtcbiAgfVxuXG4gIC8qKiBAaW5oZXJpdGRvYyAqL1xuICBiaW5kOiBJR3JhcGhQbHVnaW5CaW5kQ2FsbGJhY2sgPSAoZ3JhcGg6IENka0dyYXBoKTogdm9pZCA9PiB7XG4gICAgdGhpcy5fZ3JhcGggPSBncmFwaDtcbiAgICBjb25zdCByYyA9IChncmFwaC5jb25maWdbQ2RrR3JhcGhEaWFncmFtUGx1Z2luLklEXSB8fFxuICAgICAge30pIGFzIFBhcnRpYWw8SVBsdWdpbkNvbmZpZz47XG4gICAgbGV0IGRpYWdyYW1zOiBJRGlhZ3JhbUNvbmZpZ1tdID0gW1xuICAgICAgLi4uKHJjLmRpYWdyYW1zIHx8IFtdKSxcbiAgICAgIC4uLih0aGlzLl9jb25maWc/LmRpYWdyYW1zIHx8IFtdKSxcbiAgICBdO1xuICAgIGlmIChkaWFncmFtcy5sZW5ndGggPT09IDApIHtcbiAgICAgIGRpYWdyYW1zID0gW0RFRkFVTFRfRElBR1JBTV07XG4gICAgfVxuICAgIHRoaXMuX2NvbmZpZyA9IHtcbiAgICAgIGRlZmF1bHRzOiB7XG4gICAgICAgIC4uLkNPTkZJR19ERUZBVUxUUyxcbiAgICAgICAgLi4ucmMuZGVmYXVsdHMsXG4gICAgICAgIC4uLnRoaXMuX2NvbmZpZz8uZGVmYXVsdHMsXG4gICAgICB9LFxuICAgICAgZGlhZ3JhbXMsXG4gICAgfTtcbiAgfTtcblxuICBwcml2YXRlIHN0cmVhbVRvQnVmZmVyID0gKHN0cmVhbTogTm9kZUpTLlJlYWRhYmxlU3RyZWFtKTogUHJvbWlzZTxCdWZmZXI+ID0+IHtcbiAgICBjb25zdCBjaHVua3M6IEJ1ZmZlcltdID0gW107XG4gICAgcmV0dXJuIG5ldyBQcm9taXNlKChyZXNvbHZlLCByZWplY3QpID0+IHtcbiAgICAgIHN0cmVhbS5vbihcImRhdGFcIiwgKGNodW5rKSA9PiBjaHVua3MucHVzaChCdWZmZXIuZnJvbShjaHVuaykpKTtcbiAgICAgIHN0cmVhbS5vbihcImVycm9yXCIsIChlcnIpID0+IHJlamVjdChlcnIpKTtcbiAgICAgIHN0cmVhbS5vbihcImVuZFwiLCAoKSA9PiByZXNvbHZlKEJ1ZmZlci5jb25jYXQoY2h1bmtzKSkpO1xuICAgIH0pO1xuICB9O1xuXG4gIHByaXZhdGUgaXNHcmFwaHZpekluc3RhbGxlZEdsb2JhbGx5ID0gKCkgPT4ge1xuICAgIHRyeSB7XG4gICAgICBleGVjU3luYyhcImRvdCAtVlwiLCB7IHN0ZGlvOiBcImlnbm9yZVwiIH0pO1xuICAgICAgcmV0dXJuIHRydWU7XG4gICAgfSBjYXRjaCAoZSkge1xuICAgICAgY29uc29sZS53YXJuKFxuICAgICAgICBjaGFsay55ZWxsb3dCcmlnaHQoXG4gICAgICAgICAgXCJTVkcgYW5kIFBORyBhcmUgbm90IHN1cHBvcnRlZCBhcyBncmFwaHZpeiBpcyBub3QgaW5zdGFsbGVkLiBQbGVhc2UgaW5zdGFsbCBncmFwaHZpeiAoaHR0cHM6Ly9ncmFwaHZpei5vcmcvZG93bmxvYWQvKSBnbG9iYWxseSBhbmQgcmUtdHJ5LlwiXG4gICAgICAgIClcbiAgICAgICk7XG4gICAgICByZXR1cm4gZmFsc2U7XG4gICAgfVxuICB9O1xuXG4gIC8qKiBAaW5oZXJpdGRvYyAqL1xuICByZXBvcnQ/OiBJR3JhcGhSZXBvcnRDYWxsYmFjayA9IGFzeW5jIChcbiAgICBjb250ZXh0OiBDZGtHcmFwaENvbnRleHRcbiAgKTogUHJvbWlzZTx2b2lkPiA9PiB7XG4gICAgY29uc3QgcGx1Z2luQ29uZmlnID0gdGhpcy5jb25maWcgYXMgUmVxdWlyZWQ8SVBsdWdpbkNvbmZpZz47XG5cbiAgICBmb3IgKGNvbnN0IGRpYWdyYW1Db25maWcgb2YgcGx1Z2luQ29uZmlnLmRpYWdyYW1zKSB7XG4gICAgICBjb25zdCBjb25maWc6IElEaWFncmFtQ29uZmlnID0ge1xuICAgICAgICAuLi4oZGlhZ3JhbUNvbmZpZy5pZ25vcmVEZWZhdWx0cyA/IHt9IDogcGx1Z2luQ29uZmlnLmRlZmF1bHRzKSxcbiAgICAgICAgLi4uZGlhZ3JhbUNvbmZpZyxcbiAgICAgIH07XG5cbiAgICAgIGxldCBmb3JtYXRzOiBEaWFncmFtRm9ybWF0W10gPSBBcnJheS5pc0FycmF5KGNvbmZpZy5mb3JtYXQpXG4gICAgICAgID8gY29uZmlnLmZvcm1hdFxuICAgICAgICA6IFtjb25maWcuZm9ybWF0IHx8IERpYWdyYW1Gb3JtYXQuUE5HXTtcbiAgICAgIGlmICghZm9ybWF0cy5sZW5ndGgpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgIGBEaWFncmFtIGNvbmZpZyBzcGVjaWZpZXMgZW1wdHkgbGlzdCBvZiBmb3JtYXRzOyBtdXN0IHByb3ZpZGUgYXQgbGVhc3QgMSBvciB1bmRlZmluZWQgdG8gdXNlIGRlZmF1bHQuYFxuICAgICAgICApO1xuICAgICAgfVxuXG4gICAgICAvLyBlYWNoIGRpYWdyYW0gaXMgZGVzdHJ1Y3RpdmUsIHNvIHdlIG5lZWQgYSBjbG9uZVxuICAgICAgY29uc3Qgc3RvcmUgPSBjb250ZXh0LnN0b3JlLmNsb25lKCk7XG5cbiAgICAgIGlmIChjb25maWcuZmlsdGVyUGxhbikge1xuICAgICAgICBwZXJmb3JtR3JhcGhGaWx0ZXJQbGFuKHN0b3JlLCBjb25maWcuZmlsdGVyUGxhbik7XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IGdlbmVyYXRlUG5nID0gZm9ybWF0cy5pbmNsdWRlcyhEaWFncmFtRm9ybWF0LlBORyk7XG4gICAgICBjb25zdCBnZW5lcmF0ZVN2ZyA9IGdlbmVyYXRlUG5nIHx8IGZvcm1hdHMuaW5jbHVkZXMoRGlhZ3JhbUZvcm1hdC5TVkcpO1xuICAgICAgY29uc3QgZ2VuZXJhdGVEb3QgPSBnZW5lcmF0ZVN2ZyB8fCBmb3JtYXRzLmluY2x1ZGVzKERpYWdyYW1Gb3JtYXQuRE9UKTtcblxuICAgICAgSVNfREVCVUcgJiZcbiAgICAgICAgY29udGV4dC53cml0ZUFydGlmYWN0KFxuICAgICAgICAgIHRoaXMsXG4gICAgICAgICAgXCJmaWx0ZXJlZC5ncmFwaC5cIiArIGNvbmZpZy5uYW1lICsgXCJcIixcbiAgICAgICAgICBcImRlYnVnL2ZpbHRlcmVkLWdyYXBoL1wiICsgY29uZmlnLm5hbWUgKyBcIi5qc29uXCIsXG4gICAgICAgICAgSlNPTi5zdHJpbmdpZnkoc3RvcmUuc2VyaWFsaXplKCksIG51bGwsIDIpLFxuICAgICAgICAgIFwiREVCVUdcIlxuICAgICAgICApO1xuXG4gICAgICBjb25zdCBkaWFncmFtID0gYnVpbGREaWFncmFtKHN0b3JlLCB7XG4gICAgICAgIHRpdGxlOiBjb25maWcudGl0bGUsXG4gICAgICAgIHByZXNldDogY29uZmlnLmZpbHRlclBsYW4/LnByZXNldCxcbiAgICAgICAgdGhlbWU6IGNvbmZpZy50aGVtZSxcbiAgICAgIH0pO1xuXG4gICAgICBjb25zdCBkb3QgPSBkaWFncmFtLnRvRG90KCk7XG5cbiAgICAgIGlmIChnZW5lcmF0ZURvdCkge1xuICAgICAgICAvLyBHcmFwaHZpeiBwcm92aWRlclxuICAgICAgICBjb250ZXh0LndyaXRlQXJ0aWZhY3QoXG4gICAgICAgICAgdGhpcyxcbiAgICAgICAgICBDZGtHcmFwaERpYWdyYW1QbHVnaW4uYXJ0aWZhY3RJZChjb25maWcubmFtZSwgRGlhZ3JhbUZvcm1hdC5ET1QpLFxuICAgICAgICAgIENka0dyYXBoRGlhZ3JhbVBsdWdpbi5hcnRpZmFjdEZpbGVuYW1lKFxuICAgICAgICAgICAgY29uZmlnLm5hbWUsXG4gICAgICAgICAgICBEaWFncmFtRm9ybWF0LkRPVFxuICAgICAgICAgICksXG4gICAgICAgICAgZG90LFxuICAgICAgICAgIGBEaWFncmFtIGdlbmVyYXRlZCBcImRvdFwiIGZpbGUgZm9yICR7Y29uZmlnLm5hbWV9IC0gXCIke2NvbmZpZy50aXRsZX1cImBcbiAgICAgICAgKTtcblxuICAgICAgICBpZiAoZ2VuZXJhdGVTdmcgJiYgdGhpcy5pc0dyYXBodml6SW5zdGFsbGVkR2xvYmFsbHkoKSkge1xuICAgICAgICAgIGNvbnN0IHN2Z0ZpbGUgPSBwYXRoLmpvaW4oXG4gICAgICAgICAgICBjb250ZXh0Lm91dGRpcixcbiAgICAgICAgICAgIENka0dyYXBoRGlhZ3JhbVBsdWdpbi5hcnRpZmFjdEZpbGVuYW1lKFxuICAgICAgICAgICAgICBjb25maWcubmFtZSxcbiAgICAgICAgICAgICAgRGlhZ3JhbUZvcm1hdC5TVkdcbiAgICAgICAgICAgIClcbiAgICAgICAgICApO1xuXG4gICAgICAgICAgY29uc3Qgc3ZnID0gYXdhaXQgdGhpcy5zdHJlYW1Ub0J1ZmZlcihcbiAgICAgICAgICAgIGF3YWl0IHRvU3RyZWFtKGRvdCwgeyBmb3JtYXQ6IFwic3ZnXCIgfSlcbiAgICAgICAgICApO1xuICAgICAgICAgIGNvbnN0IHJlc29sdmVkU3ZnID0gYXdhaXQgcmVzb2x2ZVN2Z0F3c0FyY2hBc3NldEltYWdlc0lubGluZShcbiAgICAgICAgICAgIHN2Zy50b1N0cmluZygpXG4gICAgICAgICAgKTtcblxuICAgICAgICAgIGZzLmVuc3VyZURpclN5bmMocGF0aC5kaXJuYW1lKHN2Z0ZpbGUpKTtcbiAgICAgICAgICBmcy53cml0ZUZpbGVTeW5jKHN2Z0ZpbGUsIHJlc29sdmVkU3ZnKTtcblxuICAgICAgICAgIGNvbnRleHQubG9nQXJ0aWZhY3QoXG4gICAgICAgICAgICB0aGlzLFxuICAgICAgICAgICAgQ2RrR3JhcGhEaWFncmFtUGx1Z2luLmFydGlmYWN0SWQoY29uZmlnLm5hbWUsIERpYWdyYW1Gb3JtYXQuU1ZHKSxcbiAgICAgICAgICAgIHN2Z0ZpbGUsXG4gICAgICAgICAgICBgRGlhZ3JhbSBnZW5lcmF0ZWQgXCJzdmdcIiBmaWxlIGZvciAke2NvbmZpZy5uYW1lfSAtIFwiJHtjb25maWcudGl0bGV9XCJgXG4gICAgICAgICAgKTtcblxuICAgICAgICAgIGlmIChnZW5lcmF0ZVBuZyAmJiB0aGlzLmlzR3JhcGh2aXpJbnN0YWxsZWRHbG9iYWxseSgpKSB7XG4gICAgICAgICAgICBjb25zdCBwbmdGaWxlID0gcGF0aC5qb2luKFxuICAgICAgICAgICAgICBjb250ZXh0Lm91dGRpcixcbiAgICAgICAgICAgICAgQ2RrR3JhcGhEaWFncmFtUGx1Z2luLmFydGlmYWN0RmlsZW5hbWUoXG4gICAgICAgICAgICAgICAgY29uZmlnLm5hbWUsXG4gICAgICAgICAgICAgICAgRGlhZ3JhbUZvcm1hdC5QTkdcbiAgICAgICAgICAgICAgKVxuICAgICAgICAgICAgKTtcblxuICAgICAgICAgICAgdHJ5IHtcbiAgICAgICAgICAgICAgLy8gU1ZHJ3MgZG9uJ3QgcmVuZGVyIGNvcnJlY3RseSBpbiBub24tc3ZnIGZvcm1hdHMuIFJlcGxhY2Ugd2l0aCBwbmcgZXF1aXZhbGVudC5cbiAgICAgICAgICAgICAgY29uc3QgcG5naWZpZWREb3QgPSBkb3QucmVwbGFjZSgvXFwuc3ZnL2csIFwiLnBuZ1wiKTtcbiAgICAgICAgICAgICAgY29uc3QgcG5nID0gYXdhaXQgdGhpcy5zdHJlYW1Ub0J1ZmZlcihcbiAgICAgICAgICAgICAgICBhd2FpdCB0b1N0cmVhbShwbmdpZmllZERvdCwge1xuICAgICAgICAgICAgICAgICAgZm9ybWF0OiBcInBuZ1wiLFxuICAgICAgICAgICAgICAgIH0pXG4gICAgICAgICAgICAgICk7XG5cbiAgICAgICAgICAgICAgZnMuZW5zdXJlRGlyU3luYyhwYXRoLmRpcm5hbWUocG5nRmlsZSkpO1xuICAgICAgICAgICAgICBmcy53cml0ZUZpbGVTeW5jKHBuZ0ZpbGUsIHBuZyk7XG5cbiAgICAgICAgICAgICAgY29udGV4dC5sb2dBcnRpZmFjdChcbiAgICAgICAgICAgICAgICB0aGlzLFxuICAgICAgICAgICAgICAgIENka0dyYXBoRGlhZ3JhbVBsdWdpbi5hcnRpZmFjdElkKFxuICAgICAgICAgICAgICAgICAgY29uZmlnLm5hbWUsXG4gICAgICAgICAgICAgICAgICBEaWFncmFtRm9ybWF0LlBOR1xuICAgICAgICAgICAgICAgICksXG4gICAgICAgICAgICAgICAgcG5nRmlsZSxcbiAgICAgICAgICAgICAgICBgRGlhZ3JhbSBnZW5lcmF0ZWQgXCJwbmdcIiBmaWxlIGZvciAke2NvbmZpZy5uYW1lfSAtIFwiJHtjb25maWcudGl0bGV9XCJgXG4gICAgICAgICAgICAgICk7XG4gICAgICAgICAgICB9IGNhdGNoIChlcnJvcikge1xuICAgICAgICAgICAgICBjb25zb2xlLmVycm9yKGVycm9yKTtcbiAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICAgICAgIGBGYWlsZWQgdG8gZ2VuZXJhdGUgUE5HIGRpYWdyYW0gZm9yICR7XG4gICAgICAgICAgICAgICAgICBjb25maWcubmFtZVxuICAgICAgICAgICAgICAgIH0gYXQgXCIke3BuZ0ZpbGV9XCIgLSAke1N0cmluZyhlcnJvcil9YFxuICAgICAgICAgICAgICApO1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfVxuXG4gICAgICAvLyBOQjogYWRkIGRyYXdpbyBwcm92aWRlciBzdXBwb3J0IGhlcmVcbiAgICB9XG4gIH07XG59XG4iXX0=