"use strict";
var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s;
Object.defineProperty(exports, "__esModule", { value: true });
exports.deserializeStore = exports.isResourceLike = exports.RootNode = exports.AppNode = exports.StageNode = exports.NestedStackNode = exports.StackNode = exports.ParameterNode = exports.OutputNode = exports.CfnResourceNode = exports.ResourceNode = exports.Node = exports.ImportReference = exports.AttributeReference = exports.Reference = exports.Dependency = exports.Edge = exports.BaseEntity = exports.Store = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const console_1 = require("console");
const constructs_1 = require("constructs");
const cloneDeep = require("lodash.clonedeep"); // eslint-disable-line @typescript-eslint/no-require-imports
const isEmpty = require("lodash.isempty"); // eslint-disable-line @typescript-eslint/no-require-imports
const omit = require("lodash.omit"); // eslint-disable-line @typescript-eslint/no-require-imports
const uniq = require("lodash.uniq"); // eslint-disable-line @typescript-eslint/no-require-imports
const counter_1 = require("./counter");
const types_1 = require("./types");
const utils_1 = require("./utils");
/** Store class provides the in-memory database-like interface for managing all entities in the graph */
class Store {
    /** Builds store from serialized store data */
    static fromSerializedStore(serializedStore) {
        return deserializeStore(serializedStore);
    }
    constructor(allowDestructiveMutations = false) {
        /** Current SemVer version of the store */
        this.version = "0.0.0";
        /** @internal */
        this._edges = new Map();
        /** @internal */
        this._nodes = new Map();
        /** @internal */
        this._stacks = new Map();
        /** @internal */
        this._stages = new Map();
        /** @internal */
        this._logicalIdLookup = new Map();
        /** @internal */
        this._importArnTokenLookup = new Map();
        /** @internal */
        this._counters = {
            cfnResources: new counter_1.Counter(),
            nodeTypes: new counter_1.Counter(),
            edgeTypes: new counter_1.Counter(),
        };
        this._root = new RootNode(this);
        this.allowDestructiveMutations = allowDestructiveMutations;
    }
    /**
     * Root node in the store. The **root** node is not the computed root, but the graph root
     * which is auto-generated and can not be mutated.
     */
    get root() {
        return this._root;
    }
    /**
     * Gets all stored **edges**
     * @type ReadonlyArray<Edge>
     */
    get edges() {
        return Array.from(this._edges.values());
    }
    /**
     * Gets all stored **nodes**
     * @type ReadonlyArray<Node>
     */
    get nodes() {
        return Array.from(this._nodes.values());
    }
    /**
     * Gets all stored **stack** nodes
     * @type ReadonlyArray<StackNode>
     */
    get stacks() {
        return Array.from(this._stacks.values());
    }
    /**
     * Gets all stored **stage** nodes
     * @type ReadonlyArray<StageNode>
     */
    get stages() {
        return Array.from(this._stages.values());
    }
    /**
     * Gets all stored **root stack** nodes
     * @type ReadonlyArray<StackNode>
     */
    get rootStacks() {
        return this.stacks.filter((stack) => StackNode.isStackNode(stack));
    }
    /** Get record of all store counters */
    get counts() {
        return {
            nodes: this._nodes.size,
            edges: this._edges.size,
            stacks: this._stacks.size,
            stages: this._stages.size,
            nodeTypes: this._counters.nodeTypes.counts,
            edgeTypes: this._counters.edgeTypes.counts,
            cfnResources: this._counters.cfnResources.counts,
        };
    }
    /** Add **edge** to the store */
    addEdge(edge) {
        this._edges.set(edge.uuid, edge);
        this._counters.edgeTypes.add(edge.edgeType);
    }
    /** Get stored **edge** by UUID */
    getEdge(uuid) {
        const edge = this._edges.get(uuid);
        if (edge != null) {
            return edge;
        }
        throw new Error(`Edge ${uuid} is not defined`);
    }
    /** Add **node** to the store */
    addNode(node) {
        // Do not store root node
        if (RootNode.isRootNode(node) === true) {
            return;
        }
        this._nodes.set(node.uuid, node);
        this._counters.nodeTypes.add(node.nodeType);
        if (CfnResourceNode.isCfnResourceNode(node) && node.cfnType) {
            this._counters.cfnResources.add(node.cfnType);
        }
    }
    /** Get stored **node** by UUID */
    getNode(uuid) {
        // Root node is not stored in "nodes" map
        if (uuid === RootNode.UUID && this.root) {
            return this.root;
        }
        const node = this._nodes.get(uuid);
        if (node != null) {
            return node;
        }
        throw new Error(`Node ${uuid} is not defined`);
    }
    /** Add **stack** node to the store */
    addStack(stack) {
        this._stacks.set(stack.uuid, stack);
    }
    /** Get stored **stack** node by UUID */
    getStack(uuid) {
        const stack = this._stacks.get(uuid);
        if (stack != null) {
            return stack;
        }
        throw new Error(`Stack ${uuid} is not defined`);
    }
    /** Add **stage** to the store */
    addStage(stage) {
        this._stages.set(stage.uuid, stage);
    }
    /** Get stored **stage** node by UUID */
    getStage(uuid) {
        const stage = this._stages.get(uuid);
        if (stage != null) {
            return stage;
        }
        throw new Error(`Stage ${uuid} is not defined`);
    }
    /**
     * Compute **universal** *logicalId* based on parent stack and construct *logicalId* (`<stack>:<logicalId>`).
     *
     * Construct *logicalIds are only unique within their containing stack, so to use *logicalId*
     * lookups universally (like resolving references) we need a universal key.
     */
    computeLogicalUniversalId(stack, logicalId) {
        return `${stack.uuid}:${logicalId}`;
    }
    /** Find node by **universal** *logicalId* (`<stack>:<logicalId>`) */
    findNodeByLogicalUniversalId(uid) {
        const [stackUUID, logicalId] = uid.split(":");
        const stack = this.getStack(stackUUID);
        return this.findNodeByLogicalId(stack, logicalId);
    }
    /** Find node within given **stack** with given *logicalId* */
    findNodeByLogicalId(stack, logicalId) {
        const uid = this.computeLogicalUniversalId(stack, logicalId);
        const nodeUUID = this._logicalIdLookup.get(uid);
        if (nodeUUID == null) {
            if (stack instanceof NestedStackNode && stack.parentStack) {
                return this.findNodeByLogicalId(stack.parentStack, logicalId);
            }
            throw new Error(`Failed to find node by logicalId: ${uid}`);
        }
        const node = this._nodes.get(nodeUUID);
        if (node != null) {
            return node;
        }
        throw new Error(`Unable to find node mapped to logical id ${logicalId}`);
    }
    /** Record a **universal** *logicalId* to node mapping in the store */
    recordLogicalId(stack, logicalId, resource) {
        const uid = this.computeLogicalUniversalId(stack, logicalId);
        this._logicalIdLookup.set(uid, resource.uuid);
    }
    /**
     * Records arn tokens from imported resources (eg: `s3.Bucket.fromBucketArn()`)
     * that are used for resolving references.
     */
    recordImportArn(arnToken, resource) {
        this._importArnTokenLookup.set(arnToken, resource.uuid);
    }
    /**
     * Attempts to lookup the {@link Node} associated with a given *import arn token*.
     * @param value Import arn value, which is either object to tokenize or already tokenized string.
     * @returns Returns matching {@link Node} if found, otherwise undefined.
     */
    findNodeByImportArn(value) {
        if (typeof value !== "string") {
            value = (0, utils_1.tokenizeImportArn)(value);
        }
        const nodeUUID = this._importArnTokenLookup.get(value);
        if (nodeUUID) {
            return this._nodes.get(nodeUUID);
        }
        return undefined;
    }
    /** Serialize the store */
    serialize() {
        return {
            version: this.version,
            tree: this.root._serialize(),
            edges: Array.from(this.edges).map((edge) => edge._serialize()),
        };
    }
    /**
     * Clone the store to allow destructive mutations.
     * @param allowDestructiveMutations Indicates if destructive mutations are allowed; defaults to `true`
     * @returns {Store} Returns a clone of the store that allows destructive mutations
     */
    clone(allowDestructiveMutations = true) {
        return deserializeStore(this.serialize(), allowDestructiveMutations);
    }
    /**
     * Verifies that the store allows destructive mutations.
     * @throws Error is store does **not** allow mutations
     */
    verifyDestructiveMutationAllowed() {
        if (!this.allowDestructiveMutations) {
            throw new Error("GraphStore must be a clone to perform destructive mutations");
        }
    }
    /**
     * Remove **edge** from the store
     * @destructive
     */
    mutateRemoveEdge(edge) {
        const deleted = this._edges.delete(edge.uuid);
        if (deleted) {
            this._counters.edgeTypes.subtract(edge.edgeType);
        }
        return deleted;
    }
    /**
     * Remove **node** from the store
     * @destructive
     */
    mutateRemoveNode(node) {
        // Root node can not be removed
        if (RootNode.isRootNode(node) === true) {
            throw new Error("Root not can not be removed");
        }
        if (node.logicalId && node.stack) {
            this._logicalIdLookup.delete(this.computeLogicalUniversalId(node.stack, node.logicalId));
        }
        if (StackNode.isStackNode(node)) {
            this._stacks.delete(node.uuid);
        }
        const deleted = this._nodes.delete(node.uuid);
        if (deleted) {
            this._counters.nodeTypes.subtract(node.nodeType);
            if (CfnResourceNode.isCfnResourceNode(node) && node.cfnType) {
                this._counters.cfnResources.subtract(node.cfnType);
            }
        }
        return deleted;
    }
}
exports.Store = Store;
_a = JSII_RTTI_SYMBOL_1;
Store[_a] = { fqn: "@aws/pdk.cdk_graph.Store", version: "0.23.61" };
/** Base class for all store entities (Node and Edges) */
class BaseEntity {
    constructor(props) {
        /** @internal */
        this._destroyed = false;
        this.store = props.store;
        this.uuid = props.uuid;
        this._attributes = props.attributes || {};
        this._metadata = props.metadata || [];
        this._tags = new Map(Object.entries(props.tags || {}));
        this._flags = new Set(props.flags);
    }
    /**
     * Get *readonly* record of all attributes
     * @type Readonly<SerializedGraph.Attributes>
     */
    get attributes() {
        return cloneDeep(this._attributes);
    }
    /**
     * Get *readonly* list of all metadata entries
     * @type Readonly<SerializedGraph.Metadata>
     */
    get metadata() {
        return cloneDeep(this._metadata);
    }
    /**
     * Get *readonly* record of all tags
     * @type Readonly<SerializedGraph.Tags>
     */
    get tags() {
        return Object.fromEntries(this._tags);
    }
    /**
     * Get *readonly* list of all flags
     * @type ReadonlyArray<FlagEnum>
     */
    get flags() {
        return Array.from(this._flags);
    }
    /** Indicates if the entity has been destroyed (eg: removed from store) */
    get isDestroyed() {
        return this._destroyed;
    }
    /** Indicates if the entity has had destructive mutations applied */
    get isMutated() {
        return this.hasFlag(types_1.FlagEnum.MUTATED);
    }
    /** Indicates if entity has a given attribute defined, and optionally with a specific value */
    hasAttribute(key, value) {
        if (key in this._attributes) {
            if (value !== undefined) {
                return this._attributes[key] === value;
            }
            return true;
        }
        return false;
    }
    /**
     * Add attribute.
     *
     * @throws Error if attribute for key already exists
     */
    addAttribute(key, value) {
        if (this.hasAttribute(key)) {
            throw new Error(`Entity ${String(this)} already has attribute ${key}; use setAttribute to override`);
        }
        this.setAttribute(key, value);
    }
    /** Set attribute. This will overwrite existing attribute. */
    setAttribute(key, value) {
        // @ts-ignore
        this._attributes[key] = value;
    }
    /** Get attribute by key */
    getAttribute(key) {
        return this._attributes[key];
    }
    /** Add metadata entry */
    addMetadata(metadataType, data) {
        this._metadata.push({
            type: metadataType,
            data,
        });
    }
    /** Indicates if entity has matching metadata entry */
    hasMetadata(metadataType, data) {
        return !!this._metadata.find((metadata) => {
            if (metadata.type !== metadataType)
                return false;
            if (metadata.data !== data)
                return false;
            return true;
        });
    }
    /**
     * Retrieves all metadata entries of a given type
     * @type Readonly<SerializedGraph.Metadata>
     */
    findMetadata(metadataType) {
        return this._metadata.filter((entry) => entry.type === metadataType);
    }
    /**
     * Add tag.
     * @throws Throws Error is tag for key already exists
     */
    addTag(key, value) {
        if (this.hasTag(key)) {
            throw new Error(`Entity ${String(this)} already has tag ${key}; use setTag to override`);
        }
        this.setTag(key, value);
    }
    /** Set tag. Will overwrite existing tag. */
    setTag(key, value) {
        this._tags.set(key, value);
    }
    /** Indicates if entity has tag, optionally verifying tag value */
    hasTag(key, value) {
        if (!this._tags.has(key))
            return false;
        if (value !== undefined && this._tags.get(key) !== value)
            return false;
        return true;
    }
    /** Get tag by key */
    getTag(key) {
        return this._tags.get(key);
    }
    /** Add flag */
    addFlag(flag) {
        this._flags.add(flag);
    }
    /** Indicates if entity has a given flag */
    hasFlag(flag) {
        return this._flags.has(flag);
    }
    /**
     * Applies data (attributes, metadata, tags, flag) to entity.
     *
     * Generally used only for mutations such as collapse and consume to retain data.
     * @param data - The data to apply
     * @param {boolean} [applyFlags=false] - Indicates if data is overwritten
     * @param {boolean} [applyFlags=false] - Indicates if flags should be applied
     */
    applyData(data, overwrite = false, applyFlags = false) {
        if (data.attributes) {
            Object.entries(data.attributes).forEach(([key, value]) => {
                if (overwrite || !this.hasAttribute(key)) {
                    this.setAttribute(key, value);
                }
            });
        }
        if (data.metadata) {
            data.metadata.forEach((v) => {
                if (!this.hasMetadata(v.type, v.data)) {
                    this.addMetadata(v.type, v.data);
                }
            });
        }
        if (data.tags) {
            Object.entries(data.tags).forEach(([key, value]) => {
                if (overwrite || !this.hasTag(key)) {
                    this.setTag(key, value);
                }
            });
        }
        if (applyFlags && data.flags) {
            data.flags.forEach((flag) => {
                this.addFlag(flag);
            });
        }
    }
    /**
     * Performs pre-mutate operations on entity and store
     * @internal
     */
    _preMutate() {
        this.store.verifyDestructiveMutationAllowed();
        this.addFlag(types_1.FlagEnum.MUTATED);
    }
    /**
     * Serialize entity
     * @internal
     */
    _serialize() {
        return {
            uuid: this.uuid,
            attributes: isEmpty(this._attributes) ? undefined : this._attributes,
            metadata: isEmpty(this._metadata) ? undefined : this._metadata,
            tags: this._tags.size ? Object.fromEntries(this._tags) : undefined,
            flags: this._flags.size ? Array.from(this._flags) : undefined,
        };
    }
}
exports.BaseEntity = BaseEntity;
_b = JSII_RTTI_SYMBOL_1;
BaseEntity[_b] = { fqn: "@aws/pdk.cdk_graph.BaseEntity", version: "0.23.61" };
/** Edge class defines a link (relationship) between nodes, as in standard [graph theory](https://en.wikipedia.org/wiki/Graph_theory) */
class Edge extends BaseEntity {
    /** Find first edge matching predicate within an EdgeChain */
    static findInChain(chain, predicate) {
        for (const entry of chain) {
            if (Array.isArray(entry)) {
                const edge = Edge.findInChain(entry, predicate);
                if (edge)
                    return edge;
            }
            else {
                if (predicate.filter(entry))
                    return entry;
            }
        }
        return undefined;
    }
    /** Find all matching edges based on predicate within an EdgeChain */
    static findAllInChain(chain, predicate) {
        const edges = [];
        for (const entry of chain) {
            if (Array.isArray(entry)) {
                const edge = Edge.findInChain(entry, predicate);
                if (edge) {
                    edges.push(edge);
                }
            }
            else {
                if (predicate.filter(entry)) {
                    edges.push(entry);
                }
            }
        }
        return edges;
    }
    /** Edge **source** is the node that defines the edge (tail) */
    get source() {
        return this._source;
    }
    /** Edge **target** is the node being referenced by the **source** (head) */
    get target() {
        return this._target;
    }
    /** Indicates the direction in which the edge is directed */
    get direction() {
        return this._direction;
    }
    /** Indicates if **source** and **target** nodes reside in different *root* stacks */
    get isCrossStack() {
        return this._source.rootStack !== this._target.rootStack;
    }
    /**
     * Indicates if the Edge's **source** and **target** are the same, or were the same
     * when it was created (prior to mutations).
     *
     * To check whether it was originally closed, use `hasFlag(FlagEnum.CLOSED_EDGE)` instead.
     */
    get isClosed() {
        return this._source === this._target || this.hasFlag(types_1.FlagEnum.CLOSED_EDGE);
    }
    /**
     * Indicates if edge is extraneous which is determined by explicitly having *EXTRANEOUS* flag
     * added and/or being a closed loop (source===target).
     */
    get isExtraneous() {
        return this.hasFlag(types_1.FlagEnum.EXTRANEOUS) || this.isClosed;
    }
    constructor(props) {
        super(props);
        this.edgeType = props.edgeType;
        this._direction = props.direction;
        this._source = props.source;
        this._target = props.target;
        // Do not change original closed edge flag from a mutation.
        if (this._target === this._source && this.hasFlag(types_1.FlagEnum.MUTATED)) {
            this.addFlag(types_1.FlagEnum.CLOSED_EDGE);
        }
        // wire up links
        this._source.addLink(this);
        this._target.addReverseLink(this);
        this.store.addEdge(this);
    }
    /**
     * Indicates if this edge is equivalent to another edge.
     *
     * Edges are considered equivalent if they share same type, source, and target.
     */
    isEquivalent(edge) {
        if (edge.edgeType !== this.edgeType)
            return false;
        if (edge.source !== this.source)
            return false;
        if (edge.target !== this.target)
            return false;
        return true;
    }
    /** Indicates if edge allows destructive mutations */
    get allowDestructiveMutations() {
        return this.store.allowDestructiveMutations;
    }
    /**
     * Change the edge **direction**
     * @destructive
     */
    mutateDirection(direction) {
        this._preMutate();
        this._direction = direction;
    }
    /**
     * Change the edge **source**
     * @destructive
     */
    mutateSource(node) {
        this._preMutate();
        this._source.mutateRemoveLink(this);
        this._source = node;
        this._source.addLink(this);
    }
    /**
     * Change the edge **target**
     * @destructive
     */
    mutateTarget(node) {
        this._preMutate();
        this._target.mutateRemoveReverseLink(this);
        this._target = node;
        this._target.addReverseLink(this);
    }
    /**
     * Destroy the edge. Remove all references and remove from store.
     * @destructive
     */
    mutateDestroy(_strict = false) {
        this._preMutate();
        this.source.mutateRemoveLink(this);
        this.target.mutateRemoveReverseLink(this);
        this.store.mutateRemoveEdge(this);
        this._destroyed = true;
    }
    /**
     * Merge an equivalent edge's data into this edge and destroy the other edge.
     *
     * Used during filtering operations to consolidate equivalent edges.
     * @param edge - The edge to consume
     * @throws Error is edge is not *equivalent*
     * @destructive
     */
    mutateConsume(edge) {
        this._preMutate();
        if (!this.isEquivalent(edge)) {
            throw new Error(`Only equivalent edges can be consumed: ${edge} > ${this}`);
        }
        // propagate edge data
        this.applyData(edge);
        // destroy the consumed edge
        edge.mutateDestroy();
    }
    /** Get string representation of this edge */
    toString() {
        return `Edge:${this.edgeType}::${this.uuid}::${this.direction}(${this.source}->${this.target})`;
    }
    /** @internal */
    _serialize() {
        return {
            ...super._serialize(),
            edgeType: this.edgeType,
            direction: this.direction,
            source: this.source.uuid,
            target: this.target.uuid,
        };
    }
}
exports.Edge = Edge;
_c = JSII_RTTI_SYMBOL_1;
Edge[_c] = { fqn: "@aws/pdk.cdk_graph.Edge", version: "0.23.61" };
/** Dependency edge class defines CloudFormation dependency between resources */
class Dependency extends Edge {
    /** Indicates if given edge is a {@link Dependency} edge */
    static isDependency(edge) {
        return edge.edgeType === types_1.EdgeTypeEnum.DEPENDENCY;
    }
    constructor(props) {
        super({
            ...props,
            edgeType: types_1.EdgeTypeEnum.DEPENDENCY,
            direction: types_1.EdgeDirectionEnum.FORWARD,
        });
        this.addFlag(types_1.FlagEnum.EXTRANEOUS);
    }
}
exports.Dependency = Dependency;
_d = JSII_RTTI_SYMBOL_1;
Dependency[_d] = { fqn: "@aws/pdk.cdk_graph.Dependency", version: "0.23.61" };
/** Edge prefix to denote dependency edge  */
Dependency.PREFIX = "DEP:";
/** Reference edge class defines a directed relationship between nodes  */
class Reference extends Edge {
    /** Indicates if edge is a {@link Reference} */
    static isReference(edge) {
        return edge.edgeType === types_1.EdgeTypeEnum.REFERENCE;
    }
    /** Indicates if edge is a **Ref** based {@link Reference} edge */
    static isRef(edge) {
        return edge.referenceType === types_1.ReferenceTypeEnum.REF;
    }
    constructor(props) {
        super({
            edgeType: types_1.EdgeTypeEnum.REFERENCE,
            direction: types_1.EdgeDirectionEnum.FORWARD,
            ...props,
        });
        this.setAttribute(Reference.ATT_TYPE, props.referenceType || types_1.ReferenceTypeEnum.REF);
    }
    /** Get type of reference */
    get referenceType() {
        return this.getAttribute(Reference.ATT_TYPE);
    }
    /** Resolve reference chain */
    resolveChain() {
        if (OutputNode.isOutputNode(this.target)) {
            function _resolveChain(_ref) {
                if (OutputNode.isOutputNode(_ref.target)) {
                    return [
                        _ref,
                        ..._ref.target.referenceLinks.map(_resolveChain),
                    ];
                }
                return [_ref];
            }
            return [
                this,
                ...this.target.referenceLinks.map(_resolveChain),
            ];
        }
        return [this];
    }
    /**
     * Resolve targets by following potential edge chain.
     *
     * @see {@link EdgeChain}
     */
    resolveTargets() {
        if (OutputNode.isOutputNode(this.target)) {
            function resolveOutputTarget(_target) {
                if (OutputNode.isOutputNode(_target))
                    return resolveOutputTarget(_target);
                return [_target];
            }
            return this.target.referenceLinks.flatMap((ref) => resolveOutputTarget(ref.target));
        }
        return [this.target];
    }
}
exports.Reference = Reference;
_e = JSII_RTTI_SYMBOL_1;
Reference[_e] = { fqn: "@aws/pdk.cdk_graph.Reference", version: "0.23.61" };
/** Edge prefix to denote **Ref** type reference edge  */
Reference.PREFIX = "REF:";
/** Attribute defining the type of reference */
Reference.ATT_TYPE = "graph:reference:type";
/** Attribute type reference edge */
class AttributeReference extends Reference {
    /** Indicates if edge in an **Fn::GetAtt** {@link Reference} */
    static isAtt(edge) {
        return edge.referenceType === types_1.ReferenceTypeEnum.ATTRIBUTE;
    }
    constructor(props) {
        super({
            ...props,
            referenceType: types_1.ReferenceTypeEnum.ATTRIBUTE,
        });
        this.setAttribute(AttributeReference.ATT_VALUE, props.value);
    }
    /** Get the resolved attribute value */
    get value() {
        return this.getAttribute(AttributeReference.ATT_VALUE);
    }
}
exports.AttributeReference = AttributeReference;
_f = JSII_RTTI_SYMBOL_1;
AttributeReference[_f] = { fqn: "@aws/pdk.cdk_graph.AttributeReference", version: "0.23.61" };
/** Edge prefix to denote **Fn::GetAtt** type reference edge  */
AttributeReference.PREFIX = "ATT:";
/** Attribute key for resolved value of attribute reference */
AttributeReference.ATT_VALUE = "graph:reference:attribute:value";
/** Import reference defines **Fn::ImportValue** type reference edge. */
class ImportReference extends Reference {
    /** Indicates if edge is **Fn::ImportValue** based {@link Reference} */
    static isImport(edge) {
        return edge.referenceType === types_1.ReferenceTypeEnum.IMPORT;
    }
    constructor(props) {
        super({
            ...props,
            referenceType: types_1.ReferenceTypeEnum.IMPORT,
        });
    }
}
exports.ImportReference = ImportReference;
_g = JSII_RTTI_SYMBOL_1;
ImportReference[_g] = { fqn: "@aws/pdk.cdk_graph.ImportReference", version: "0.23.61" };
/** Edge prefix to denote **Fn::ImportValue** type reference edge */
ImportReference.PREFIX = "IMP:";
/** Node class is the base definition of **node** entities in the graph, as in standard [graph theory](https://en.wikipedia.org/wiki/Graph_theory) */
class Node extends BaseEntity {
    /** Stack the node is contained in */
    get stack() {
        return this._stack;
    }
    /** Parent node. Only the root node should not have parent. */
    get parent() {
        return this._parent;
    }
    constructor(props) {
        super(props);
        /** @internal */
        this._children = new Map();
        /** @internal */
        this._links = new Map();
        /** @internal */
        this._reverseLinks = new Map();
        this.nodeType = props.nodeType;
        this.id = props.id;
        this.path = props.path;
        this.constructInfo = props.constructInfo;
        this._cfnType = props.cfnType;
        this._parent = props.parent;
        this.depth = this.parent ? this.parent.depth + 1 : 0;
        this._stack = props.stack || (this instanceof StackNode ? this : undefined);
        this.logicalId = props.logicalId;
        if (this.logicalId) {
            if (this.stack == null) {
                throw new Error(`LogicalId defined outside of stack: ${this.logicalId} - ${String(this)}`);
            }
            this.store.recordLogicalId(this.stack, this.logicalId, this);
        }
        if (this.parent) {
            this.parent.addChild(this);
        }
        this.store.addNode(this);
    }
    /** Gets descending ordered list of ancestors from the root */
    get scopes() {
        if (this.parent) {
            return [...this.parent.scopes, this.parent];
        }
        return [];
    }
    /** Indicates if node is direct child of the graph root node */
    get isTopLevel() {
        return this.parent === this.store.root;
    }
    /** Get **root** stack */
    get rootStack() {
        if (StackNode.isStackNode(this))
            return this;
        return this.scopes.find((scope) => StackNode.isStackNode(scope));
    }
    /** Get all direct child nodes */
    get children() {
        return Array.from(this._children.values());
    }
    /** Indicates if this node is a *leaf* node, which means it does not have children */
    get isLeaf() {
        return this._children.size === 0;
    }
    /** Gets all links (edges) in which this node is the **source** */
    get links() {
        return Array.from(this._links.values());
    }
    /** Gets all links (edges) in which this node is the **target** */
    get reverseLinks() {
        return Array.from(this._reverseLinks.values());
    }
    /** Synthesized construct information defining jii resolution data */
    get constructInfoFqn() {
        return this.constructInfo?.fqn;
    }
    /** Indicates if node is a *Custom Resource* */
    get isCustomResource() {
        return types_1.ConstructInfoFqnEnum.CUSTOM_RESOURCE === this.constructInfoFqn;
    }
    /**
     * Indicates if node ConstructInfoFqn denotes a `aws-cdk-lib.*.Cfn*` construct.
     * @see {@link FlagEnum.CFN_FQN}
     */
    get isCfnFqn() {
        return this.hasFlag(types_1.FlagEnum.CFN_FQN);
    }
    /** Gets CloudFormation properties for this node */
    get cfnProps() {
        return this.attributes[types_1.CfnAttributesEnum.PROPS];
    }
    /** Get the CloudFormation resource type for this node */
    get cfnType() {
        return this._cfnType;
    }
    /** Gets list of {@link Dependency} links (edges) where this node is the **source** */
    get dependencyLinks() {
        return Array.from(this._links.values()).filter((link) => {
            return link.edgeType === types_1.EdgeTypeEnum.DEPENDENCY;
        });
    }
    /** Gets list of {@link Dependency} links (edges) where this node is the **target** */
    get reverseDependencyLinks() {
        return Array.from(this._reverseLinks.values()).filter((link) => {
            return link.edgeType === types_1.EdgeTypeEnum.DEPENDENCY;
        });
    }
    /** Gets list of {@link Reference} links (edges) where this node is the **source** */
    get referenceLinks() {
        return Array.from(this._links.values()).filter((link) => {
            return link.edgeType === types_1.EdgeTypeEnum.REFERENCE;
        });
    }
    /** Gets list of {@link Reference} links (edges) where this node is the **target** */
    get reverseReferenceLinks() {
        return Array.from(this._reverseLinks.values()).filter((link) => {
            return link.edgeType === types_1.EdgeTypeEnum.REFERENCE;
        });
    }
    /**
     * Get list of **Nodes** that *this node references*
     * @see {@link Node.referenceLinks}
     */
    get references() {
        return uniq(this.referenceLinks.flatMap((link) => link.resolveTargets()));
    }
    /**
     * Get list of **Nodes** that *reference this node*
     * @see {@link Node.reverseReferenceLinks}
     */
    get referencedBy() {
        return uniq(this.reverseReferenceLinks.flatMap((link) => link.source));
    }
    /**
     * Get list of **Nodes** that *this node depends on*
     * @see {@link Node.dependencyLinks}
     */
    get dependencies() {
        return uniq(this.dependencyLinks.flatMap((link) => link.target));
    }
    /**
     * Get list of **Nodes** that *depend on this node*
     * @see {@link Node.reverseDependencyLinks}
     */
    get dependedOnBy() {
        return uniq(this.reverseDependencyLinks.flatMap((link) => link.source));
    }
    /** Indicates if this node is considered a {@link FlagEnum.GRAPH_CONTAINER} */
    get isGraphContainer() {
        return this.hasFlag(types_1.FlagEnum.GRAPH_CONTAINER);
    }
    /** Indicates if this node is considered a {@link FlagEnum.CLUSTER} */
    get isCluster() {
        return this.hasFlag(types_1.FlagEnum.CLUSTER);
    }
    /**
     * Indicates if this node is considered a {@link FlagEnum.EXTRANEOUS} node
     * or determined to be extraneous:
     * - Clusters that contain no children
     */
    get isExtraneous() {
        return this.hasFlag(types_1.FlagEnum.EXTRANEOUS) || (this.isCluster && this.isLeaf);
    }
    /** Indicates if this node is considered a {@link FlagEnum.ASSET} */
    get isAsset() {
        return this.hasFlag(types_1.FlagEnum.ASSET);
    }
    /** Get list of *siblings* of this node. */
    get siblings() {
        if (this.parent) {
            return this.parent.children.filter((child) => child !== this);
        }
        return [];
    }
    /** Get specific CloudFormation property */
    getCfnProp(key) {
        return this.cfnProps && this.cfnProps[key];
    }
    /** Add *link* to another node */
    addLink(edge) {
        this._links.set(edge.uuid, edge);
    }
    /** Add *link* from another node */
    addReverseLink(edge) {
        this._reverseLinks.set(edge.uuid, edge);
    }
    /** Add *child* node */
    addChild(node) {
        this._children.set(node.id, node);
    }
    /** Indicates if specific *node* is a *child* of *this node* */
    isChild(node) {
        for (const child of this._children.values()) {
            if (child === node)
                return true;
        }
        return false;
    }
    /** Indicates if a specific *node* is an *ancestor* of *this node* */
    isAncestor(ancestor) {
        return this.scopes.includes(ancestor);
    }
    /**
     * Find nearest *ancestor* of *this node* matching given predicate.
     * @param predicate - Predicate to match ancestor
     * @max {number} [max] - Optional maximum levels to ascend
     */
    findAncestor(predicate, max) {
        let ancestors = this.scopes.slice().reverse();
        if (max) {
            ancestors = ancestors.slice(0, max);
        }
        return ancestors.find(predicate.filter);
    }
    /**
     * Gets the nearest **common** *ancestor* shared between *this node* and another *node*.
     * @throws Error if *node* does not share a **common** *ancestor*
     */
    getNearestAncestor(node) {
        if (node === this)
            throw new Error("Node is the current node");
        const aScopes = this.scopes.reverse();
        const bScopes = node.scopes.reverse();
        for (const aScope of aScopes) {
            for (const bScope of bScopes) {
                if (aScope === bScope)
                    return aScope;
            }
        }
        throw new Error(`Nodes do not share common ancestor: ${String(this)} ^ ${String(node)}`);
    }
    /**
     * Return this construct and all of its sub-nodes in the given order.
     *
     * Optionally filter nodes based on predicate.
     */
    findAll(options) {
        const { predicate, order = constructs_1.ConstructOrder.PREORDER } = options || {};
        const all = new Array();
        function visit(c) {
            if (order === constructs_1.ConstructOrder.PREORDER && !RootNode.isRootNode(c)) {
                all.push(c);
            }
            for (const child of c.children) {
                visit(child);
            }
            if (order === constructs_1.ConstructOrder.POSTORDER && !RootNode.isRootNode(c)) {
                all.push(c);
            }
        }
        visit(this);
        if (predicate) {
            return all.filter(predicate.filter);
        }
        return all;
    }
    /** Recursively find the nearest sub-node matching predicate */
    find(predicate) {
        if (predicate.filter(this))
            return this;
        for (const child of this.children) {
            const node = child.find(predicate);
            if (node != null)
                return node;
        }
        return undefined;
    }
    /**
     * Get *child* node with given *id*.
     *
     * @throws Error if no child with given id
     */
    getChild(id) {
        const child = this._children.get(id);
        if (child == null) {
            throw new Error(`${String(this)} does not have child with id "${id}"`);
        }
        return child;
    }
    /** Find child with given *id*. Similar to `find` but does not throw error if no child found. */
    findChild(id) {
        return this._children.get(id);
    }
    /**
     * Return all direct links of this node and that of all sub-nodes.
     *
     * Optionally filter links based on predicate.
     */
    findAllLinks(options) {
        const { predicate, order = constructs_1.ConstructOrder.PREORDER, reverse, } = options || {};
        const all = new Array();
        visit(this);
        if (predicate) {
            return all.filter(predicate.filter);
        }
        return all;
        function visit(c) {
            if (order === constructs_1.ConstructOrder.PREORDER) {
                all.push(...c[reverse ? "reverseLinks" : "links"]);
            }
            for (const child of c.children) {
                visit(child);
            }
            if (order === constructs_1.ConstructOrder.POSTORDER) {
                all.push(...c[reverse ? "reverseLinks" : "links"]);
            }
        }
    }
    /**
     * Resolve all link chains
     * @see {@link EdgeChain}
     */
    getLinkChains(reverse = false) {
        let links = this[reverse ? "reverseLinks" : "links"];
        return links.map((link) => {
            if (Reference.isReference(link)) {
                return link.resolveChain();
            }
            return [link];
        });
    }
    /**
     * Find link of this node based on predicate. By default this will follow link
     * chains to evaluate the predicate against and return the matching direct link
     * of this node.
     *
     * @param predicate Edge predicate function to match edge
     * @param reverse Indicates if links are search in reverse order
     * @param follow Indicates if link chain is followed
     * @param direct Indicates that only *direct* links should be searched
     * @returns
     */
    findLink(predicate, reverse = false, follow = true, direct = true) {
        if (follow) {
            const chains = this.getLinkChains(reverse);
            for (const chain of chains) {
                const edge = Edge.findInChain(chain, predicate);
                if (edge) {
                    if (direct)
                        return chain[0];
                    return edge;
                }
            }
            return undefined;
        }
        return this[reverse ? "reverseLinks" : "links"].find(predicate.filter);
    }
    /**
     * Find all links of this node based on predicate. By default this will follow link
     * chains to evaluate the predicate against and return the matching direct links
     * of this node.
     *
     * @param predicate Edge predicate function to match edge
     * @param reverse Indicates if links are search in reverse order
     * @param follow Indicates if link chain is followed
     * @param direct Indicates that only *direct* links should be searched
     * @returns
     */
    findLinks(predicate, reverse = false, follow = true, direct = true) {
        if (follow) {
            return this.getLinkChains(reverse).flatMap((chain) => {
                const edges = Edge.findAllInChain(chain, predicate);
                if (direct) {
                    return edges.length ? [chain[0]] : [];
                }
                return edges;
            });
        }
        return this[reverse ? "reverseLinks" : "links"].filter(predicate.filter);
    }
    /** Indicates if *this node* references *another node* */
    doesReference(node) {
        return this.references.includes(node);
    }
    /** Indicates if *this node* depends on *another node* */
    doesDependOn(node) {
        return this.dependencies.includes(node);
    }
    /**
     * Indicates if this node allows destructive mutations
     * @see {@link Store.allowDestructiveMutations}
     */
    get allowDestructiveMutations() {
        return this.store.allowDestructiveMutations;
    }
    /**
     * Collapses all sub-nodes of *this node* into *this node*.
     * @destructive
     */
    mutateCollapse() {
        this._preMutate();
        this.children.forEach((child) => child.mutateCollapseToParent());
        this._mutateReconcileLinks();
    }
    /**
     * Collapses *this node* into *it's parent node*
     * @destructive
     */
    mutateCollapseToParent() {
        this._preMutate();
        if (this.parent == null) {
            throw new Error(`${this} does not have parent to collapse to.`);
        }
        return this.mutateCollapseTo(this.parent);
    }
    /**
     * Collapses *this node* into *an ancestor*
     * @destructive
     */
    mutateCollapseTo(ancestor) {
        this._preMutate();
        if (!this.isAncestor(ancestor)) {
            throw new Error(`${ancestor} is not an ancestor of ${this}`);
        }
        // TODO: should we retain the child attributes somewhere?
        this.children.forEach((child) => {
            if (child.isDestroyed)
                return;
            child.mutateCollapseToParent();
        });
        this._mutateReconcileLinks();
        // redirect all links to parent
        // while also deleting links to parent
        this.links.forEach((link) => {
            if (link.isDestroyed)
                return;
            if (link.target === ancestor) {
                link.mutateDestroy();
            }
            else {
                link.mutateSource(ancestor);
            }
        });
        // redirect all "reverse" links to parent
        // while also deleting links from parent
        this.reverseLinks.forEach((link) => {
            if (link.isDestroyed)
                return;
            if (link.source === ancestor) {
                link.mutateDestroy();
            }
            else {
                link.mutateTarget(ancestor);
            }
        });
        this.mutateDestroy(true);
        ancestor._mutateReconcileLinks();
        return ancestor;
    }
    /**
     * Destroys this node by removing all references and removing this node from the store.
     * @param {boolean} [strict=false] - Indicates that this node must not have references
     * @destructive
     */
    mutateDestroy(strict = false) {
        this._preMutate();
        if (strict) {
            if (this.children.length) {
                throw new Error(`[strict] ${this} can not destroys because it has children`);
            }
            if (this.links.length || this.reverseLinks.length) {
                throw new Error(`[strict] ${this} can not destroys because there are links referencing it`);
            }
        }
        if (strict && (this.links.length || this.reverseLinks.length)) {
            throw new Error(`[strict] ${this} can not destroys because there are links referencing it`);
        }
        this.children.forEach((child) => {
            child.mutateDestroy();
        });
        this.links.forEach((link) => {
            link.mutateDestroy();
        });
        this.reverseLinks.forEach((link) => {
            link.mutateDestroy();
        });
        if (this.parent) {
            this.parent.mutateRemoveChild(this);
        }
        this._parent = undefined;
        this._stack = undefined;
        this.store.mutateRemoveNode(this);
        this._destroyed = true;
    }
    /**
     * Reconciles links defined by this node. During mutations, multiple *equivalent* links may exist and should be
     * consolidated into a single link. This operation should be called after collapsing children to remove duplicates.
     * @internal
     * @destructive
     */
    _mutateReconcileLinks() {
        this._preMutate();
        const links = this.links;
        for (const a of links) {
            if (a.isDestroyed)
                continue;
            if (a.isClosed && a.edgeType !== types_1.EdgeTypeEnum.CUSTOM) {
                a.mutateDestroy();
                continue;
            }
            for (const b of links) {
                if (a === b || b.isDestroyed)
                    continue;
                if (a.isEquivalent(b)) {
                    a.mutateConsume(b);
                }
            }
        }
        const reverseLinks = this.reverseLinks;
        for (const a of reverseLinks) {
            if (a.isDestroyed)
                continue;
            if (a.isClosed && a.edgeType !== types_1.EdgeTypeEnum.CUSTOM) {
                a.mutateDestroy();
                continue;
            }
            for (const b of reverseLinks) {
                if (a === b || b.isDestroyed)
                    continue;
                if (a.isEquivalent(b)) {
                    a.mutateConsume(b);
                }
            }
        }
    }
    /**
     * Remove a *child* node from *this node*
     * @destructive
     */
    mutateRemoveChild(node) {
        this._preMutate();
        if (!this.isChild(node)) {
            throw new Error(`${node} is not a child of ${this}`);
        }
        // NB: children are stored by "id" not "uuid"
        return this._children.delete(node.id);
    }
    /**
     * Remove a *link* from *this node*
     * @destructive
     */
    mutateRemoveLink(link) {
        this._preMutate();
        return this._links.delete(link.uuid);
    }
    /**
     * Remove a *link* to *this node*
     * @destructive
     */
    mutateRemoveReverseLink(link) {
        this._preMutate();
        return this._reverseLinks.delete(link.uuid);
    }
    /**
     * Hoist *this node* to an *ancestor* by removing it from its current parent node and
     * in turn moving it to the ancestor.
     * @destructive
     */
    mutateHoist(newParent) {
        this._preMutate();
        if (!this.isAncestor(newParent)) {
            throw new Error(`${newParent} is not an ancestor of ${this}`);
        }
        if (this.parent) {
            this.parent.mutateRemoveChild(this);
        }
        this._parent = newParent;
        newParent.addChild(this);
        if (this.stack &&
            this.stack !== this &&
            !this.isAncestor(this.stack)) {
            this._stack = this.findAncestor({
                filter: (node) => StackNode.isStackNode(node) ||
                    NestedStackNode.isNestedStackNode(node),
            });
        }
    }
    /**
     * Hoist all children to parent and collapse node to parent.
     * @destructive
     */
    mutateUncluster() {
        this._preMutate();
        if (this.parent && !this.isLeaf) {
            for (const child of this.children) {
                child.mutateHoist(this.parent);
            }
            this.mutateCollapseToParent();
        }
    }
    /**
     * Move this node into a new parent node.
     * @param {Node} newParent - The parent to move this node to.
     * @destructive
     */
    mutateMove(newParent) {
        this._preMutate();
        if (this.parent) {
            this.parent.mutateRemoveChild(this);
            this.parent._mutateReconcileLinks();
        }
        newParent.addChild(this);
        this._parent = newParent;
        newParent._mutateReconcileLinks();
    }
    /** Get string representation of this node */
    toString() {
        return `Node:${this.nodeType}::${this.uuid}`;
    }
    /**
     * Serialize this node
     * @internal
     */
    _serialize() {
        return {
            ...super._serialize(),
            nodeType: this.nodeType,
            stack: this.stack?.uuid,
            parent: this.parent?.uuid,
            id: this.id,
            path: this.path,
            constructInfo: this.constructInfo,
            logicalId: this.logicalId,
            cfnType: this.cfnType,
            edges: this._links.size
                ? Array.from(this._links.values()).map(({ uuid }) => uuid)
                : undefined,
            children: this._children.size
                ? Object.fromEntries(Array.from(this._children.entries()).map(([key, node]) => [
                    key,
                    node._serialize(),
                ]))
                : undefined,
        };
    }
}
exports.Node = Node;
_h = JSII_RTTI_SYMBOL_1;
Node[_h] = { fqn: "@aws/pdk.cdk_graph.Node", version: "0.23.61" };
/** ResourceNode class defines a L2 cdk resource construct */
class ResourceNode extends Node {
    /** Indicates if node is a {@link ResourceNode} */
    static isResourceNode(node) {
        return node.nodeType === types_1.NodeTypeEnum.RESOURCE;
    }
    constructor(props) {
        super({
            nodeType: types_1.NodeTypeEnum.RESOURCE,
            ...props,
        });
        if (props.cdkOwned) {
            this.addFlag(types_1.FlagEnum.CDK_OWNED);
        }
    }
    /** Get the CloudFormation resource type for this L2 resource or for the L1 resource is wraps. */
    get cfnType() {
        return (super.cfnType ||
            this.getAttribute(ResourceNode.ATT_WRAPPED_CFN_TYPE) ||
            this.cfnResource?.cfnType ||
            (this.isCustomResource ? types_1.CfnResourceTypes.CUSTOM_RESOURCE : undefined));
    }
    /** Indicates if this resource is owned by cdk (defined in cdk library) */
    get isCdkOwned() {
        return this.hasFlag(types_1.FlagEnum.CDK_OWNED);
    }
    /** Indicates if Resource wraps a single CfnResource */
    get isWrapper() {
        return this.children.length === 1 && this.cfnResource !== undefined;
    }
    /** Get the default/primary CfnResource that this Resource wraps */
    get cfnResource() {
        if (this._cfnResource !== undefined) {
            if (this._cfnResource && this._cfnResource.isDestroyed)
                return undefined;
            return this._cfnResource || undefined;
        }
        const resourceNode = this.findChild(types_1.CdkConstructIds.RESOURCE);
        if (resourceNode) {
            this._cfnResource = resourceNode;
            return resourceNode;
        }
        const defaultNode = this.findChild(types_1.CdkConstructIds.DEFAULT);
        if (defaultNode) {
            this._cfnResource = defaultNode;
            return defaultNode;
        }
        if (this.isCdkOwned && this.children.length === 1) {
            const child = this.children[0];
            if (CfnResourceNode.isCfnResourceNode(child)) {
                this._cfnResource = child;
                return child;
            }
        }
        // prevent looking up again by setting to `null`
        this._cfnResource = null;
        return undefined;
    }
    /** Get the cfn properties from the L1 resource that this L2 resource wraps */
    get cfnProps() {
        if (this.cfnResource) {
            return this.cfnResource.cfnProps;
        }
        return this.getAttribute(ResourceNode.ATT_WRAPPED_CFN_PROPS);
    }
    /**
     * Modifies the L1 resource wrapped by this L2 resource
     * @param cfnResource
     * @destructive
     */
    mutateCfnResource(cfnResource) {
        this._preMutate();
        this._cfnResource = cfnResource || null;
    }
    /** @inheritdoc */
    mutateRemoveChild(node) {
        if (this._cfnResource === node) {
            this.mutateCfnResource(undefined);
        }
        return super.mutateRemoveChild(node);
    }
}
exports.ResourceNode = ResourceNode;
_j = JSII_RTTI_SYMBOL_1;
ResourceNode[_j] = { fqn: "@aws/pdk.cdk_graph.ResourceNode", version: "0.23.61" };
/** Attribute key for cfn resource type */
ResourceNode.ATT_WRAPPED_CFN_TYPE = "graph:resource:cfn-type";
/** Attribute key for cfn properties */
ResourceNode.ATT_WRAPPED_CFN_PROPS = "graph:resource:cfn-props";
/** CfnResourceNode defines an L1 cdk resource */
class CfnResourceNode extends Node {
    /** Indicates if a node is a {@link CfnResourceNode} */
    static isCfnResourceNode(node) {
        return node.nodeType === types_1.NodeTypeEnum.CFN_RESOURCE;
    }
    constructor(props) {
        super({
            nodeType: types_1.NodeTypeEnum.CFN_RESOURCE,
            ...props,
        });
        // All cfn resource must have cfnType, but imported cfnType is inferred so might be missing
        if (this.cfnType == null && !this.hasFlag(types_1.FlagEnum.IMPORT)) {
            throw new Error("CfnResourceNode requires `cfnType` property");
        }
        if (props.importArnToken) {
            this.setAttribute(CfnResourceNode.ATT_IMPORT_ARN_TOKEN, props.importArnToken);
        }
        if (this.hasAttribute(CfnResourceNode.ATT_IMPORT_ARN_TOKEN)) {
            this.store.recordImportArn(this.getAttribute(CfnResourceNode.ATT_IMPORT_ARN_TOKEN), this);
        }
        this._resource = this._findNearestResource();
    }
    /** @inheritdoc */
    get isExtraneous() {
        return super.isExtraneous || this.resource != null;
    }
    /** Indicates if this CfnResource is imported (eg: `s3.Bucket.fromBucketArn`) */
    get isImport() {
        return this.hasFlag(types_1.FlagEnum.IMPORT);
    }
    /** Reference to the L2 Resource that wraps this L1 CfnResource if it is wrapped. */
    get resource() {
        return this._resource;
    }
    /**
     * Evaluates if CfnResourceNode fqn is equivalent to ResourceNode fqn.
     * @example `aws-cdk-lib.aws_lambda.Function` => `aws-cdk-lib.aws_lambda.CfnFunction`
     * @param resource - {@link Graph.ResourceNode} to compare
     * @returns Returns `true` if equivalent, otherwise `false`
     */
    isEquivalentFqn(resource) {
        const resourceFqnStub = resource.constructInfoFqn
            ?.split(".")
            .pop()
            ?.toLowerCase();
        const cfnResourceFqnStub = this.constructInfoFqn
            ?.split(".")
            .pop()
            ?.toLowerCase();
        if (!resourceFqnStub || !cfnResourceFqnStub) {
            return false;
        }
        return `cfn${resourceFqnStub}` === cfnResourceFqnStub;
    }
    /**
     * Finds the near *ancestor* that is a {@link ResourceNode}
     * @internal
     */
    _findNearestResource() {
        return this.scopes
            .slice()
            .reverse()
            .find((scope) => ResourceNode.isResourceNode(scope));
    }
    /**
     * @inheritdoc
     */
    mutateDestroy(strict) {
        const resource = this.resource;
        if (resource?.cfnResource === this) {
            resource.setAttribute(ResourceNode.ATT_WRAPPED_CFN_TYPE, this.cfnType);
            resource.setAttribute(ResourceNode.ATT_WRAPPED_CFN_PROPS, this.cfnProps);
            resource.mutateCfnResource(undefined);
        }
        super.mutateDestroy(strict);
    }
}
exports.CfnResourceNode = CfnResourceNode;
_k = JSII_RTTI_SYMBOL_1;
CfnResourceNode[_k] = { fqn: "@aws/pdk.cdk_graph.CfnResourceNode", version: "0.23.61" };
/** Normalized CfnReference attribute */
CfnResourceNode.ATT_IMPORT_ARN_TOKEN = "graph:cfn-resource:import-arn-token";
/** OutputNode defines a cdk CfnOutput resources */
class OutputNode extends Node {
    /** Indicates if node is an {@link OutputNode} */
    static isOutputNode(node) {
        return node.nodeType === types_1.NodeTypeEnum.OUTPUT;
    }
    constructor(props) {
        super({
            ...props,
            nodeType: types_1.NodeTypeEnum.OUTPUT,
        });
        /** Indicates if {@link OutputNode} is **exported** */
        this.isExport = false;
        if (this.stack == null) {
            throw new Error(`OutputNode instantiated outside of stack: ${this}`);
        }
        this.addFlag(types_1.FlagEnum.EXTRANEOUS);
        this.setAttribute(OutputNode.ATTR_VALUE, props.value);
        if (props.exportName) {
            this.isExport = true;
            this.setAttribute(OutputNode.ATTR_EXPORT_NAME, props.exportName);
        }
        props.description && this.setAttribute("description", props.description);
        this.stack.addOutput(this);
    }
    /** Get the *value** attribute */
    get value() {
        return this.getAttribute(OutputNode.ATTR_VALUE);
    }
    /** Get the export name attribute */
    get exportName() {
        return this.getAttribute(OutputNode.ATTR_EXPORT_NAME);
    }
    /** @inheritdoc */
    mutateDestroy(strict) {
        super.mutateDestroy(strict);
        this.stack?.mutateRemoveOutput(this);
    }
}
exports.OutputNode = OutputNode;
_l = JSII_RTTI_SYMBOL_1;
OutputNode[_l] = { fqn: "@aws/pdk.cdk_graph.OutputNode", version: "0.23.61" };
/** Attribute key where output value is stored */
OutputNode.ATTR_VALUE = "graph:output:value";
/** Attribute key where output export name is stored */
OutputNode.ATTR_EXPORT_NAME = "graph:output:export-name";
/** ParameterNode defines a CfnParameter node */
class ParameterNode extends Node {
    /** Indicates if node is a {@link ParameterNode} */
    static isParameterNode(node) {
        return node.nodeType === types_1.NodeTypeEnum.PARAMETER;
    }
    constructor(props) {
        super({
            ...props,
            nodeType: types_1.NodeTypeEnum.PARAMETER,
        });
        if (this.stack == null) {
            throw new Error(`ParameterNode instantiated outside of stack: ${this}`);
        }
        this.addFlag(types_1.FlagEnum.EXTRANEOUS);
        this.setAttribute(ParameterNode.ATTR_VALUE, props.value);
        this.setAttribute(ParameterNode.ATTR_TYPE, props.parameterType);
        props.description && this.setAttribute("description", props.description);
        this.isStackReference = this.id.startsWith("reference-to-");
        this.stack.addParameter(this);
    }
    /** Get the value attribute */
    get value() {
        return this.getAttribute(ParameterNode.ATTR_VALUE);
    }
    /** Get the parameter type attribute */
    get parameterType() {
        return this.getAttribute(ParameterNode.ATTR_TYPE);
    }
    /** @inheritdoc */
    mutateDestroy(strict) {
        super.mutateDestroy(strict);
        this.stack?.mutateRemoveParameter(this);
    }
}
exports.ParameterNode = ParameterNode;
_m = JSII_RTTI_SYMBOL_1;
ParameterNode[_m] = { fqn: "@aws/pdk.cdk_graph.ParameterNode", version: "0.23.61" };
/** Attribute key where parameter value is store */
ParameterNode.ATTR_VALUE = "graph:parameter:value";
/** Attribute key where parameter type is stored */
ParameterNode.ATTR_TYPE = "graph:parameter:type";
/** StackNode defines a cdk Stack */
class StackNode extends Node {
    /** Indicates if node is a {@link StackNode} */
    static isStackNode(node) {
        return node.nodeType === types_1.NodeTypeEnum.STACK;
    }
    /**
     * Gets the {@link StackNode} containing a given resource
     * @throws Error is node is not contained in a stack
     */
    static of(node) {
        const stack = node.stack;
        if (stack == null) {
            throw new Error(`${String(node)} is not within StackNode`);
        }
        return stack;
    }
    /** Get {@link StageNode} containing this stack */
    get stage() {
        return this._stage;
    }
    /** Get all {@link OutputNode}s defined by this stack */
    get outputs() {
        return Array.from(this._outputs);
    }
    /** Get all {@link ParameterNode}s defined by this stack */
    get parameters() {
        return Array.from(this._parameters);
    }
    constructor(props) {
        super({
            nodeType: types_1.NodeTypeEnum.STACK,
            ...props,
        });
        /** @internal */
        this._outputs = new Set();
        /** @internal */
        this._parameters = new Set();
        if (this.stack !== this) {
            throw new Error(`Stack.stack is not self: ${this.uuid}`);
        }
        this.addFlag(types_1.FlagEnum.CLUSTER);
        this.store.addStack(this);
        const stage = this.findAncestor({
            filter: StageNode.isStageNode,
        });
        if (stage) {
            this._stage = stage;
            stage.addStack(this);
        }
    }
    /** Get all **exported** {@link OutputNode}s defined by this stack */
    get exports() {
        return this.outputs.filter((node) => node.isExport);
    }
    /** Associate {@link OutputNode} with this stack */
    addOutput(node) {
        this._outputs.add(node);
    }
    /**
     * Find {@link OutputNode} with *logicalId* defined by this stack
     * @throws Error is no output found matching *logicalId*
     */
    findOutput(logicalId) {
        const output = this.outputs.find((_output) => _output.logicalId === logicalId);
        if (output == null) {
            console.debug(`${this}.Outputs: [logicalId]`, this.outputs.map((n) => n.logicalId));
            throw new Error(`Output ${logicalId} does not exist in ${this}`);
        }
        return output;
    }
    /** Associate {@link ParameterNode} with this stack */
    addParameter(node) {
        this._parameters.add(node);
    }
    /**
     * Find {@link ParameterNode} with *parameterId* defined by this stack
     * @throws Error is no parameter found matching *parameterId*
     */
    findParameter(parameterId) {
        const parameter = this.parameters.find((_parameter) => _parameter.id === parameterId);
        if (parameter == null) {
            console.debug(`${this}.Parameters: [id]`, this.parameters.map((n) => n.id));
            throw new Error(`Parameter ${parameterId} does not exist in ${this}`);
        }
        return parameter;
    }
    /**
     * Disassociate {@link OutputNode} from this stack
     * @destructive
     */
    mutateRemoveOutput(node) {
        this._preMutate();
        return this._outputs.delete(node);
    }
    /**
     * Disassociate {@link ParameterNode} from this stack
     * @destructive
     */
    mutateRemoveParameter(node) {
        this._preMutate();
        return this._parameters.delete(node);
    }
    /** @inheritdoc */
    mutateDestroy(strict) {
        super.mutateDestroy(strict);
        this.stage?.mutateRemoveStack(this);
    }
    /** @inheritdoc */
    mutateHoist(newParent) {
        super.mutateHoist(newParent);
        if (this.stage && this.isAncestor(this.stage)) {
            this.stage.mutateRemoveStack(this);
            this._stage = this.findAncestor({
                filter: (node) => StageNode.isStageNode(node),
            });
            if (this._stage) {
                this._stage.addStack(this);
            }
        }
    }
}
exports.StackNode = StackNode;
_o = JSII_RTTI_SYMBOL_1;
StackNode[_o] = { fqn: "@aws/pdk.cdk_graph.StackNode", version: "0.23.61" };
/** NestedStackNode defines a cdk NestedStack */
class NestedStackNode extends StackNode {
    /** Indicates if node is a {@link NestedStackNode} */
    static isNestedStackNode(node) {
        return node.nodeType === types_1.NodeTypeEnum.NESTED_STACK;
    }
    /** Get parent stack of this nested stack */
    get parentStack() {
        return this._parentStack;
    }
    constructor(props) {
        super({
            ...props,
            nodeType: types_1.NodeTypeEnum.NESTED_STACK,
        });
        this._parentStack = props.parentStack;
    }
    /** @inheritdoc */
    mutateHoist(newParent) {
        super.mutateHoist(newParent);
        if (this.parentStack && this.isAncestor(this.parentStack)) {
            this._parentStack = this.findAncestor({
                filter: (node) => StackNode.isStackNode(node),
            });
        }
    }
}
exports.NestedStackNode = NestedStackNode;
_p = JSII_RTTI_SYMBOL_1;
NestedStackNode[_p] = { fqn: "@aws/pdk.cdk_graph.NestedStackNode", version: "0.23.61" };
/** StageNode defines a cdk Stage */
class StageNode extends Node {
    /** Indicates if node is a {@link StageNode} */
    static isStageNode(node) {
        return node.nodeType === types_1.NodeTypeEnum.STAGE;
    }
    /**
     * Gets the {@link StageNode} containing a given resource
     * @throws Error is node is not contained in a stage
     */
    static of(node) {
        const stage = node.rootStack?.stage;
        if (stage == null) {
            throw new Error(`${node} is not within a stage`);
        }
        return stage;
    }
    /** Gets all stacks contained by this stage */
    get stacks() {
        return Array.from(this._stacks);
    }
    constructor(props) {
        super({
            ...props,
            nodeType: types_1.NodeTypeEnum.STAGE,
        });
        /** @internal */
        this._stacks = new Set();
        this.store.addStage(this);
        this.addFlag(types_1.FlagEnum.CLUSTER);
    }
    /** Associate a {@link StackNode} with this stage */
    addStack(stack) {
        this._stacks.add(stack);
    }
    /**
     * Disassociate {@link StackNode} from this stage
     * @destructive
     */
    mutateRemoveStack(stack) {
        this._preMutate();
        return this._stacks.delete(stack);
    }
}
exports.StageNode = StageNode;
_q = JSII_RTTI_SYMBOL_1;
StageNode[_q] = { fqn: "@aws/pdk.cdk_graph.StageNode", version: "0.23.61" };
/** AppNode defines a cdk App */
class AppNode extends Node {
    /** Indicates if node is a {@link AppNode} */
    static isAppNode(node) {
        return node.nodeType === types_1.NodeTypeEnum.APP;
    }
    constructor(props) {
        super({
            ...props,
            nodeType: types_1.NodeTypeEnum.APP,
            uuid: AppNode.UUID,
            id: AppNode.UUID,
            path: AppNode.PATH,
        });
        this.addFlag(types_1.FlagEnum.GRAPH_CONTAINER);
        this.addFlag(types_1.FlagEnum.CLUSTER);
    }
}
exports.AppNode = AppNode;
_r = JSII_RTTI_SYMBOL_1;
AppNode[_r] = { fqn: "@aws/pdk.cdk_graph.AppNode", version: "0.23.61" };
/** Fixed UUID for App node */
AppNode.UUID = "App";
/** Fixed path of the App  */
AppNode.PATH = "/";
/** RootNode represents the root of the store tree */
class RootNode extends Node {
    /** Indicates if node is a {@link RootNode} */
    static isRootNode(node) {
        return node.nodeType === types_1.NodeTypeEnum.ROOT;
    }
    constructor(store) {
        super({
            store,
            nodeType: types_1.NodeTypeEnum.ROOT,
            uuid: RootNode.UUID,
            id: RootNode.UUID,
            path: RootNode.PATH,
        });
        this.addFlag(types_1.FlagEnum.GRAPH_CONTAINER);
        this.addFlag(types_1.FlagEnum.CLUSTER);
    }
    /**
     * @inheritdoc **The root not is excluded from list**
     */
    findAll(options) {
        return super.findAll(options);
    }
    /**
     * > {@link RootNode} does not support this mutation
     * @throws Error does not support
     * @inheritdoc
     */
    mutateCollapse() {
        throw new Error("Root node can not be collapsed");
    }
    /**
     * > {@link RootNode} does not support this mutation
     * @throws Error does not support
     * @inheritdoc
     */
    mutateCollapseToParent() {
        throw new Error("Root node can not be collapsed to parent");
    }
    /**
     * > {@link RootNode} does not support this mutation
     * @throws Error does not support
     * @inheritdoc
     */
    mutateCollapseTo(_ancestor) {
        throw new Error("Root node can not be collapsed");
    }
    /**
     * > {@link RootNode} does not support this mutation
     * @throws Error does not support
     * @inheritdoc
     */
    mutateDestroy(_strict = false) {
        throw new Error("Root node can not be destroyed");
    }
    /**
     * > {@link RootNode} does not support this mutation
     * @throws Error does not support
     * @inheritdoc
     */
    mutateHoist(_newParent) {
        throw new Error("Root node can not be hoisted");
    }
}
exports.RootNode = RootNode;
_s = JSII_RTTI_SYMBOL_1;
RootNode[_s] = { fqn: "@aws/pdk.cdk_graph.RootNode", version: "0.23.61" };
/** Fixed UUID of root */
RootNode.UUID = "Root";
/** Fixed path of root */
RootNode.PATH = "";
function isResourceLike(node) {
    switch (node.nodeType) {
        case types_1.NodeTypeEnum.RESOURCE:
        case types_1.NodeTypeEnum.CFN_RESOURCE: {
            return true;
        }
        default: {
            return false;
        }
    }
}
exports.isResourceLike = isResourceLike;
/**
 * Deserializes a *serialized store object* into an *in-memory store instance**.
 * @param serializedStore - The serialized store to deserialize
 * @param allowDestructiveMutations - Indicates if the store instance allows destructive mutations.
 * @throws Error if a serialized node's parent does not match visitor parent
 * @throws Error if a serialized node type deserialization mapping is not defined
 * @throws Error if edge type deserialization mapping is not defined
 */
function deserializeStore(serializedStore, allowDestructiveMutations = false) {
    const store = new Store(allowDestructiveMutations);
    // TODO: ensure store versions are compatible
    function visit(sNode, parent) {
        const nodeProps = {
            ...omit(sNode, [
                "children",
                "parent",
                "stack",
                "nodeType",
            ]),
            parent: sNode.parent ? store.getNode(sNode.parent) : undefined,
            // resolve stack node, unless stack is itself
            stack: sNode.stack && sNode.stack !== sNode.uuid
                ? store.getStack(sNode.stack)
                : undefined,
            store,
        };
        if (nodeProps.parent?.uuid !== parent.uuid) {
            throw new Error(`SerializedNode parent ${sNode.parent} does not match visitor parent ${parent.uuid}`);
        }
        let node = undefined;
        switch (sNode.nodeType) {
            case types_1.NodeTypeEnum.APP: {
                node = new AppNode({
                    ...nodeProps,
                    parent,
                });
                break;
            }
            case types_1.NodeTypeEnum.STAGE: {
                node = new StageNode(nodeProps);
                break;
            }
            case types_1.NodeTypeEnum.STACK: {
                node = new StackNode(nodeProps);
                break;
            }
            case types_1.NodeTypeEnum.NESTED_STACK: {
                node = new NestedStackNode({
                    ...nodeProps,
                    parentStack: StackNode.of(parent),
                });
                break;
            }
            case types_1.NodeTypeEnum.OUTPUT: {
                node = new OutputNode({
                    ...nodeProps,
                    value: nodeProps.attributes[OutputNode.ATTR_VALUE],
                    exportName: nodeProps.attributes[OutputNode.ATTR_EXPORT_NAME],
                    description: nodeProps.attributes.description,
                });
                break;
            }
            case types_1.NodeTypeEnum.PARAMETER: {
                node = new ParameterNode({
                    ...nodeProps,
                    value: nodeProps.attributes[ParameterNode.ATTR_VALUE],
                    parameterType: nodeProps.attributes[ParameterNode.ATTR_TYPE],
                    description: nodeProps.attributes.description,
                });
                break;
            }
            case types_1.NodeTypeEnum.CFN_RESOURCE: {
                node = new CfnResourceNode(nodeProps);
                break;
            }
            case types_1.NodeTypeEnum.RESOURCE: {
                node = new ResourceNode({
                    ...nodeProps,
                    cdkOwned: !!nodeProps.flags?.includes(types_1.FlagEnum.CDK_OWNED),
                });
                break;
            }
            case types_1.NodeTypeEnum.DEFAULT: {
                node = new Node({
                    ...nodeProps,
                    nodeType: types_1.NodeTypeEnum.DEFAULT,
                });
                break;
            }
        }
        if (node == null) {
            console.debug(sNode.nodeType, sNode);
            throw new Error(`NodeType ${sNode.nodeType} missing deserialization mapping`);
        }
        // ensure node is registered in store
        (0, console_1.assert)(store.getNode(sNode.uuid) === node, `Node ${sNode.uuid} did not register in store`);
        Object.values(sNode.children || {}).forEach((sChild) => {
            visit(sChild, node);
        });
    }
    Object.values(serializedStore.tree.children || {}).forEach((sNode) => {
        visit(sNode, store.root);
    });
    serializedStore.edges.forEach((sEdge) => {
        const edgeProps = {
            ...sEdge,
            store,
            source: store.getNode(sEdge.source),
            target: store.getNode(sEdge.target),
        };
        switch (sEdge.edgeType) {
            case types_1.EdgeTypeEnum.DEPENDENCY: {
                new Dependency({
                    ...edgeProps,
                });
                break;
            }
            case types_1.EdgeTypeEnum.REFERENCE: {
                const referenceType = sEdge.attributes[Reference.ATT_TYPE];
                if (referenceType === types_1.ReferenceTypeEnum.ATTRIBUTE) {
                    new AttributeReference({
                        ...edgeProps,
                        value: sEdge.attributes[AttributeReference.ATT_VALUE],
                    });
                }
                else if (referenceType === types_1.ReferenceTypeEnum.IMPORT) {
                    new ImportReference({
                        ...edgeProps,
                    });
                }
                else if (referenceType === types_1.ReferenceTypeEnum.REF) {
                    new Reference(edgeProps);
                }
                else {
                    throw new Error(`Unsupported reference type of ${referenceType}`);
                }
                break;
            }
            default: {
                // TODO: support custom edge types explicitly
                new Edge(edgeProps);
            }
        }
    });
    return store;
}
exports.deserializeStore = deserializeStore;
//# sourceMappingURL=data:application/json;base64,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