"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.compactFilterPreset = exports.nonExtraneousFilterPreset = exports.focusFilter = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const filters_1 = require("./filters");
/**
 * Focus the graph on a specific node.
 * @throws Error if store is not filterable
 * @destructive
 */
function focusFilter(store, plan) {
    filters_1.Filters.verifyFilterable(store);
    if (plan.focus == null)
        return; // noop
    let focusedNode;
    let hoist = true;
    const { filter: _filter, noHoist: _noHoist } = plan.focus;
    focusedNode = _filter.filter(store);
    hoist = !_noHoist;
    if (focusedNode === store.root)
        return; // noop
    const ancestors = focusedNode.scopes.slice();
    // remove the actual store.root from ancestors (we can't destroy that)
    const rootAncestor = ancestors.shift();
    if (rootAncestor !== store.root) {
        throw new Error(`${focusedNode} is not within the store root graph: it has root of ${rootAncestor}`);
    }
    if (hoist) {
        // Move focused node as direct child of store root and prune all other ancestors
        focusedNode.mutateHoist(store.root);
        // Only need to destroy to first non-root ancestor to prune the ancestral tree
        if (ancestors.length) {
            ancestors[0].mutateDestroy();
        }
        // prune all other root children (unless preserved)
        store.root.children.forEach((child) => {
            if (child !== focusedNode) {
                child.mutateDestroy();
            }
        });
    }
    else {
        // keep the focused node in place, but prune non-direct ancestor chain nodes
        // the direct ancestor chain is only the nodes scopes
        ancestors.reverse().forEach((ancestor) => {
            ancestor.siblings.forEach((ancestorSibling) => {
                ancestorSibling.mutateDestroy();
            });
        });
        // prune all planRoot siblings
        focusedNode.siblings.forEach((sibling) => {
            sibling.mutateDestroy();
        });
    }
}
exports.focusFilter = focusFilter;
/**
 * Preset that performs {@link Filters.pruneExtraneous} filter on store.
 * @throws Error if store is not filterable
 * @destructive
 */
function nonExtraneousFilterPreset(store) {
    return filters_1.Filters.pruneExtraneous().filter(store);
}
exports.nonExtraneousFilterPreset = nonExtraneousFilterPreset;
/**
 * Preset that performs {@link Filters.compact} filter on store.
 * @throws Error if store is not filterable
 * @destructive
 */
function compactFilterPreset(store) {
    filters_1.Filters.compact().filter(store);
}
exports.compactFilterPreset = compactFilterPreset;
//# sourceMappingURL=data:application/json;base64,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