"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InfrastructureJavaProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const monorepo_1 = require("../../../monorepo");
const Mustache = require("mustache");
const projen_1 = require("projen");
const awscdk_1 = require("projen/lib/awscdk");
const infrastructure_commands_1 = require("../../components/infrastructure-commands");
const consts_1 = require("../../consts");
/**
 * Synthesizes a Infrastructure Java Project.
 */
class InfrastructureJavaProject extends awscdk_1.AwsCdkJavaApp {
    constructor(options) {
        const groupId = options.groupId ?? "software.aws.infra";
        const artifactId = options.artifactId ?? "infra";
        super({
            ...options,
            cdkVersion: options.cdkVersion ?? "2.133.0",
            cdkVersionPinning: true,
            sample: false,
            junit: false,
            groupId,
            artifactId,
            mainClass: `${groupId}.Main`,
            version: options.version ?? "0.0.0",
            name: options.name,
            readme: {
                contents: fs
                    .readFileSync(path.resolve(__dirname, "../../../samples/infrastructure/java/README.md"))
                    .toString(),
            },
        });
        // Pin constructs version
        this.deps.removeDependency("software.constructs/constructs", projen_1.DependencyType.RUNTIME);
        [
            "software.constructs/constructs@10.3.0",
            "io.github.cdklabs/cdknag@2.15.16",
        ].forEach((d) => this.addDependency(d));
        infrastructure_commands_1.InfrastructureCommands.ensure(this);
        this.pom.addPlugin("org.apache.maven.plugins/maven-surefire-plugin@3.1.2");
        this.pom.addPlugin("org.apache.maven.plugins/maven-compiler-plugin@3.8.1", {
            configuration: {
                release: "11",
            },
        });
        if (options.junit !== false) {
            [
                "org.junit.jupiter/junit-jupiter-api@5.10.2",
                "org.junit.jupiter/junit-jupiter-engine@5.10.2",
                "io.github.origin-energy/java-snapshot-testing-junit5@4.0.7",
                "io.github.origin-energy/java-snapshot-testing-plugin-jackson@4.0.7",
                "org.slf4j/slf4j-simple@2.0.0-alpha0",
            ].forEach((d) => this.addTestDependency(d));
            this.testTask.exec("mvn test");
        }
        this.addDependency(`software.aws/pdk@${monorepo_1.ProjectUtils.getPdkVersion()}`);
        const srcDir = path.resolve(__dirname, "../../../samples/infrastructure/java/src");
        const testDir = path.resolve(__dirname, "../../../samples/infrastructure/java/test");
        const typeSafeApis = [
            ...(options.typeSafeApis || []),
            ...(options.typeSafeApi ? [options.typeSafeApi] : []),
        ];
        const cloudscapeReactTsWebsites = [
            ...(options.cloudscapeReactTsWebsites || []),
            ...(options.cloudscapeReactTsWebsite
                ? [options.cloudscapeReactTsWebsite]
                : []),
        ];
        typeSafeApis.forEach((tsApi) => {
            if (!tsApi.infrastructure.java) {
                throw new Error("Cannot pass in a Type Safe Api without Java Infrastructure configured!");
            }
            monorepo_1.NxProject.ensure(this).addJavaDependency(tsApi.infrastructure.java);
            // Ensure handlers are built before infra
            tsApi.all.handlers?.forEach((handler) => {
                monorepo_1.NxProject.ensure(this).addImplicitDependency(handler);
            });
        });
        cloudscapeReactTsWebsites.forEach((csWebsite) => {
            // Ensure website is built before infra
            monorepo_1.NxProject.ensure(this).addImplicitDependency(csWebsite);
        });
        const mustacheConfig = {
            stackName: options.stackName || consts_1.DEFAULT_STACK_NAME,
            allowSignup: options.allowSignup ?? false,
            groupId,
            stages: options.stages || [],
            typeSafeApis: this.generateTypeSafeMustacheConfig(groupId, typeSafeApis),
            cloudscapeReactTsWebsites: cloudscapeReactTsWebsites.map((csWebsite) => {
                const websiteName = this.capitalize(csWebsite.package.packageName
                    .replace(/[^a-z0-9_]+/gi, "")
                    .replace(/^[0-9]+/gi, ""));
                return {
                    websiteName,
                    websiteNameLowercase: websiteName.toLowerCase(),
                    groupId,
                    websiteDistRelativePath: path.relative(this.outdir, `${csWebsite.outdir}/build`),
                    typeSafeApis: this.generateTypeSafeMustacheConfig(groupId, csWebsite.typeSafeApis),
                };
            }),
        };
        options.sample !== false &&
            this.emitSampleFiles(srcDir, ["src", "main"], mustacheConfig);
        options.sample !== false &&
            this.emitSampleFiles(testDir, ["src", "test"], mustacheConfig);
    }
    generateTypeSafeMustacheConfig(groupId, typeSafeApis) {
        return typeSafeApis?.map((tsApi, idx) => {
            const apiName = this.capitalize(tsApi.model
                .apiName.replace(/[^a-z0-9_]+/gi, "")
                .replace(/^[0-9]+/gi, ""));
            return {
                apiName,
                apiNameLowercase: apiName?.toLowerCase(),
                groupId,
                infraPackage: `${tsApi?.infrastructure.java?.pom.groupId}.${tsApi?.infrastructure.java?.pom.name}.infra`,
                isLast: idx === typeSafeApis.length - 1,
            };
        });
    }
    capitalize(word) {
        return word.charAt(0).toUpperCase() + word.slice(1);
    }
    emitSampleFiles(dir, pathPrefixes = [], mustacheConfig) {
        fs.readdirSync(dir, { withFileTypes: true }).forEach((f) => {
            if (f.isDirectory()) {
                return this.emitSampleFiles(`${dir}/${f.name}`, [
                    ...pathPrefixes,
                    ...(f.name === "groupId"
                        ? mustacheConfig.groupId.split(".")
                        : [f.name]),
                ], mustacheConfig);
            }
            else if (f.name.endsWith("ApiConstruct.java.mustache")) {
                mustacheConfig.typeSafeApis.forEach((tsApi) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${tsApi.apiName}.java`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), tsApi),
                    });
                });
            }
            else if (f.name.endsWith("WebsiteConstruct.java.mustache")) {
                mustacheConfig.cloudscapeReactTsWebsites.forEach((csWebsite) => {
                    new projen_1.SampleFile(this, `${path.join(...pathPrefixes, `${csWebsite.websiteName}.java`)}`, {
                        contents: Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), csWebsite),
                    });
                });
            }
            else {
                const contents = Mustache.render(fs.readFileSync(`${dir}/${f.name}`).toString(), mustacheConfig);
                return new projen_1.SampleFile(this, `${path.join(...pathPrefixes, f.name.replace(".mustache", ""))}`, {
                    contents,
                    sourcePath: (!contents && `${dir}/${f.name}`) || undefined,
                });
            }
        });
    }
}
exports.InfrastructureJavaProject = InfrastructureJavaProject;
_a = JSII_RTTI_SYMBOL_1;
InfrastructureJavaProject[_a] = { fqn: "@aws/pdk.infrastructure.InfrastructureJavaProject", version: "0.23.61" };
//# sourceMappingURL=data:application/json;base64,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