"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MonorepoJavaProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const java_1 = require("projen/lib/java");
const nx_configurator_1 = require("../../components/nx-configurator");
const projen_dependency_1 = require("../../components/projen-dependency");
const utils_1 = require("../../utils");
const MVN_PLUGIN_PATH = "./.nx/plugins/nx_plugin.js";
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid monorepo-java
 */
class MonorepoJavaProject extends java_1.JavaProject {
    constructor(options) {
        // Pin default projen version for java
        const projenVersion = options.projenrcJavaOptions?.projenVersion ??
            projen_dependency_1.DEFAULT_PROJEN_VERSION.replace(/^\^/, "");
        super({
            ...options,
            sample: false,
            junit: false,
            version: options.version ?? "0.0.0",
            groupId: options.groupId ?? "com.aws",
            artifactId: options.artifactId ?? "monorepo",
            projenrcJavaOptions: {
                ...options.projenrcJavaOptions,
                projenVersion,
            },
        });
        this.projenVersion = projenVersion;
        // Pin constructs version
        this.deps.removeDependency("software.constructs/constructs", projen_1.DependencyType.BUILD);
        this.addDependency("software.constructs/constructs@10.3.0");
        this.addTestDependency(`software.aws/pdk@${utils_1.ProjectUtils.getPdkVersion()}`);
        this.nxConfigurator = new nx_configurator_1.NxConfigurator(this, {
            defaultReleaseBranch: options.defaultReleaseBranch ?? "main",
            licenseOptions: options.disableDefaultLicenses
                ? {
                    disableDefaultLicenses: options.disableDefaultLicenses,
                }
                : undefined,
        });
        // Setup maven nx plugin
        new projen_1.TextFile(this, MVN_PLUGIN_PATH, {
            readonly: true,
            lines: fs
                .readFileSync(path.join(__dirname, "plugin/mvn_plugin.js"))
                .toString("utf-8")
                .split("\n"),
        });
        this.nx.plugins.push("@jnxplus/nx-maven", MVN_PLUGIN_PATH);
        this.installTask = this.nxConfigurator.ensureNxInstallTask({
            "@jnxplus/nx-maven": "^1.x",
        });
        // Map tasks to nx run-many
        this.nxConfigurator._overrideNxBuildTask(this.buildTask, { target: "build" }, { force: true });
        this.nxConfigurator._overrideNxBuildTask(this.preCompileTask, {
            target: "pre-compile",
        });
        this.nxConfigurator._overrideNxBuildTask(this.compileTask, {
            target: "compile",
        });
        this.nxConfigurator._overrideNxBuildTask(this.postCompileTask, {
            target: "post-compile",
        });
        this.nxConfigurator._overrideNxBuildTask(this.testTask, {
            target: "test",
        });
        this.nxConfigurator._overrideNxBuildTask(this.packageTask, {
            target: "package",
        });
    }
    /**
     * @inheritdoc
     */
    get nx() {
        return this.nxConfigurator.nx;
    }
    /**
     * @inheritdoc
     */
    execNxRunManyCommand(options) {
        return this.nxConfigurator.execNxRunManyCommand(options);
    }
    /**
     * @inheritdoc
     */
    composeNxRunManyCommand(options) {
        return this.nxConfigurator.composeNxRunManyCommand(options);
    }
    /**
     * @inheritdoc
     */
    addNxRunManyTask(name, options) {
        return this.nxConfigurator.addNxRunManyTask(name, options);
    }
    /**
     * @inheritdoc
     */
    addImplicitDependency(dependent, dependee) {
        this.nxConfigurator.addImplicitDependency(dependent, dependee);
    }
    /**
     * @inheritdoc
     */
    addJavaDependency(dependent, dependee) {
        this.nxConfigurator.addJavaDependency(dependent, dependee);
    }
    /**
     * @inheritdoc
     */
    addPythonPoetryDependency(dependent, dependee) {
        this.nxConfigurator.addPythonPoetryDependency(dependent, dependee);
    }
    addUpgradeDepsTask() {
        const upgradeDepsTask = new projen_1.Task("upgrade-deps", {
            description: "Upgrade dependencies in the monorepo",
        });
        this.nxConfigurator._overrideNxBuildTask(upgradeDepsTask, {
            target: "upgrade",
        });
        this.nxConfigurator._configurePythonSubprojectUpgradeDeps(this, upgradeDepsTask);
    }
    /**
     * @inheritdoc
     */
    preSynthesize() {
        // Calling before super() to ensure proper pre-synth of NxProject component and its nested components
        this.nxConfigurator.preSynthesize();
        this.addUpgradeDepsTask();
        super.preSynthesize();
        (0, projen_dependency_1.syncProjenVersions)(this.subprojects, this.projenVersion);
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.nxConfigurator.synth();
        super.synth();
    }
    postSynthesize() {
        super.postSynthesize();
        this.installNx();
    }
    /**
     * Run the install task which will install nx locally
     */
    installNx() {
        this.logger.info("Installing dependencies...");
        const runtime = new projen_1.TaskRuntime(this.outdir);
        runtime.runTask(this.installTask.name);
    }
}
exports.MonorepoJavaProject = MonorepoJavaProject;
_a = JSII_RTTI_SYMBOL_1;
MonorepoJavaProject[_a] = { fqn: "@aws/pdk.monorepo.MonorepoJavaProject", version: "0.23.61" };
//# sourceMappingURL=data:application/json;base64,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