"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.satisfiesPackageManifest = void 0;
const types_1 = require("@pnpm/types");
const equals_1 = __importDefault(require("ramda/src/equals"));
const pickBy_1 = __importDefault(require("ramda/src/pickBy"));
const omit_1 = __importDefault(require("ramda/src/omit"));
function satisfiesPackageManifest(opts, importer, pkg) {
    if (!importer)
        return { satisfies: false, detailedReason: 'no importer' };
    let existingDeps = { ...pkg.devDependencies, ...pkg.dependencies, ...pkg.optionalDependencies };
    if (opts?.autoInstallPeers) {
        pkg = {
            ...pkg,
            dependencies: {
                ...(0, omit_1.default)(Object.keys(existingDeps), pkg.peerDependencies),
                ...pkg.dependencies,
            },
        };
        existingDeps = {
            ...pkg.peerDependencies,
            ...existingDeps,
        };
    }
    const pickNonLinkedDeps = (0, pickBy_1.default)((spec) => !spec.startsWith('link:'));
    let specs = importer.specifiers;
    if (opts?.excludeLinksFromLockfile) {
        existingDeps = pickNonLinkedDeps(existingDeps);
        specs = pickNonLinkedDeps(specs);
    }
    if (!(0, equals_1.default)(existingDeps, specs)) {
        return {
            satisfies: false,
            detailedReason: `specifiers in the lockfile (${JSON.stringify(specs)}) don't match specs in package.json (${JSON.stringify(existingDeps)})`,
        };
    }
    if (importer.publishDirectory !== pkg.publishConfig?.directory) {
        return {
            satisfies: false,
            detailedReason: `"publishDirectory" in the lockfile (${importer.publishDirectory ?? 'undefined'}) doesn't match "publishConfig.directory" in package.json (${pkg.publishConfig?.directory ?? 'undefined'})`,
        };
    }
    if (!(0, equals_1.default)(pkg.dependenciesMeta ?? {}, importer.dependenciesMeta ?? {})) {
        return {
            satisfies: false,
            detailedReason: `importer dependencies meta (${JSON.stringify(importer.dependenciesMeta)}) doesn't match package manifest dependencies meta (${JSON.stringify(pkg.dependenciesMeta)})`,
        };
    }
    for (const depField of types_1.DEPENDENCIES_FIELDS) {
        const importerDeps = importer[depField] ?? {};
        let pkgDeps = pkg[depField] ?? {};
        if (opts?.excludeLinksFromLockfile) {
            pkgDeps = pickNonLinkedDeps(pkgDeps);
        }
        let pkgDepNames;
        switch (depField) {
            case 'optionalDependencies':
                pkgDepNames = Object.keys(pkgDeps);
                break;
            case 'devDependencies':
                pkgDepNames = Object.keys(pkgDeps)
                    .filter((depName) => !pkg.optionalDependencies?.[depName] && !pkg.dependencies?.[depName]);
                break;
            case 'dependencies':
                pkgDepNames = Object.keys(pkgDeps)
                    .filter((depName) => !pkg.optionalDependencies?.[depName]);
                break;
            default:
                throw new Error(`Unknown dependency type "${depField}"`);
        }
        if (pkgDepNames.length !== Object.keys(importerDeps).length &&
            pkgDepNames.length !== countOfNonLinkedDeps(importerDeps)) {
            return {
                satisfies: false,
                detailedReason: `"${depField}" in the lockfile (${JSON.stringify(importerDeps)}) doesn't match the same field in package.json (${JSON.stringify(pkgDeps)})`,
            };
        }
        for (const depName of pkgDepNames) {
            if (!importerDeps[depName] || importer.specifiers?.[depName] !== pkgDeps[depName]) {
                return {
                    satisfies: false,
                    detailedReason: `importer ${depField}.${depName} specifier ${importer.specifiers[depName]} don't match package manifest specifier (${pkgDeps[depName]})`,
                };
            }
        }
    }
    return { satisfies: true };
}
exports.satisfiesPackageManifest = satisfiesPackageManifest;
function countOfNonLinkedDeps(lockfileDeps) {
    return Object.values(lockfileDeps).filter((ref) => !ref.includes('link:') && !ref.includes('file:')).length;
}
//# sourceMappingURL=satisfiesPackageManifest.js.map