"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __exportStar = (this && this.__exportStar) || function(m, exports) {
    for (var p in m) if (p !== "default" && !Object.prototype.hasOwnProperty.call(exports, p)) __createBinding(exports, m, p);
};
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDKPipeline = exports.DEFAULT_BRANCH_NAME = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codecommit_1 = require("aws-cdk-lib/aws-codecommit");
const aws_codepipeline_1 = require("aws-cdk-lib/aws-codepipeline");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const sonar_code_scanner_1 = require("./code_scanner/sonar-code-scanner");
const feature_branches_1 = require("./feature-branches");
__exportStar(require("./code_scanner/sonar-code-scanner"), exports);
exports.DEFAULT_BRANCH_NAME = "mainline";
/**
 * An extension to CodePipeline which configures sane defaults for a NX Monorepo
 * codebase. In addition to this, it also creates a CodeCommit repository with
 * automated PR builds and approvals.
 */
class PDKPipeline extends constructs_1.Construct {
    /**
     * A helper function to normalize the branch name with only alphanumeric characters and hypens ('-').
     * @param branchName The name of the branch to normalize.
     * @returns The normalized branch name.
     */
    static normalizeBranchName(branchName) {
        return branchName.replace(/[^a-zA-Z0-9-]/g, "-");
    }
    /**
     * A helper function to determine if the current branch is the default branch.
     *
     * If there is no BRANCH environment variable, then assume this is the default
     * branch. Otherwise, check that BRANCH matches the default branch name.
     *
     * The default branch name is determined in the following priority:
     *
     * 1. defaultBranchName property
     * 2. defaultBranchName context
     * 3. PDKPipeline.defaultBranchName constant
     *
     * @param props? {
     *    defaultBranchName? Specify the default branch name without context.
     *    node? The current app to fetch defaultBranchName from context.
     * }
     * @returns True if the current branch is the default branch.
     */
    static isDefaultBranch(props = {
        defaultBranchName: undefined,
        node: undefined,
    }) {
        if (!process.env.BRANCH) {
            return true;
        }
        const defaultBranchName = props.defaultBranchName ||
            (props.node && props.node.tryGetContext("defaultBranchName")) ||
            PDKPipeline.defaultBranchName;
        return defaultBranchName === process.env.BRANCH;
    }
    /**
     * A helper function to create a branch prefix. The prefix is empty on the default branch.
     * @param props? {
     *    defaultBranchName? Specify the default branch name without context.
     *    node? The current app to fetch defaultBranchName from context.
     * }
     * @returns The branch prefix.
     */
    static getBranchPrefix(props = {
        defaultBranchName: undefined,
        node: undefined,
    }) {
        return PDKPipeline.isDefaultBranch(props)
            ? ""
            : PDKPipeline.normalizeBranchName(process.env.BRANCH) + "-";
    }
    constructor(scope, id, props) {
        super(scope, id);
        this.node.setContext("@aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy", true);
        let codeRepository;
        // process.env.BRANCH is set only in CodeBuild builds
        if (PDKPipeline.isDefaultBranch({
            node: this.node,
            defaultBranchName: props.defaultBranchName,
        })) {
            // In the default branch, create a CodeCommit repository
            codeRepository = new aws_codecommit_1.Repository(this, "CodeRepository", {
                repositoryName: props.repositoryName,
            });
            codeRepository.applyRemovalPolicy(props.codeCommitRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.RETAIN);
        }
        else {
            // In a non-default branch, use an existing CodeCommit repository
            codeRepository = aws_codecommit_1.Repository.fromRepositoryName(scope, "CodeRepository", props.repositoryName);
        }
        const accessLogsBucket = new aws_s3_1.Bucket(this, "AccessLogsBucket", {
            versioned: false,
            enforceSSL: true,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            objectOwnership: aws_s3_1.ObjectOwnership.OBJECT_WRITER,
            publicReadAccess: false,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
        });
        const artifactBucket = new aws_s3_1.Bucket(this, "ArtifactsBucket", {
            enforceSSL: true,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: props.crossAccountKeys
                ? aws_s3_1.BucketEncryption.KMS
                : aws_s3_1.BucketEncryption.S3_MANAGED,
            encryptionKey: props.crossAccountKeys
                ? new aws_kms_1.Key(this, "ArtifactKey", {
                    enableKeyRotation: true,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                })
                : undefined,
            objectOwnership: aws_s3_1.ObjectOwnership.BUCKET_OWNER_ENFORCED,
            publicReadAccess: false,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            serverAccessLogsPrefix: "access-logs",
            serverAccessLogsBucket: accessLogsBucket,
        });
        const codePipeline = new aws_codepipeline_1.Pipeline(this, "CodePipeline", {
            enableKeyRotation: props.crossAccountKeys,
            restartExecutionOnUpdate: true,
            crossAccountKeys: props.crossAccountKeys,
            artifactBucket,
            pipelineType: aws_codepipeline_1.PipelineType.V1,
        });
        // ignore input and primaryOutputDirectory
        const { input, primaryOutputDirectory, commands, ...synthShellStepPartialProps } = props.synthShellStepPartialProps || {};
        const branch = process.env.BRANCH || props.defaultBranchName || exports.DEFAULT_BRANCH_NAME;
        const synthShellStep = new pipelines_1.ShellStep("Synth", {
            input: pipelines_1.CodePipelineSource.codeCommit(codeRepository, branch),
            env: props.branchNamePrefixes && props.branchNamePrefixes.length > 0
                ? {
                    BRANCH: branch,
                }
                : undefined,
            installCommands: ["npm install -g aws-cdk pnpm", "npx projen install"],
            commands: commands && commands.length > 0 ? commands : ["npx projen build"],
            primaryOutputDirectory: props.primarySynthDirectory,
            ...(synthShellStepPartialProps || {}),
        });
        synthShellStep.addOutputDirectory(".");
        const codePipelineProps = {
            codePipeline,
            ...props,
            crossAccountKeys: undefined,
            synth: synthShellStep,
        };
        this.codePipeline = new pipelines_1.CodePipeline(this, id, codePipelineProps);
        this.codeRepository = codeRepository;
        this.sonarCodeScannerConfig = props.sonarCodeScannerConfig
            ? {
                cdkOutDir: props.primarySynthDirectory,
                ...props.sonarCodeScannerConfig,
            }
            : undefined;
        this.branchNamePrefixes = props.branchNamePrefixes;
        this.defaultBranchName = props.defaultBranchName;
        this.repositoryName = props.repositoryName;
        if (props.branchNamePrefixes &&
            PDKPipeline.isDefaultBranch({
                node: this.node,
                defaultBranchName: props.defaultBranchName,
            })) {
            new feature_branches_1.FeatureBranches(this, "FeatureBranchPipelines", {
                codeRepository: this.codeRepository,
                cdkSrcDir: props.cdkSrcDir || path.dirname(props.primarySynthDirectory),
                synthShellStepPartialProps: props.synthShellStepPartialProps,
                cdkCommand: props.cdkCommand,
                branchNamePrefixes: props.branchNamePrefixes,
                defaultBranchName: props.defaultBranchName || exports.DEFAULT_BRANCH_NAME,
                codeBuildDefaults: props.codeBuildDefaults,
                dockerEnabledForSynth: props.dockerEnabledForSynth,
            });
        }
        else if (props.branchNamePrefixes) {
            aws_cdk_lib_1.Tags.of(aws_cdk_lib_1.Stack.of(this)).add("FeatureBranch", branch);
            aws_cdk_lib_1.Tags.of(aws_cdk_lib_1.Stack.of(this)).add("RepoName", this.repositoryName);
        }
        new aws_cdk_lib_1.CfnOutput(this, "CodeRepositoryGRCUrl", {
            value: this.codeRepository.repositoryCloneUrlGrc,
        });
    }
    /**
     * @inheritDoc
     */
    addStage(stage, options) {
        if (this.branchNamePrefixes &&
            !PDKPipeline.isDefaultBranch({
                node: stage.node,
                defaultBranchName: this.defaultBranchName,
            })) {
            aws_cdk_lib_1.Tags.of(stage).add("FeatureBranch", process.env.BRANCH);
            aws_cdk_lib_1.Tags.of(stage).add("RepoName", this.repositoryName);
        }
        // Add any root Aspects to the stage level as currently this doesn't happen automatically
        aws_cdk_lib_1.Aspects.of(stage.node.root).all.forEach((aspect) => aws_cdk_lib_1.Aspects.of(stage).add(aspect));
        return this.codePipeline.addStage(stage, options);
    }
    buildPipeline() {
        this.codePipeline.buildPipeline();
        this.sonarCodeScannerConfig &&
            new sonar_code_scanner_1.SonarCodeScanner(this, "SonarCodeScanner", {
                artifactBucketArn: this.codePipeline.pipeline.artifactBucket.bucketArn,
                artifactBucketKeyArn: this.codePipeline.pipeline.artifactBucket.encryptionKey?.keyArn,
                synthBuildArn: this.codePipeline.synthProject.projectArn,
                ...this.sonarCodeScannerConfig,
            });
        this.suppressCDKViolations();
    }
    suppressCDKViolations() {
        this.suppressRules(["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"], "Wildcards are needed for dynamically created resources.");
        this.suppressRules([
            "AwsSolutions-CB4",
            "AwsPrototyping-CodeBuildProjectKMSEncryptedArtifacts",
        ], "Encryption of Codebuild is not required.");
        this.suppressRules(["AwsSolutions-S1", "AwsPrototyping-S3BucketLoggingEnabled"], "Access Log buckets should not have s3 bucket logging");
    }
    suppressRules(rules, reason) {
        cdk_nag_1.NagSuppressions.addResourceSuppressions(this, rules.map((r) => ({
            id: r,
            reason,
        })), true);
    }
}
exports.PDKPipeline = PDKPipeline;
_a = JSII_RTTI_SYMBOL_1;
PDKPipeline[_a] = { fqn: "@aws/pdk.pipeline.PDKPipeline", version: "0.23.61" };
PDKPipeline.ALL_BRANCHES = [""];
PDKPipeline.defaultBranchName = exports.DEFAULT_BRANCH_NAME;
//# sourceMappingURL=data:application/json;base64,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