"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaIntegration = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const integration_1 = require("./integration");
const snap_start_java_function_1 = require("../functions/snap-start-java-function");
const utils_1 = require("../spec/utils");
/**
 * A lambda integration
 */
class LambdaIntegration extends integration_1.Integration {
    constructor(lambdaFunction) {
        super();
        // Snap Start applies only to versions, so if the function is a SnapStartFunction, we'll reference the current version
        if (lambdaFunction instanceof snap_start_java_function_1.SnapStartFunction) {
            this.lambdaFunction = lambdaFunction.currentVersion;
        }
        else {
            this.lambdaFunction = lambdaFunction;
        }
    }
    /**
     * Render the lambda integration as a snippet of OpenAPI
     */
    render(_props) {
        return {
            type: "AWS_PROXY",
            httpMethod: "POST",
            uri: (0, utils_1.functionInvocationUri)(this.lambdaFunction),
            passthroughBehavior: "WHEN_NO_MATCH",
        };
    }
    getOperationPermissionId(operationId) {
        return `LambdaPermission-${operationId}`;
    }
    /**
     * Grant API Gateway permissions to invoke the lambda
     */
    grant({ scope, api, operationId, method, path, operationLookup, }) {
        // Router permissions are unique to a function
        const routerPermissionId = `LambdaRouterPermission-${this.lambdaFunction.node.addr.slice(-8)}`;
        // Check if we've already granted a router permission for this lambda
        if (scope.node.tryFindChild(routerPermissionId)) {
            return; // The function already has access to all operations
        }
        // Check if a permission has been added for other operations for the same function arn
        const otherOperationPermissions = Object.keys(operationLookup)
            .map((opId) => scope.node.tryFindChild(this.getOperationPermissionId(opId)))
            .filter((permission) => permission &&
            permission instanceof aws_lambda_1.CfnPermission &&
            permission.functionName === this.lambdaFunction.functionArn);
        if (otherOperationPermissions.length > 0) {
            // This lambda function is reused, so we add the "router permission" which allows
            // invocation for any operation, to save exceeding the policy size limit for large
            // numbers of operations.
            otherOperationPermissions.forEach((permission) => scope.node.tryRemoveChild(permission.node.id));
            new aws_lambda_1.CfnPermission(scope, routerPermissionId, {
                action: "lambda:InvokeFunction",
                principal: "apigateway.amazonaws.com",
                functionName: this.lambdaFunction.functionArn,
                sourceArn: aws_cdk_lib_1.Stack.of(scope).formatArn({
                    service: "execute-api",
                    resource: api.restApiId,
                    // Permissions for all
                    resourceName: "*/*/*",
                }),
            });
        }
        else {
            // Add an individual operation permission since this lambda is not reused for multiple operations
            new aws_lambda_1.CfnPermission(scope, this.getOperationPermissionId(operationId), {
                action: "lambda:InvokeFunction",
                principal: "apigateway.amazonaws.com",
                functionName: this.lambdaFunction.functionArn,
                sourceArn: aws_cdk_lib_1.Stack.of(scope).formatArn({
                    service: "execute-api",
                    resource: api.restApiId,
                    // Scope permissions to any stage and a specific method and path of the operation.
                    // Path parameters (eg {param} are replaced with wildcards)
                    resourceName: `*/${method.toUpperCase()}${path.replace(/{[^\}]*\}/g, "*")}`,
                }),
            });
        }
    }
}
exports.LambdaIntegration = LambdaIntegration;
_a = JSII_RTTI_SYMBOL_1;
LambdaIntegration[_a] = { fqn: "@aws/pdk.type_safe_api.LambdaIntegration", version: "0.23.61" };
//# sourceMappingURL=data:application/json;base64,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