"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareSecuritySchemes = exports.getAllAuthorizers = exports.serializeAsAuthorizerReference = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const utils_1 = require("./utils");
const authorizers_1 = require("../authorizers");
const predicates_1 = require("../authorizers/predicates");
const constants_1 = require("../prepare-spec-event-handler/constants");
// Regex to match against a single header identity source
const SINGLE_HEADER_IDENTITY_SOURCE_REGEX = /^method.request.header.(?<header>[^\.\s,]+)$/;
/**
 * Build a serialized reference to an authorizer for use in an api method
 * @param authorizer the author to serialize
 */
const serializeAsAuthorizerReference = (authorizer) => authorizer
    ? {
        authorizerId: authorizer.authorizerId,
        authorizationScopes: authorizer.authorizationScopes,
    }
    : undefined;
exports.serializeAsAuthorizerReference = serializeAsAuthorizerReference;
/**
 * Create an OpenAPI security scheme definition for an iam authorizer
 * @see https://docs.amazonaws.cn/en_us/apigateway/latest/developerguide/api-gateway-swagger-extensions-authtype.html
 */
const iamSecurityScheme = () => ({
    type: "apiKey",
    name: "Authorization",
    in: "header",
    "x-amazon-apigateway-authtype": "awsSigv4",
});
/**
 * Create an API key security scheme
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-key-source.html
 */
const apiKeySecurityScheme = () => ({
    type: "apiKey",
    // Header must be x-api-key for API Gateway
    // See: https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-key-source.html
    name: "x-api-key",
    in: "header",
});
/**
 * Create an OpenAPI security scheme definition for a cognito authorizer
 * @param authorizer cognito authorizer
 */
const cognitoSecurityScheme = (authorizer) => ({
    type: "apiKey",
    name: "Authorization",
    in: "header",
    "x-amazon-apigateway-authtype": authorizer.authorizationType,
    "x-amazon-apigateway-authorizer": {
        type: aws_apigateway_1.AuthorizationType.COGNITO,
        providerARNs: authorizer.userPools.map((pool) => pool.userPoolArn),
    },
});
/**
 * Create an OpenAPI security scheme definition for a custom authorizer
 * @param scope the scope in which the api construct is defined
 * @param authorizer custom authorizer
 */
const customSecurityScheme = (_scope, authorizer) => {
    const singleHeaderMatch = authorizer.identitySource.match(SINGLE_HEADER_IDENTITY_SOURCE_REGEX);
    const singleHeader = singleHeaderMatch
        ? singleHeaderMatch.groups.header
        : undefined;
    // Token type must use a single header
    if (!singleHeader && authorizer.type === authorizers_1.CustomAuthorizerType.TOKEN) {
        throw new Error(`identitySource must be a single header for a ${authorizers_1.CustomAuthorizerType.TOKEN} authorizer, eg method.request.header.Authorization`);
    }
    return {
        type: "apiKey",
        in: "header",
        // When the identity source is not a single header, the value must be "Unused"
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-authorizer.html
        name: singleHeader ? singleHeader : "Unused",
        "x-amazon-apigateway-authtype": authorizer.authorizationType,
        "x-amazon-apigateway-authorizer": {
            type: authorizer.type,
            authorizerUri: (0, utils_1.functionInvocationUri)(authorizer.function),
            authorizerResultTtlInSeconds: authorizer.authorizerResultTtlInSeconds,
            identitySource: authorizer.identitySource,
        },
    };
};
/**
 * Return a list of all unique authorizers used in the api
 */
const getAllAuthorizers = (integrations, defaultAuthorizer) => Object.values(Object.fromEntries([
    ...(defaultAuthorizer ? [defaultAuthorizer] : []),
    ...Object.values(integrations).flatMap(({ authorizer }) => authorizer ? [authorizer] : []),
].map((a) => [a.authorizerId, a])));
exports.getAllAuthorizers = getAllAuthorizers;
/**
 * Generate the security schemes section of an OpenAPI specification
 */
const prepareSecuritySchemes = (scope, integrations, defaultAuthorizer, apiKeyOptions) => {
    // All the defined authorizers
    const allAuthorizers = (0, exports.getAllAuthorizers)(integrations, defaultAuthorizer);
    // Cognito, IAM and custom authorizers must be declared in security schemes
    return {
        ...Object.fromEntries(allAuthorizers
            .filter((authorizer) => (0, predicates_1.isCognitoAuthorizer)(authorizer))
            .map((authorizer) => [
            authorizer.authorizerId,
            cognitoSecurityScheme(authorizer),
        ])),
        ...Object.fromEntries(allAuthorizers
            .filter((authorizer) => (0, predicates_1.isCustomAuthorizer)(authorizer))
            .map((authorizer) => [
            authorizer.authorizerId,
            customSecurityScheme(scope, authorizer),
        ])),
        ...Object.fromEntries(allAuthorizers
            .filter((authorizer) => (0, predicates_1.isIamAuthorizer)(authorizer))
            .map((authorizer) => [authorizer.authorizerId, iamSecurityScheme()])),
        // A security scheme must be defined if the api key is in the header
        ...(apiKeyOptions?.source === aws_apigateway_1.ApiKeySourceType.HEADER
            ? {
                [constants_1.DefaultAuthorizerIds.API_KEY]: apiKeySecurityScheme(),
            }
            : {}),
    };
};
exports.prepareSecuritySchemes = prepareSecuritySchemes;
//# sourceMappingURL=data:application/json;base64,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