"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayWebAcl = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const pdk_nag_1 = require("../../../pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_wafv2_1 = require("aws-cdk-lib/aws-wafv2");
const constructs_1 = require("constructs");
/**
 * Associate an AWS WAF v2 Web ACL with the given api
 */
class OpenApiGatewayWebAcl extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const aclName = `${pdk_nag_1.PDKNag.getStackPrefix(aws_cdk_lib_1.Stack.of(this))
            .split("/")
            .join("-")}${id}-${this.node.addr.slice(-8)}`;
        const ipSetName = `${aclName}-IPSet`;
        // Create the IP Set if requested
        this.ipSet = props.cidrAllowList
            ? new aws_wafv2_1.CfnIPSet(this, "ApiIPSet", {
                name: ipSetName,
                addresses: props.cidrAllowList.cidrRanges,
                ipAddressVersion: props.cidrAllowList.cidrType,
                scope: "REGIONAL",
            })
            : undefined;
        // TODO: vendor property is deprecated, to be removed in the future iterations
        // and vendorName will be required
        const anyMissingVendor = props.managedRules?.some((q) => !q.vendorName && !q.vendor);
        if (anyMissingVendor) {
            throw new Error("The provided managed rules need to define either the vendor or vendorName (preferred) property");
        }
        const managedRules = props.managedRules ?? [
            { vendorName: "AWS", name: "AWSManagedRulesCommonRuleSet" },
        ];
        const rules = [
            // Add a rule for the IP Set if specified
            ...(this.ipSet
                ? [
                    {
                        name: ipSetName,
                        priority: 1,
                        visibilityConfig: {
                            metricName: ipSetName,
                            cloudWatchMetricsEnabled: true,
                            sampledRequestsEnabled: true,
                        },
                        action: {
                            block: {},
                        },
                        statement: {
                            notStatement: {
                                statement: {
                                    ipSetReferenceStatement: {
                                        arn: this.ipSet.attrArn,
                                    },
                                },
                            },
                        },
                    },
                ]
                : []),
            // Add the managed rules
            ...managedRules.map(({ vendor, vendorName, name, ...others }, i) => {
                // TODO: the usage of `vendor` it's for backward compatibility
                // it will be removed in the next PDK versions
                const vendorNameToUser = (vendor || vendorName);
                return {
                    name: `${vendorNameToUser}-${name}`,
                    priority: i + 2,
                    statement: {
                        managedRuleGroupStatement: {
                            ...others,
                            vendorName: vendorNameToUser,
                            name,
                        },
                    },
                    overrideAction: { none: {} },
                    visibilityConfig: {
                        metricName: `${aclName}-${vendorNameToUser}-${name}`,
                        cloudWatchMetricsEnabled: true,
                        sampledRequestsEnabled: true,
                    },
                };
            }),
        ];
        this.webAcl = new aws_wafv2_1.CfnWebACL(this, "ApiWebACL", {
            name: aclName,
            defaultAction: {
                // Allow by default, and use rules to deny unwanted requests
                allow: {},
            },
            scope: "REGIONAL",
            visibilityConfig: {
                cloudWatchMetricsEnabled: true,
                sampledRequestsEnabled: true,
                metricName: aclName,
            },
            rules,
        });
        this.webAclAssociation = new aws_wafv2_1.CfnWebACLAssociation(this, "WebACLAssociation", {
            resourceArn: props.apiDeploymentStageArn,
            webAclArn: this.webAcl.attrArn,
        });
    }
}
exports.OpenApiGatewayWebAcl = OpenApiGatewayWebAcl;
//# sourceMappingURL=data:application/json;base64,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