"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiToolsJsonFile = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const node_assert_1 = require("node:assert");
const path_1 = require("path");
const monorepo_1 = require("../../../../monorepo");
const projen_1 = require("projen");
/**
 * Represents an openapitools.json file
 * @see https://github.com/OpenAPITools/openapi-generator-cli#configuration
 */
class OpenApiToolsJsonFile extends projen_1.JsonFile {
    /**
     * Retrieves an instance of OpenApiToolsJsonFile if one is associated to the given project.
     *
     * @param project project instance.
     */
    static of(project) {
        return project.components.find((c) => monorepo_1.ProjectUtils.isNamedInstanceOf(c, OpenApiToolsJsonFile));
    }
    /**
     * Retrieves an instance of OpenApiToolsJsonFile if one is associated to the given project,
     * otherwise creates a OpenApiToolsJsonFile instance for the project.
     *
     * @param project project instance.
     */
    static ensure(project) {
        return (OpenApiToolsJsonFile.of(project) || new OpenApiToolsJsonFile(project));
    }
    constructor(project) {
        if (OpenApiToolsJsonFile.of(project)) {
            throw new Error(`Project ${project.name} already has associated OpenApiToolsJsonFile component.`);
        }
        const localFilePath = "openapitools.json";
        const dynamicFilePath = path_1.posix.join(".pdk/dynamic-files", localFilePath);
        super(project, dynamicFilePath, {
            obj: {
                // Schema is located in node_modules when generator cli is installed in tmp dir
                $schema: "node_modules/@openapitools/openapi-generator-cli/config.schema.json",
                spaces: 2,
                "generator-cli": () => this.config,
            },
        });
        /**
         * Configuration for OpenAPI Generator CLI
         * @private
         */
        this.config = {
            version: "6.3.0",
            storageDir: "~/.open-api-generator-cli",
        };
        // Ignore the location that gets copied to
        project.addGitIgnore(`/${localFilePath}`);
        // Create the task, but don't attach it anywhere yet. This is because of wanting
        // to do the generation as part of the "generate" tasks, which may not yet exist.
        this.createTask = project.addTask(`create-openapitools.json`, {
            steps: [{ exec: `cp -f ${dynamicFilePath} ${localFilePath}` }],
        });
    }
    preSynthesize() {
        const generateTask = this.project.tasks.tryFind("generate");
        (0, node_assert_1.ok)(generateTask);
        generateTask.prependSpawn(this.createTask);
    }
    /**
     * Update the OpenAPI Generator config. Shallow-merges with any existing default config
     */
    addOpenApiGeneratorCliConfig(config) {
        this.config = {
            ...this.config,
            ...config,
        };
    }
}
exports.OpenApiToolsJsonFile = OpenApiToolsJsonFile;
//# sourceMappingURL=data:application/json;base64,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