"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getHandlersProjectVendorExtensions = exports.buildInvokeMockDataGeneratorCommand = exports.buildCleanOpenApiGeneratedCodeCommand = exports.buildInvokeOpenApiGeneratorCommandArgs = exports.buildCodegenCommandArgs = exports.buildTypeSafeApiExecCommand = exports.getTypeSafeApiTaskEnvironment = exports.TypeSafeApiScript = exports.OtherGenerators = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const monorepo_1 = require("../../../../monorepo");
const runtime_version_utils_1 = require("../runtime-version-utils");
/**
 * Enum for generator directories for non-runtime generators
 */
var OtherGenerators;
(function (OtherGenerators) {
    OtherGenerators["DOCS"] = "docs";
    // Infrastructure
    OtherGenerators["TYPESCRIPT_CDK_INFRASTRUCTURE"] = "typescript-cdk-infrastructure";
    OtherGenerators["PYTHON_CDK_INFRASTRUCTURE"] = "python-cdk-infrastructure";
    OtherGenerators["JAVA_CDK_INFRASTRUCTURE"] = "java-cdk-infrastructure";
    OtherGenerators["TYPESCRIPT_ASYNC_CDK_INFRASTRUCTURE"] = "typescript-async-cdk-infrastructure";
    OtherGenerators["PYTHON_ASYNC_CDK_INFRASTRUCTURE"] = "python-async-cdk-infrastructure";
    OtherGenerators["JAVA_ASYNC_CDK_INFRASTRUCTURE"] = "java-async-cdk-infrastructure";
    // Handlers
    OtherGenerators["TYPESCRIPT_LAMBDA_HANDLERS"] = "typescript-lambda-handlers";
    OtherGenerators["PYTHON_LAMBDA_HANDLERS"] = "python-lambda-handlers";
    OtherGenerators["JAVA_LAMBDA_HANDLERS"] = "java-lambda-handlers";
    OtherGenerators["TYPESCRIPT_ASYNC_LAMBDA_HANDLERS"] = "typescript-async-lambda-handlers";
    OtherGenerators["PYTHON_ASYNC_LAMBDA_HANDLERS"] = "python-async-lambda-handlers";
    OtherGenerators["JAVA_ASYNC_LAMBDA_HANDLERS"] = "java-async-lambda-handlers";
    // Async runtime
    OtherGenerators["TYPESCRIPT_ASYNC_RUNTIME"] = "typescript-async-runtime";
    OtherGenerators["PYTHON_ASYNC_RUNTIME"] = "python-async-runtime";
    OtherGenerators["JAVA_ASYNC_RUNTIME"] = "java-async-runtime";
})(OtherGenerators || (exports.OtherGenerators = OtherGenerators = {}));
/**
 * Built in scripts.
 * If adding a script here, ensure you map it in TypeSafeApiProject (in /projenrc/projects)
 */
var TypeSafeApiScript;
(function (TypeSafeApiScript) {
    TypeSafeApiScript["PARSE_OPENAPI_SPEC"] = "type-safe-api parse-openapi-spec";
    TypeSafeApiScript["GENERATE"] = "type-safe-api.generate";
    TypeSafeApiScript["GENERATE_NEXT"] = "type-safe-api generate";
    TypeSafeApiScript["GENERATE_MOCK_DATA"] = "type-safe-api generate-mock-data";
    TypeSafeApiScript["GENERATE_HTML_REDOC_DOCS"] = "type-safe-api.generate-html-redoc-docs";
    TypeSafeApiScript["GENERATE_ASYNCAPI_HTML_DOCS"] = "type-safe-api.generate-asyncapi-html-docs";
    TypeSafeApiScript["GENERATE_ASYNCAPI_MARKDOWN_DOCS"] = "type-safe-api.generate-asyncapi-markdown-docs";
    TypeSafeApiScript["CLEAN_OPENAPI_GENERATED_CODE"] = "type-safe-api.clean-openapi-generated-code";
    TypeSafeApiScript["COPY_GRADLE_WRAPPER"] = "type-safe-api.copy-gradle-wrapper";
    TypeSafeApiScript["COPY_ASYNC_SMITHY_TRANSFORMER"] = "type-safe-api.copy-async-smithy-transformer";
    TypeSafeApiScript["GENERATE_ASYNCAPI_SPEC"] = "type-safe-api.generate-asyncapi-spec";
})(TypeSafeApiScript || (exports.TypeSafeApiScript = TypeSafeApiScript = {}));
/**
 * Return the environment that should be used for executing type safe api commands
 */
const getTypeSafeApiTaskEnvironment = () => ({
    AWS_PDK_VERSION: monorepo_1.ProjectUtils.getPdkVersion(),
});
exports.getTypeSafeApiTaskEnvironment = getTypeSafeApiTaskEnvironment;
/**
 * Build a command for running a script from this project's bin
 */
const buildTypeSafeApiExecCommand = (script, args) => {
    return `npx --yes -p @aws/pdk@$AWS_PDK_VERSION ${script}${args ? ` ${args}` : ""}`;
};
exports.buildTypeSafeApiExecCommand = buildTypeSafeApiExecCommand;
const serializeProperties = (properties) => Object.entries(properties)
    .map(([key, value]) => `${key}=${value}`)
    .join(",");
const buildCodegenCommandArgs = (options) => {
    const metadata = options.metadata
        ? ` --metadata '${JSON.stringify(options.metadata)}'`
        : "";
    return `--specPath ${options.specPath} --outputPath . --templateDirs ${options.templateDirs
        .map((t) => `"${t}"`)
        .join(" ")}${metadata}`;
};
exports.buildCodegenCommandArgs = buildCodegenCommandArgs;
/**
 * Generate code or docs by invoking the root generate script
 */
const buildInvokeOpenApiGeneratorCommandArgs = (options) => {
    const srcDir = options.srcDir ?? "src";
    const tstDir = options.tstDir ?? "test";
    const additionalProperties = options.additionalProperties
        ? ` --additional-properties "${serializeProperties(options.additionalProperties)}"`
        : "";
    const normalizers = options.normalizers
        ? ` --openapi-normalizer "${serializeProperties(Object.fromEntries(Object.entries(options.normalizers).map(([k, v]) => [k, `${v}`])))}"`
        : "";
    const extensions = options.extraVendorExtensions
        ? ` --extra-vendor-extensions '${JSON.stringify(options.extraVendorExtensions)}'`
        : "";
    const generateAliasAsModel = options.generateAliasAsModel ?? true ? " --generate-alias-as-model" : "";
    const specPath = options.specPath;
    const outputPath = ".";
    return `--generator ${options.generator} --spec-path ${specPath} --output-path ${outputPath} --generator-dir ${options.generatorDirectory} --src-dir ${srcDir} --tst-dir ${tstDir}${additionalProperties}${normalizers}${extensions}${generateAliasAsModel}`;
};
exports.buildInvokeOpenApiGeneratorCommandArgs = buildInvokeOpenApiGeneratorCommandArgs;
/**
 * Builds a command to clean up files which were previously generated by openapi generator
 */
const buildCleanOpenApiGeneratedCodeCommand = () => {
    return (0, exports.buildTypeSafeApiExecCommand)(TypeSafeApiScript.CLEAN_OPENAPI_GENERATED_CODE, `--code-path .`);
};
exports.buildCleanOpenApiGeneratedCodeCommand = buildCleanOpenApiGeneratedCodeCommand;
/**
 * Invoke the mock data generator script
 */
const buildInvokeMockDataGeneratorCommand = (options) => {
    const outputPath = options.outputSubDir ?? ".";
    const locale = options.locale ? ` --locale ${options.locale}` : "";
    const maxArrayLength = options.maxArrayLength !== undefined
        ? ` --maxArrayLength ${options.maxArrayLength}`
        : "";
    const seed = options.seed !== undefined ? ` --seed ${options.seed}` : "";
    return (0, exports.buildTypeSafeApiExecCommand)(TypeSafeApiScript.GENERATE_MOCK_DATA, `--specPath ${options.specPath} --outputPath ${outputPath}${locale}${maxArrayLength}${seed}`);
};
exports.buildInvokeMockDataGeneratorCommand = buildInvokeMockDataGeneratorCommand;
/**
 * Return vendor extensions containing details about the handler projects
 */
const getHandlersProjectVendorExtensions = (targetProject, { java, python, typescript }) => ({
    "x-handlers-python-module": python?.moduleName ?? "",
    "x-handlers-java-package": java?.packageName ?? "",
    "x-handlers-typescript-asset-path": typescript
        ? path.join(path.relative(targetProject.outdir, typescript.outdir), "dist", "lambda")
        : "",
    "x-handlers-python-asset-path": python
        ? path.join(path.relative(targetProject.outdir, python.outdir), "dist", "lambda")
        : "",
    "x-handlers-java-asset-path": java
        ? path.join(path.relative(targetProject.outdir, java.outdir), java.distdir, ...java.pom.groupId.split("."), java.pom.artifactId, java.pom.version, `${java.pom.artifactId}-${java.pom.version}.jar`)
        : "",
    "x-handlers-node-lambda-runtime-version": typescript
        ? runtime_version_utils_1.RuntimeVersionUtils.NODE.getLambdaRuntime(typescript.runtimeVersion)
        : "",
    "x-handlers-python-lambda-runtime-version": python
        ? runtime_version_utils_1.RuntimeVersionUtils.PYTHON.getLambdaRuntime(python.runtimeVersion)
        : "",
    "x-handlers-java-lambda-runtime-version": java
        ? runtime_version_utils_1.RuntimeVersionUtils.JAVA.getLambdaRuntime(java.runtimeVersion)
        : "",
});
exports.getHandlersProjectVendorExtensions = getHandlersProjectVendorExtensions;
//# sourceMappingURL=data:application/json;base64,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