"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedJavaHandlersBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const java_1 = require("projen/lib/java");
const languages_1 = require("../../languages");
const open_api_generator_handlebars_ignore_file_1 = require("../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
const runtime_version_utils_1 = require("../runtime-version-utils");
class GeneratedJavaHandlersBaseProject extends java_1.JavaProject {
    constructor(options) {
        super({
            sample: false,
            junit: false,
            compileOptions: runtime_version_utils_1.RuntimeVersionUtils.JAVA.getMavenCompileOptions(options.runtimeVersion),
            ...options,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)(this.buildOpenApiGeneratorOptions());
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        this.runtimeVersion = options.runtimeVersion ?? languages_1.JavaVersion.JAVA_17;
        this.packageName = `${this.pom.groupId}.${this.name}.handlers`;
        this.srcDir = path.join("src", "main", "java", ...this.packageName.split("."));
        this.tstDir = path.join("src", "test", "java", ...this.packageName.split("."));
        [
            `${options.generatedJavaTypes.pom.groupId}/${options.generatedJavaTypes.pom.artifactId}@${options.generatedJavaTypes.pom.version}`,
        ]
            .filter((dep) => !this.deps.tryGetDependency(dep.split("@")[0], projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        // Pin constructs version
        this.deps.removeDependency("software.constructs/constructs", projen_1.DependencyType.BUILD);
        this.addDependency("software.constructs/constructs@10.3.0");
        // Dependency on junit for tests
        this.deps.addDependency("org.junit.jupiter/junit-jupiter-engine@5.10.2", projen_1.DependencyType.TEST);
        // Remove the projen test dependency since otherwise it takes precedence, causing projen to be unavailable at synth time
        this.deps.removeDependency("io.github.cdklabs/projen", projen_1.DependencyType.TEST);
        // Add a dependency on the generated java types repository
        this.pom.addRepository({
            url: `file://${path.relative(this.outdir, options.generatedJavaTypes.outdir)}/dist/java`,
            id: `${options.generatedJavaTypes.pom.groupId}-${options.generatedJavaTypes.pom.artifactId}-repo`,
        });
        // Ignore everything for the first mustache pass
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*");
        // Ignore everything but the handler files for the handlebars pass
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", 
        // This will be split into a file per targeted handler
        `!${this.srcDir}/__all_handlers.java`, `!${this.tstDir}/__all_tests.java`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore the openapi generator metadata files
        this.gitignore.addPatterns(".openapi-generator");
        // Use the maven shade plugin to build a "super jar" which we can deploy to AWS Lambda
        this.pom.addPlugin("org.apache.maven.plugins/maven-shade-plugin@3.3.0", {
            configuration: {
                createDependencyReducedPom: false,
                transformers: [
                    {
                        // Transformer required for merging log4j2 plugins cache file
                        // https://docs.aws.amazon.com/lambda/latest/dg/java-logging.html#java-logging-cdk
                        transformer: {
                            "@implementation": "com.github.edwgiz.maven_shade_plugin.log4j2_cache_transformer.PluginsCacheFileTransformer",
                        },
                    },
                ],
            },
            executions: [
                {
                    id: "shade-task",
                    phase: "package",
                    goals: ["shade"],
                },
            ],
            dependencies: [
                "com.github.edwgiz/maven-shade-plugin.log4j2-cachefile-transformer@2.15.0",
            ],
        });
        // Use the maven surefire plugin to run junit tests
        this.pom.addPlugin("org.apache.maven.plugins/maven-surefire-plugin@3.2.1");
        // Log4j2 configuration for powertools logger
        new projen_1.SampleDir(this, "src/main/resources", {
            files: {
                "log4j2.xml": `<?xml version="1.0" encoding="UTF-8"?>
<Configuration>
    <Appenders>
        <Console name="JsonAppender" target="SYSTEM_OUT">
            <JsonTemplateLayout eventTemplateUri="classpath:LambdaJsonLayout.json" />
        </Console>
    </Appenders>
    <Loggers>
        <Logger name="JsonLogger" level="INFO" additivity="false">
            <AppenderRef ref="JsonAppender"/>
        </Logger>
        <Root level="info">
            <AppenderRef ref="JsonAppender"/>
        </Root>
    </Loggers>
</Configuration>`,
            },
        });
    }
}
exports.GeneratedJavaHandlersBaseProject = GeneratedJavaHandlersBaseProject;
//# sourceMappingURL=data:application/json;base64,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