"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonHandlersBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const languages_1 = require("../../languages");
const types_1 = require("../../types");
const open_api_generator_handlebars_ignore_file_1 = require("../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
const runtime_version_utils_1 = require("../runtime-version-utils");
class GeneratedPythonHandlersBaseProject extends python_1.PythonProject {
    constructor(options) {
        super({
            pytest: true,
            poetryOptions: {
                packages: [{ include: options.moduleName }],
                // Module must be explicitly added to include since poetry excludes everything in .gitignore by default
                include: [options.moduleName, `${options.moduleName}/**/*.py`],
            },
            ...options,
            sample: false,
            poetry: true,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)(this.buildOpenApiGeneratorOptions());
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        this.runtimeVersion = options.runtimeVersion ?? languages_1.PythonVersion.PYTHON_3_11;
        this.tstDir = "test";
        if (options.pytest ?? true) {
            // Pytest fails with exit code 5 when there are no tests.
            // We want to allow users to delete all their tests, or to upgrade an existing project without breaking their build
            // See: https://github.com/pytest-dev/pytest/issues/2393
            this.testTask.reset(`pytest ${this.tstDir}/ || ([ $? = 5 ] && exit 0 || exit $?)`);
        }
        [
            runtime_version_utils_1.RuntimeVersionUtils.PYTHON.getPythonDependencyVersion(this.runtimeVersion),
            `${options.generatedPythonTypes.name}@{path="${path.relative(this.outdir, options.generatedPythonTypes.outdir)}", develop=true}`,
        ]
            .filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        // Ignore everything for the first mustache pass
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*");
        // Ignore everything but the handler files for the handlebars pass
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", 
        // This will be split into a file per targeted handler
        `!${this.moduleName}/__all_handlers.py`, `!${this.tstDir}/__all_tests.py`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig({
            version: "6.6.0",
            ...options.openApiGeneratorCliConfig,
        });
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        // Ignore the generated code
        this.gitignore.addPatterns(".openapi-generator");
        // Write __init__.py as sample code
        new projen_1.SampleFile(this, path.join(this.moduleName, "__init__.py"), {
            contents: "#",
        });
        // Package task to build a distributable which can be deployed to lambda
        this.packageTask.exec("mkdir -p dist/lambda && rm -rf dist/lambda/*");
        this.packageTask.exec(`cp -r ${this.moduleName} dist/lambda/${this.moduleName}`);
        this.packageTask.exec("poetry export --without-hashes --format=requirements.txt | sed -E 's/^-e[[:space:]]+//' > dist/lambda/requirements.txt");
        // Select the platform based on the specified architecture, defaulting to x86_64
        // See: https://docs.aws.amazon.com/lambda/latest/dg/python-package.html#python-package-native-libraries
        const platform = options.architecture === types_1.Architecture.ARM_64
            ? "manylinux2014_aarch64"
            : "manylinux2014_x86_64";
        this.packageTask.exec(`pip install -r dist/lambda/requirements.txt --target dist/lambda --upgrade --platform ${platform} --only-binary :all: --python-version ${runtime_version_utils_1.RuntimeVersionUtils.PYTHON.getPipPackagingPythonVersion(this.runtimeVersion)}`);
    }
}
exports.GeneratedPythonHandlersBaseProject = GeneratedPythonHandlersBaseProject;
//# sourceMappingURL=data:application/json;base64,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