"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedTypescriptHandlersBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const languages_1 = require("../../languages");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
const runtime_version_utils_1 = require("../runtime-version-utils");
class GeneratedTypescriptHandlersBaseProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            tsconfig: {
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    // Generated code imports all models, and may not reference them all
                    noUnusedLocals: false,
                    noUnusedParameters: false,
                    skipLibCheck: true,
                    ...options?.tsconfig?.compilerOptions,
                },
            },
            npmignoreEnabled: false,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildCodegenCommandArgs)(this.buildCodegenOptions());
        };
        this.options = options;
        this.runtimeVersion = options.runtimeVersion ?? languages_1.NodeVersion.NODE_18;
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.addDeps(...[
            // If within a monorepo, add a regular dependency. Otherwise, use a file dependency to ensure the runtime can be
            // resolved
            options.isWithinMonorepo
                ? `${options.generatedTypescriptTypes.package.packageName}@${options.generatedTypescriptTypes.package.manifest.version}`
                : `${options.generatedTypescriptTypes.package.packageName}@file:${path.relative(this.outdir, options.generatedTypescriptTypes.outdir)}`,
        ].filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME)));
        this.addDevDeps("esbuild", "@types/aws-lambda");
        // Minimal .npmignore to avoid impacting OpenAPI Generator
        const npmignore = new projen_1.IgnoreFile(this, ".npmignore");
        npmignore.addPatterns("/.projen/", "/src", "/dist");
        this.generateTask = this.addTask("generate");
        this.generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE_NEXT, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(this.generateTask);
        // Ignore the openapi generator metadata
        this.gitignore.addPatterns(".openapi-generator", ".tsapi-metadata");
        // Create a separate lambda bundle for each handler as part of the package task.
        // Note that every typescript file directly in src is bundled by default, but users may specify their own
        // entry point globs if they prefer a different directory structure.
        this.packageTask.exec(`mkdir -p dist/lambda && rm -rf dist/lambda/*`);
        this.packageTask.exec(`esbuild --bundle ${(options.handlerEntryPoints ?? [`${this.srcdir}/*.ts`]).join(" ")} --platform=node --outdir=dist/lambda --target=${runtime_version_utils_1.RuntimeVersionUtils.NODE.getEsbuildNodeTarget(this.runtimeVersion)}`);
        // Move each bundled file into a separate directory
        this.packageTask.exec("for f in $(ls dist/lambda); do mkdir dist/lambda/$(basename $f .js) && mv dist/lambda/$f dist/lambda/$(basename $f .js)/index.js; done");
        // Create an empty index.ts sample on synth so that tsc is happy if the handlers project is configured
        // but no operations have @handler(language: "typescript")
        new projen_1.SampleDir(this, this.srcdir, {
            files: {
                "index.ts": "",
            },
        });
        // If we're not in a monorepo, we need to link the generated types such that the local dependency can be resolved
        if (!options.isWithinMonorepo) {
            switch (this.package.packageManager) {
                case javascript_1.NodePackageManager.PNPM:
                    this.tasks
                        .tryFind("install")
                        ?.prependExec(`${this.package.packageManager} link /${path.relative(this.outdir, this.options.generatedTypescriptTypes.outdir)}`);
                    break;
                default:
                    this.tasks
                        .tryFind("install")
                        ?.prependExec(`${this.package.packageManager} link ${this.options.generatedTypescriptTypes.package.packageName}`);
                    break;
            }
        }
    }
}
exports.GeneratedTypescriptHandlersBaseProject = GeneratedTypescriptHandlersBaseProject;
//# sourceMappingURL=data:application/json;base64,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