"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonCdkInfrastructureBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const open_api_generator_handlebars_ignore_file_1 = require("../../components/open-api-generator-handlebars-ignore-file");
const open_api_generator_ignore_file_1 = require("../../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../../components/type-safe-api-command-environment");
const utils_1 = require("../../components/utils");
class GeneratedPythonCdkInfrastructureBaseProject extends python_1.PythonProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            pytest: false,
            poetry: true,
            poetryOptions: {
                packages: [{ include: options.moduleName }],
                // Module must be explicitly added to include since poetry excludes everything in .gitignore by default
                include: [options.moduleName, `${options.moduleName}/**/*.py`],
            },
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)(this.buildOpenApiGeneratorOptions());
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        [
            "aws_pdk@^0",
            "constructs@^10",
            "aws-cdk-lib@^2",
            "cdk-nag@^2",
            "python@^3.9",
            `${options.generatedPythonTypes.name}@{path="${path.relative(this.outdir, options.generatedPythonTypes.outdir)}", develop=true}`,
        ]
            .filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        // Ignore everything but the target files
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        this.openapiGeneratorIgnore = openapiGeneratorIgnore;
        openapiGeneratorIgnore.addPatterns("/*", "**/*", "*", `!${this.moduleName}/__init__.py`, `!${this.moduleName}/api.py`);
        const openapiGeneratorHandlebarsIgnore = new open_api_generator_handlebars_ignore_file_1.OpenApiGeneratorHandlebarsIgnoreFile(this);
        openapiGeneratorHandlebarsIgnore.addPatterns("/*", "**/*", "*", `!${this.moduleName}/__functions.py`);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig({
            version: "6.6.0",
            ...options.openApiGeneratorCliConfig,
        });
        const generateTask = this.addTask("generate");
        this.generateTask = generateTask;
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        if (!options.commitGeneratedCode) {
            // Ignore the generated code
            this.gitignore.addPatterns(this.moduleName, ".openapi-generator", "mocks");
        }
        else {
            this.gitignore.addPatterns(".openapi-generator");
        }
        // The poetry install that runs as part of post synthesis expects there to be some code present, but code isn't
        // generated until build time. This means that the first install will fail when either generating the project for
        // the first time or checking out a fresh copy (since generated code is not checked in to version control). We
        // therefore add a blank __init__.py and README.md as our first install step to keep poetry happy until the
        // generator overwrites it.
        ["install", "install:ci"].forEach((t) => this.tasks
            .tryFind(t)
            ?.prependExec(`mkdir -p ${this.moduleName} && touch ${this.moduleName}/__init__.py README.md`));
    }
}
exports.GeneratedPythonCdkInfrastructureBaseProject = GeneratedPythonCdkInfrastructureBaseProject;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ2VuZXJhdGVkLXB5dGhvbi1jZGstaW5mcmFzdHJ1Y3R1cmUtYmFzZS1wcm9qZWN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZ2VuZXJhdGVkLXB5dGhvbi1jZGstaW5mcmFzdHJ1Y3R1cmUtYmFzZS1wcm9qZWN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBO3NDQUNzQztBQUN0Qyw2QkFBNkI7QUFDN0IsbUNBQThDO0FBQzlDLDhDQUF3RTtBQUt4RSwwSEFBa0g7QUFDbEgsb0dBQTZGO0FBQzdGLHdGQUFpRjtBQUNqRiwwR0FBbUc7QUFDbkcsa0RBTWdDO0FBcUJoQyxNQUFzQiwyQ0FBNEMsU0FBUSxzQkFBYTtJQVNyRixZQUFZLE9BQTJEO1FBQ3JFLEtBQUssQ0FBQztZQUNKLEdBQUcsT0FBTztZQUNWLE1BQU0sRUFBRSxLQUFLO1lBQ2IsTUFBTSxFQUFFLEtBQUs7WUFDYixNQUFNLEVBQUUsSUFBSTtZQUNaLGFBQWEsRUFBRTtnQkFDYixRQUFRLEVBQUUsQ0FBQyxFQUFFLE9BQU8sRUFBRSxPQUFPLENBQUMsVUFBVSxFQUFFLENBQUM7Z0JBQzNDLHVHQUF1RztnQkFDdkcsT0FBTyxFQUFFLENBQUMsT0FBTyxDQUFDLFVBQVUsRUFBRSxHQUFHLE9BQU8sQ0FBQyxVQUFVLFVBQVUsQ0FBQzthQUMvRDtTQUNGLENBQUMsQ0FBQztRQWlGRSw2QkFBd0IsR0FBRyxHQUFHLEVBQUU7WUFDckMsT0FBTyxJQUFBLDhDQUFzQyxFQUMzQyxJQUFJLENBQUMsNEJBQTRCLEVBQUUsQ0FDcEMsQ0FBQztRQUNKLENBQUMsQ0FBQztRQXBGQSxpRUFBNkIsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDM0MsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUM7UUFFdkI7WUFDRSxZQUFZO1lBQ1osZ0JBQWdCO1lBQ2hCLGdCQUFnQjtZQUNoQixZQUFZO1lBQ1osYUFBYTtZQUNiLEdBQUcsT0FBTyxDQUFDLG9CQUFvQixDQUFDLElBQUksV0FBVyxJQUFJLENBQUMsUUFBUSxDQUMxRCxJQUFJLENBQUMsTUFBTSxFQUNYLE9BQU8sQ0FBQyxvQkFBb0IsQ0FBQyxNQUFNLENBQ3BDLGtCQUFrQjtTQUNwQjthQUNFLE1BQU0sQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEdBQUcsRUFBRSx1QkFBYyxDQUFDLE9BQU8sQ0FBQyxDQUFDO2FBQ3pFLE9BQU8sQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBRTdDLHlDQUF5QztRQUN6QyxNQUFNLHNCQUFzQixHQUFHLElBQUksMkRBQTBCLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDcEUsSUFBSSxDQUFDLHNCQUFzQixHQUFHLHNCQUFzQixDQUFDO1FBQ3JELHNCQUFzQixDQUFDLFdBQVcsQ0FDaEMsSUFBSSxFQUNKLE1BQU0sRUFDTixHQUFHLEVBQ0gsSUFBSSxJQUFJLENBQUMsVUFBVSxjQUFjLEVBQ2pDLElBQUksSUFBSSxDQUFDLFVBQVUsU0FBUyxDQUM3QixDQUFDO1FBRUYsTUFBTSxnQ0FBZ0MsR0FDcEMsSUFBSSxnRkFBb0MsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNqRCxnQ0FBZ0MsQ0FBQyxXQUFXLENBQzFDLElBQUksRUFDSixNQUFNLEVBQ04sR0FBRyxFQUNILElBQUksSUFBSSxDQUFDLFVBQVUsaUJBQWlCLENBQ3JDLENBQUM7UUFFRiwwQ0FBMEM7UUFDMUMsK0NBQW9CLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLDRCQUE0QixDQUFDO1lBQzdELE9BQU8sRUFBRSxPQUFPO1lBQ2hCLEdBQUcsT0FBTyxDQUFDLHlCQUF5QjtTQUNyQyxDQUFDLENBQUM7UUFFSCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQzlDLElBQUksQ0FBQyxZQUFZLEdBQUcsWUFBWSxDQUFDO1FBQ2pDLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBQSw2Q0FBcUMsR0FBRSxDQUFDLENBQUM7UUFDM0QsWUFBWSxDQUFDLElBQUksQ0FDZixJQUFBLG1DQUEyQixFQUN6Qix5QkFBaUIsQ0FBQyxRQUFRLEVBQzFCLElBQUksQ0FBQyx3QkFBd0IsRUFBRSxDQUNoQyxDQUNGLENBQUM7UUFFRixJQUFJLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUV4QyxJQUFJLENBQUMsT0FBTyxDQUFDLG1CQUFtQixFQUFFLENBQUM7WUFDakMsNEJBQTRCO1lBQzVCLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxDQUN4QixJQUFJLENBQUMsVUFBVSxFQUNmLG9CQUFvQixFQUNwQixPQUFPLENBQ1IsQ0FBQztRQUNKLENBQUM7YUFBTSxDQUFDO1lBQ04sSUFBSSxDQUFDLFNBQVMsQ0FBQyxXQUFXLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUNuRCxDQUFDO1FBRUQsK0dBQStHO1FBQy9HLGlIQUFpSDtRQUNqSCw4R0FBOEc7UUFDOUcsMkdBQTJHO1FBQzNHLDJCQUEyQjtRQUMzQixDQUFDLFNBQVMsRUFBRSxZQUFZLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUN0QyxJQUFJLENBQUMsS0FBSzthQUNQLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDWCxFQUFFLFdBQVcsQ0FDWCxZQUFZLElBQUksQ0FBQyxVQUFVLGFBQWEsSUFBSSxDQUFDLFVBQVUsd0JBQXdCLENBQ2hGLENBQ0osQ0FBQztJQUNKLENBQUM7Q0FTRjtBQTVHRCxrR0E0R0MiLCJzb3VyY2VzQ29udGVudCI6WyIvKiEgQ29weXJpZ2h0IFtBbWF6b24uY29tXShodHRwOi8vYW1hem9uLmNvbS8pLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjAgKi9cbmltcG9ydCAqIGFzIHBhdGggZnJvbSBcInBhdGhcIjtcbmltcG9ydCB7IERlcGVuZGVuY3lUeXBlLCBUYXNrIH0gZnJvbSBcInByb2plblwiO1xuaW1wb3J0IHsgUHl0aG9uUHJvamVjdCwgUHl0aG9uUHJvamVjdE9wdGlvbnMgfSBmcm9tIFwicHJvamVuL2xpYi9weXRob25cIjtcbmltcG9ydCB7XG4gIENvZGVHZW5lcmF0aW9uU291cmNlT3B0aW9ucyxcbiAgR2VuZXJhdGVkV2l0aE9wZW5BcGlHZW5lcmF0b3JPcHRpb25zLFxufSBmcm9tIFwiLi4vLi4vLi4vdHlwZXNcIjtcbmltcG9ydCB7IE9wZW5BcGlHZW5lcmF0b3JIYW5kbGViYXJzSWdub3JlRmlsZSB9IGZyb20gXCIuLi8uLi9jb21wb25lbnRzL29wZW4tYXBpLWdlbmVyYXRvci1oYW5kbGViYXJzLWlnbm9yZS1maWxlXCI7XG5pbXBvcnQgeyBPcGVuQXBpR2VuZXJhdG9ySWdub3JlRmlsZSB9IGZyb20gXCIuLi8uLi9jb21wb25lbnRzL29wZW4tYXBpLWdlbmVyYXRvci1pZ25vcmUtZmlsZVwiO1xuaW1wb3J0IHsgT3BlbkFwaVRvb2xzSnNvbkZpbGUgfSBmcm9tIFwiLi4vLi4vY29tcG9uZW50cy9vcGVuLWFwaS10b29scy1qc29uLWZpbGVcIjtcbmltcG9ydCB7IFR5cGVTYWZlQXBpQ29tbWFuZEVudmlyb25tZW50IH0gZnJvbSBcIi4uLy4uL2NvbXBvbmVudHMvdHlwZS1zYWZlLWFwaS1jb21tYW5kLWVudmlyb25tZW50XCI7XG5pbXBvcnQge1xuICBidWlsZENsZWFuT3BlbkFwaUdlbmVyYXRlZENvZGVDb21tYW5kLFxuICBidWlsZEludm9rZU9wZW5BcGlHZW5lcmF0b3JDb21tYW5kQXJncyxcbiAgYnVpbGRUeXBlU2FmZUFwaUV4ZWNDb21tYW5kLFxuICBHZW5lcmF0aW9uT3B0aW9ucyxcbiAgVHlwZVNhZmVBcGlTY3JpcHQsXG59IGZyb20gXCIuLi8uLi9jb21wb25lbnRzL3V0aWxzXCI7XG5pbXBvcnQgeyBHZW5lcmF0ZWRIYW5kbGVyc1Byb2plY3RzIH0gZnJvbSBcIi4uLy4uL2dlbmVyYXRlXCI7XG5pbXBvcnQgeyBHZW5lcmF0ZWRQeXRob25SdW50aW1lQmFzZVByb2plY3QgfSBmcm9tIFwiLi4vLi4vcnVudGltZS9nZW5lcmF0ZWQtcHl0aG9uLXJ1bnRpbWUtYmFzZS1wcm9qZWN0XCI7XG5cbmV4cG9ydCBpbnRlcmZhY2UgR2VuZXJhdGVkUHl0aG9uSW5mcmFzdHJ1Y3R1cmVCYXNlT3B0aW9uc1xuICBleHRlbmRzIFB5dGhvblByb2plY3RPcHRpb25zLFxuICAgIEdlbmVyYXRlZFdpdGhPcGVuQXBpR2VuZXJhdG9yT3B0aW9ucyB7fVxuXG5leHBvcnQgaW50ZXJmYWNlIEdlbmVyYXRlZFB5dGhvbkNka0luZnJhc3RydWN0dXJlQmFzZVByb2plY3RPcHRpb25zXG4gIGV4dGVuZHMgR2VuZXJhdGVkUHl0aG9uSW5mcmFzdHJ1Y3R1cmVCYXNlT3B0aW9ucyxcbiAgICBDb2RlR2VuZXJhdGlvblNvdXJjZU9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIGdlbmVyYXRlZCBweXRob24gdHlwZXNcbiAgICovXG4gIHJlYWRvbmx5IGdlbmVyYXRlZFB5dGhvblR5cGVzOiBHZW5lcmF0ZWRQeXRob25SdW50aW1lQmFzZVByb2plY3Q7XG4gIC8qKlxuICAgKiBHZW5lcmF0ZWQgaGFuZGxlcnMgcHJvamVjdHNcbiAgICovXG4gIHJlYWRvbmx5IGdlbmVyYXRlZEhhbmRsZXJzOiBHZW5lcmF0ZWRIYW5kbGVyc1Byb2plY3RzO1xufVxuXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgR2VuZXJhdGVkUHl0aG9uQ2RrSW5mcmFzdHJ1Y3R1cmVCYXNlUHJvamVjdCBleHRlbmRzIFB5dGhvblByb2plY3Qge1xuICAvKipcbiAgICogT3B0aW9ucyBjb25maWd1cmVkIGZvciB0aGUgcHJvamVjdFxuICAgKi9cbiAgcHJvdGVjdGVkIHJlYWRvbmx5IG9wdGlvbnM6IEdlbmVyYXRlZFB5dGhvbkNka0luZnJhc3RydWN0dXJlQmFzZVByb2plY3RPcHRpb25zO1xuXG4gIHByb3RlY3RlZCByZWFkb25seSBnZW5lcmF0ZVRhc2s6IFRhc2s7XG4gIHByb3RlY3RlZCByZWFkb25seSBvcGVuYXBpR2VuZXJhdG9ySWdub3JlOiBPcGVuQXBpR2VuZXJhdG9ySWdub3JlRmlsZTtcblxuICBjb25zdHJ1Y3RvcihvcHRpb25zOiBHZW5lcmF0ZWRQeXRob25DZGtJbmZyYXN0cnVjdHVyZUJhc2VQcm9qZWN0T3B0aW9ucykge1xuICAgIHN1cGVyKHtcbiAgICAgIC4uLm9wdGlvbnMsXG4gICAgICBzYW1wbGU6IGZhbHNlLFxuICAgICAgcHl0ZXN0OiBmYWxzZSxcbiAgICAgIHBvZXRyeTogdHJ1ZSxcbiAgICAgIHBvZXRyeU9wdGlvbnM6IHtcbiAgICAgICAgcGFja2FnZXM6IFt7IGluY2x1ZGU6IG9wdGlvbnMubW9kdWxlTmFtZSB9XSxcbiAgICAgICAgLy8gTW9kdWxlIG11c3QgYmUgZXhwbGljaXRseSBhZGRlZCB0byBpbmNsdWRlIHNpbmNlIHBvZXRyeSBleGNsdWRlcyBldmVyeXRoaW5nIGluIC5naXRpZ25vcmUgYnkgZGVmYXVsdFxuICAgICAgICBpbmNsdWRlOiBbb3B0aW9ucy5tb2R1bGVOYW1lLCBgJHtvcHRpb25zLm1vZHVsZU5hbWV9LyoqLyoucHlgXSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgVHlwZVNhZmVBcGlDb21tYW5kRW52aXJvbm1lbnQuZW5zdXJlKHRoaXMpO1xuICAgIHRoaXMub3B0aW9ucyA9IG9wdGlvbnM7XG5cbiAgICBbXG4gICAgICBcImF3c19wZGtAXjBcIixcbiAgICAgIFwiY29uc3RydWN0c0BeMTBcIixcbiAgICAgIFwiYXdzLWNkay1saWJAXjJcIixcbiAgICAgIFwiY2RrLW5hZ0BeMlwiLFxuICAgICAgXCJweXRob25AXjMuOVwiLFxuICAgICAgYCR7b3B0aW9ucy5nZW5lcmF0ZWRQeXRob25UeXBlcy5uYW1lfUB7cGF0aD1cIiR7cGF0aC5yZWxhdGl2ZShcbiAgICAgICAgdGhpcy5vdXRkaXIsXG4gICAgICAgIG9wdGlvbnMuZ2VuZXJhdGVkUHl0aG9uVHlwZXMub3V0ZGlyXG4gICAgICApfVwiLCBkZXZlbG9wPXRydWV9YCxcbiAgICBdXG4gICAgICAuZmlsdGVyKChkZXApID0+ICF0aGlzLmRlcHMudHJ5R2V0RGVwZW5kZW5jeShkZXAsIERlcGVuZGVuY3lUeXBlLlJVTlRJTUUpKVxuICAgICAgLmZvckVhY2goKGRlcCkgPT4gdGhpcy5hZGREZXBlbmRlbmN5KGRlcCkpO1xuXG4gICAgLy8gSWdub3JlIGV2ZXJ5dGhpbmcgYnV0IHRoZSB0YXJnZXQgZmlsZXNcbiAgICBjb25zdCBvcGVuYXBpR2VuZXJhdG9ySWdub3JlID0gbmV3IE9wZW5BcGlHZW5lcmF0b3JJZ25vcmVGaWxlKHRoaXMpO1xuICAgIHRoaXMub3BlbmFwaUdlbmVyYXRvcklnbm9yZSA9IG9wZW5hcGlHZW5lcmF0b3JJZ25vcmU7XG4gICAgb3BlbmFwaUdlbmVyYXRvcklnbm9yZS5hZGRQYXR0ZXJucyhcbiAgICAgIFwiLypcIixcbiAgICAgIFwiKiovKlwiLFxuICAgICAgXCIqXCIsXG4gICAgICBgISR7dGhpcy5tb2R1bGVOYW1lfS9fX2luaXRfXy5weWAsXG4gICAgICBgISR7dGhpcy5tb2R1bGVOYW1lfS9hcGkucHlgXG4gICAgKTtcblxuICAgIGNvbnN0IG9wZW5hcGlHZW5lcmF0b3JIYW5kbGViYXJzSWdub3JlID1cbiAgICAgIG5ldyBPcGVuQXBpR2VuZXJhdG9ySGFuZGxlYmFyc0lnbm9yZUZpbGUodGhpcyk7XG4gICAgb3BlbmFwaUdlbmVyYXRvckhhbmRsZWJhcnNJZ25vcmUuYWRkUGF0dGVybnMoXG4gICAgICBcIi8qXCIsXG4gICAgICBcIioqLypcIixcbiAgICAgIFwiKlwiLFxuICAgICAgYCEke3RoaXMubW9kdWxlTmFtZX0vX19mdW5jdGlvbnMucHlgXG4gICAgKTtcblxuICAgIC8vIEFkZCBPcGVuQVBJIEdlbmVyYXRvciBjbGkgY29uZmlndXJhdGlvblxuICAgIE9wZW5BcGlUb29sc0pzb25GaWxlLmVuc3VyZSh0aGlzKS5hZGRPcGVuQXBpR2VuZXJhdG9yQ2xpQ29uZmlnKHtcbiAgICAgIHZlcnNpb246IFwiNi42LjBcIixcbiAgICAgIC4uLm9wdGlvbnMub3BlbkFwaUdlbmVyYXRvckNsaUNvbmZpZyxcbiAgICB9KTtcblxuICAgIGNvbnN0IGdlbmVyYXRlVGFzayA9IHRoaXMuYWRkVGFzayhcImdlbmVyYXRlXCIpO1xuICAgIHRoaXMuZ2VuZXJhdGVUYXNrID0gZ2VuZXJhdGVUYXNrO1xuICAgIGdlbmVyYXRlVGFzay5leGVjKGJ1aWxkQ2xlYW5PcGVuQXBpR2VuZXJhdGVkQ29kZUNvbW1hbmQoKSk7XG4gICAgZ2VuZXJhdGVUYXNrLmV4ZWMoXG4gICAgICBidWlsZFR5cGVTYWZlQXBpRXhlY0NvbW1hbmQoXG4gICAgICAgIFR5cGVTYWZlQXBpU2NyaXB0LkdFTkVSQVRFLFxuICAgICAgICB0aGlzLmJ1aWxkR2VuZXJhdGVDb21tYW5kQXJncygpXG4gICAgICApXG4gICAgKTtcblxuICAgIHRoaXMucHJlQ29tcGlsZVRhc2suc3Bhd24oZ2VuZXJhdGVUYXNrKTtcblxuICAgIGlmICghb3B0aW9ucy5jb21taXRHZW5lcmF0ZWRDb2RlKSB7XG4gICAgICAvLyBJZ25vcmUgdGhlIGdlbmVyYXRlZCBjb2RlXG4gICAgICB0aGlzLmdpdGlnbm9yZS5hZGRQYXR0ZXJucyhcbiAgICAgICAgdGhpcy5tb2R1bGVOYW1lLFxuICAgICAgICBcIi5vcGVuYXBpLWdlbmVyYXRvclwiLFxuICAgICAgICBcIm1vY2tzXCJcbiAgICAgICk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuZ2l0aWdub3JlLmFkZFBhdHRlcm5zKFwiLm9wZW5hcGktZ2VuZXJhdG9yXCIpO1xuICAgIH1cblxuICAgIC8vIFRoZSBwb2V0cnkgaW5zdGFsbCB0aGF0IHJ1bnMgYXMgcGFydCBvZiBwb3N0IHN5bnRoZXNpcyBleHBlY3RzIHRoZXJlIHRvIGJlIHNvbWUgY29kZSBwcmVzZW50LCBidXQgY29kZSBpc24ndFxuICAgIC8vIGdlbmVyYXRlZCB1bnRpbCBidWlsZCB0aW1lLiBUaGlzIG1lYW5zIHRoYXQgdGhlIGZpcnN0IGluc3RhbGwgd2lsbCBmYWlsIHdoZW4gZWl0aGVyIGdlbmVyYXRpbmcgdGhlIHByb2plY3QgZm9yXG4gICAgLy8gdGhlIGZpcnN0IHRpbWUgb3IgY2hlY2tpbmcgb3V0IGEgZnJlc2ggY29weSAoc2luY2UgZ2VuZXJhdGVkIGNvZGUgaXMgbm90IGNoZWNrZWQgaW4gdG8gdmVyc2lvbiBjb250cm9sKS4gV2VcbiAgICAvLyB0aGVyZWZvcmUgYWRkIGEgYmxhbmsgX19pbml0X18ucHkgYW5kIFJFQURNRS5tZCBhcyBvdXIgZmlyc3QgaW5zdGFsbCBzdGVwIHRvIGtlZXAgcG9ldHJ5IGhhcHB5IHVudGlsIHRoZVxuICAgIC8vIGdlbmVyYXRvciBvdmVyd3JpdGVzIGl0LlxuICAgIFtcImluc3RhbGxcIiwgXCJpbnN0YWxsOmNpXCJdLmZvckVhY2goKHQpID0+XG4gICAgICB0aGlzLnRhc2tzXG4gICAgICAgIC50cnlGaW5kKHQpXG4gICAgICAgID8ucHJlcGVuZEV4ZWMoXG4gICAgICAgICAgYG1rZGlyIC1wICR7dGhpcy5tb2R1bGVOYW1lfSAmJiB0b3VjaCAke3RoaXMubW9kdWxlTmFtZX0vX19pbml0X18ucHkgUkVBRE1FLm1kYFxuICAgICAgICApXG4gICAgKTtcbiAgfVxuXG4gIHB1YmxpYyBidWlsZEdlbmVyYXRlQ29tbWFuZEFyZ3MgPSAoKSA9PiB7XG4gICAgcmV0dXJuIGJ1aWxkSW52b2tlT3BlbkFwaUdlbmVyYXRvckNvbW1hbmRBcmdzKFxuICAgICAgdGhpcy5idWlsZE9wZW5BcGlHZW5lcmF0b3JPcHRpb25zKClcbiAgICApO1xuICB9O1xuXG4gIHByb3RlY3RlZCBhYnN0cmFjdCBidWlsZE9wZW5BcGlHZW5lcmF0b3JPcHRpb25zKCk6IEdlbmVyYXRpb25PcHRpb25zO1xufVxuIl19