"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedJavaRuntimeBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const projen_1 = require("projen");
const java_1 = require("projen/lib/java");
const open_api_generator_ignore_file_1 = require("../components/open-api-generator-ignore-file");
const open_api_tools_json_file_1 = require("../components/open-api-tools-json-file");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
const DEPENDENCIES = [
    // Required for open api generated code
    "io.swagger/swagger-annotations@1.6.8",
    "com.google.code.findbugs/jsr305@3.0.2",
    "com.squareup.okhttp3/okhttp@4.10.0",
    "com.squareup.okhttp3/logging-interceptor@4.10.0",
    "com.google.code.gson/gson@2.9.1",
    "io.gsonfire/gson-fire@1.8.5",
    "org.apache.commons/commons-lang3@3.12.0",
    "jakarta.annotation/jakarta.annotation-api@1.3.5",
    "org.openapitools/jackson-databind-nullable@0.2.4",
    "javax.ws.rs/jsr311-api@1.1.1",
    "javax.ws.rs/javax.ws.rs-api@2.1.1",
    // For handler wrappers
    "com.amazonaws/aws-lambda-java-core@1.2.1",
    "com.amazonaws/aws-lambda-java-events@3.11.0",
    // Lombok is used to add the builder pattern to models for neater construction
    "org.projectlombok/lombok@1.18.24",
    // Interceptors
    "software.amazon.lambda/powertools-logging@1.18.0",
    "software.amazon.lambda/powertools-tracing@1.18.0",
    "software.amazon.lambda/powertools-metrics@1.18.0",
    // SnapStart
    "io.github.crac/org-crac@0.1.3",
];
const TEST_DEPENDENCIES = [
    "org.junit.jupiter/junit-jupiter-api@5.9.1",
    "org.mockito/mockito-core@3.12.4",
];
/**
 * Java project containing types generated using OpenAPI Generator CLI
 */
class GeneratedJavaRuntimeBaseProject extends java_1.JavaProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            junit: false,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildInvokeOpenApiGeneratorCommandArgs)(this.buildOpenApiGeneratorOptions());
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        // Ignore files that we will control via projen
        this.openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        this.openapiGeneratorIgnore.addPatterns(...GeneratedJavaRuntimeBaseProject.openApiIgnorePatterns);
        // Add OpenAPI Generator cli configuration
        open_api_tools_json_file_1.OpenApiToolsJsonFile.ensure(this).addOpenApiGeneratorCliConfig(options.openApiGeneratorCliConfig);
        // Add dependencies
        DEPENDENCIES.forEach((dep) => this.addDependency(dep));
        TEST_DEPENDENCIES.forEach((dep) => this.addTestDependency(dep));
        // Pin constructs version
        this.deps.removeDependency("software.constructs/constructs", projen_1.DependencyType.BUILD);
        this.addDependency("software.constructs/constructs@10.3.0");
        this.packageName = `${this.pom.groupId}.${this.name}.runtime`;
        // Generate the java code
        const generateTask = this.addTask("generate");
        generateTask.exec((0, utils_1.buildCleanOpenApiGeneratedCodeCommand)());
        generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(generateTask);
        if (!options.commitGeneratedCode) {
            // Ignore all the generated code
            this.gitignore.addPatterns("src", "docs", "api", "README.md", ".openapi-generator");
        }
        else {
            this.gitignore.addPatterns(".openapi-generator");
        }
    }
}
exports.GeneratedJavaRuntimeBaseProject = GeneratedJavaRuntimeBaseProject;
/**
 * Patterns that are excluded from code generation
 */
GeneratedJavaRuntimeBaseProject.openApiIgnorePatterns = [
    "pom.xml",
    "gradle",
    "gradle/**/*",
    "gradlew",
    "gradle.properties",
    "gradlew.bat",
    "build.gradle",
    "settings.gradle",
    "build.sbt",
    ".travis.yml",
    "git_push.sh",
    "src/test",
    "src/test/**/*",
    "src/main/AndroidManifest.xml",
];
//# sourceMappingURL=data:application/json;base64,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