"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedTypescriptRuntimeBaseProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const monorepo_1 = require("../../../../monorepo");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const type_safe_api_command_environment_1 = require("../components/type-safe-api-command-environment");
const utils_1 = require("../components/utils");
/**
 * Typescript project containing types generated using OpenAPI Generator CLI
 */
class GeneratedTypescriptRuntimeBaseProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            tsconfig: {
                ...options.tsconfig,
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    // Generated code isn't very strict!
                    strict: false,
                    alwaysStrict: false,
                    noImplicitAny: false,
                    noImplicitReturns: false,
                    noImplicitThis: false,
                    noUnusedLocals: false,
                    noUnusedParameters: false,
                    strictNullChecks: false,
                    strictPropertyInitialization: false,
                    skipLibCheck: true,
                    ...options?.tsconfig?.compilerOptions,
                },
            },
            eslint: false,
            // Disable tests unless explicitly enabled
            jest: options.jest ?? false,
            npmignoreEnabled: false,
        });
        this.buildGenerateCommandArgs = () => {
            return (0, utils_1.buildCodegenCommandArgs)(this.buildCodegenOptions());
        };
        type_safe_api_command_environment_1.TypeSafeApiCommandEnvironment.ensure(this);
        this.options = options;
        // Disable strict peer dependencies for pnpm as the default typescript project dependencies have type mismatches
        // (ts-jest@27 and @types/jest@28)
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            this.npmrc.addConfig("strict-peer-dependencies", "false");
        }
        // For event and context types
        this.addDeps("@types/aws-lambda", "@aws-lambda-powertools/tracer", "@aws-lambda-powertools/logger", "@aws-lambda-powertools/metrics");
        // Minimal .npmignore to avoid impacting OpenAPI Generator
        const npmignore = new projen_1.IgnoreFile(this, ".npmignore");
        npmignore.addPatterns("/.projen/", `/${this.srcdir}`, "/dist");
        this.generateTask = this.addTask("generate");
        this.generateTask.exec((0, utils_1.buildTypeSafeApiExecCommand)(utils_1.TypeSafeApiScript.GENERATE_NEXT, this.buildGenerateCommandArgs()));
        this.preCompileTask.spawn(this.generateTask);
        if (!options.commitGeneratedCode) {
            // Ignore all the generated code
            this.gitignore.addPatterns(this.srcdir, ".npmignore", "README.md");
        }
        this.gitignore.addPatterns(".openapi-generator", ".tsapi-manifest");
        // If we're not in a monorepo, we need to link the generated client such that any local dependency on it can be
        // resolved
        if (!options.isWithinMonorepo) {
            switch (this.package.packageManager) {
                case javascript_1.NodePackageManager.PNPM:
                    // Nothing to do for pnpm, since the pnpm link command handles both the dependant and dependee
                    break;
                default:
                    this.tasks
                        .tryFind("install")
                        ?.exec(monorepo_1.NodePackageUtils.command.cmd(this.package.packageManager, "link"));
                    break;
            }
        }
    }
}
exports.GeneratedTypescriptRuntimeBaseProject = GeneratedTypescriptRuntimeBaseProject;
//# sourceMappingURL=data:application/json;base64,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