"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypeSafeApiProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const monorepo_1 = require("../../monorepo");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const type_safe_api_model_project_1 = require("./model/type-safe-api-model-project");
/**
 * Project for a type-safe API, defined using Smithy or OpenAPI.
 *
 * Generates a CDK construct to deploy your API, as well as client and server code to help build your API quickly.
 *
 * @pjid type-safe-api
 */
class TypeSafeApiProject extends projen_1.Project {
    constructor(options) {
        super(options);
        this.getNxWorkspace = (options) => {
            return options.parent ? monorepo_1.NxWorkspace.of(options.parent) : undefined;
        };
        const nxWorkspace = this.getNxWorkspace(options);
        const isNxWorkspace = this.parent &&
            (monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, monorepo_1.MonorepoTsProject) ||
                monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, monorepo_1.MonorepoJavaProject) ||
                monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, monorepo_1.MonorepoPythonProject));
        const handlerLanguages = [...new Set(options.handlers?.languages ?? [])];
        // API Definition project containing the model
        const modelDir = "model";
        this.model = new type_safe_api_model_project_1.TypeSafeApiModelProject({
            parent: nxWorkspace ? this.parent : this,
            outdir: nxWorkspace ? path.join(options.outdir, modelDir) : modelDir,
            name: `${options.name}-model`,
            modelLanguage: options.model.language,
            modelOptions: options.model.options,
            handlerLanguages,
        });
        // Ensure we always generate a runtime project for the infrastructure language, regardless of what was specified by
        // the user. Likewise we generate a runtime project for any handler languages specified
        const runtimeLanguages = [
            ...new Set([
                ...(options.runtime?.languages ?? []),
                options.infrastructure.language,
                ...(options.handlers?.languages ?? []),
            ]),
        ];
        const generatedDir = "generated";
        const runtimeDir = path.join(generatedDir, "runtime");
        const runtimeDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, runtimeDir)
            : runtimeDir;
        // Path from a generated package directory (eg api/generated/runtime/typescript) to the model dir (ie api/model)
        const relativePathToModelDirFromGeneratedPackageDir = path.relative(path.join(this.outdir, runtimeDir, "language"), path.join(this.outdir, modelDir));
        const parsedSpecRelativeToGeneratedPackageDir = path.join(relativePathToModelDirFromGeneratedPackageDir, this.model.parsedSpecFile);
        // Declare the generated runtime projects
        const generatedRuntimeProjects = (0, generate_1.generateRuntimeProjects)(runtimeLanguages, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: runtimeDirRelativeToParent,
            isWithinMonorepo: isNxWorkspace,
            // Spec path relative to each generated runtime dir
            parsedSpecPath: parsedSpecRelativeToGeneratedPackageDir,
            typescriptOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected?.defaultBase ?? "mainline",
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                commitGeneratedCode: options.runtime?.options?.typescript?.commitGeneratedCode ??
                    options.commitGeneratedCode ??
                    false,
                ...options.runtime?.options?.typescript,
            },
            pythonOptions: {
                authorName: "APJ Cope",
                authorEmail: "apj-cope@amazon.com",
                version: "0.0.0",
                commitGeneratedCode: options.runtime?.options?.python?.commitGeneratedCode ??
                    options.commitGeneratedCode ??
                    false,
                ...options.runtime?.options?.python,
            },
            javaOptions: {
                version: "0.0.0",
                commitGeneratedCode: options.runtime?.options?.java?.commitGeneratedCode ??
                    options.commitGeneratedCode ??
                    false,
                ...options.runtime?.options?.java,
            },
        });
        const documentationFormats = [
            ...new Set(options.documentation?.formats ?? []),
        ];
        const docsDir = path.join(generatedDir, "documentation");
        const docsDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, docsDir)
            : docsDir;
        const generatedDocs = (0, generate_1.generateDocsProjects)(documentationFormats, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedDocsDir: docsDirRelativeToParent,
            // Spec path relative to each generated doc format dir
            parsedSpecPath: parsedSpecRelativeToGeneratedPackageDir,
            documentationOptions: options.documentation?.options,
        });
        this.documentation = {
            htmlRedoc: generatedDocs[languages_1.DocumentationFormat.HTML_REDOC],
            html2: generatedDocs[languages_1.DocumentationFormat.HTML2],
            markdown: generatedDocs[languages_1.DocumentationFormat.MARKDOWN],
            plantuml: generatedDocs[languages_1.DocumentationFormat.PLANTUML],
        };
        const libraries = [...new Set(options.library?.libraries ?? [])];
        const libraryDir = path.join(generatedDir, "libraries");
        const libraryDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, libraryDir)
            : libraryDir;
        // Declare the generated runtime projects
        const generatedLibraryProjects = (0, generate_1.generateLibraryProjects)(libraries, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: libraryDirRelativeToParent,
            isWithinMonorepo: isNxWorkspace,
            // Spec path relative to each generated library dir
            parsedSpecPath: parsedSpecRelativeToGeneratedPackageDir,
            typescriptReactQueryHooksOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected.defaultBase ?? "mainline",
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                commitGeneratedCode: options.library?.options?.typescriptReactQueryHooks
                    ?.commitGeneratedCode ??
                    options.commitGeneratedCode ??
                    false,
                ...options.library?.options?.typescriptReactQueryHooks,
            },
        });
        // Ensure the generated runtime, libraries and docs projects have a dependency on the model project
        if (this.parent) {
            [
                ...Object.values(generatedRuntimeProjects),
                ...Object.values(generatedDocs),
                ...Object.values(generatedLibraryProjects),
            ].forEach((project) => {
                monorepo_1.NxProject.ensure(project).addImplicitDependency(this.model);
            });
        }
        this.runtime = {
            typescript: generatedRuntimeProjects[languages_1.Language.TYPESCRIPT]
                ? generatedRuntimeProjects[languages_1.Language.TYPESCRIPT]
                : undefined,
            java: generatedRuntimeProjects[languages_1.Language.JAVA]
                ? generatedRuntimeProjects[languages_1.Language.JAVA]
                : undefined,
            python: generatedRuntimeProjects[languages_1.Language.PYTHON]
                ? generatedRuntimeProjects[languages_1.Language.PYTHON]
                : undefined,
        };
        this.library = {
            typescriptReactQueryHooks: generatedLibraryProjects[languages_1.Library.TYPESCRIPT_REACT_QUERY_HOOKS]
                ? generatedLibraryProjects[languages_1.Library.TYPESCRIPT_REACT_QUERY_HOOKS]
                : undefined,
        };
        const handlersDir = "handlers";
        const handlersDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, handlersDir)
            : handlersDir;
        const relativePathToModelDirFromHandlersDir = path.relative(path.join(this.outdir, handlersDir, "language"), path.join(this.outdir, modelDir));
        const parsedSpecRelativeToHandlersDir = path.join(relativePathToModelDirFromHandlersDir, this.model.parsedSpecFile);
        // Declare the generated handlers projects
        const generatedHandlersProjects = (0, generate_1.generateHandlersProjects)(handlerLanguages, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: handlersDirRelativeToParent,
            isWithinMonorepo: isNxWorkspace,
            // Spec path relative to each generated handlers package dir
            parsedSpecPath: parsedSpecRelativeToHandlersDir,
            typescriptOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected.defaultBase ?? "mainline",
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                commitGeneratedCode: options.handlers?.options?.typescript?.commitGeneratedCode ??
                    options.commitGeneratedCode ??
                    false,
                ...options.handlers?.options?.typescript,
            },
            pythonOptions: {
                authorName: "APJ Cope",
                authorEmail: "apj-cope@amazon.com",
                version: "0.0.0",
                commitGeneratedCode: options.handlers?.options?.python?.commitGeneratedCode ??
                    options.commitGeneratedCode ??
                    false,
                ...options.handlers?.options?.python,
            },
            javaOptions: {
                version: "0.0.0",
                commitGeneratedCode: options.handlers?.options?.java?.commitGeneratedCode ??
                    options.commitGeneratedCode ??
                    false,
                ...options.handlers?.options?.java,
            },
            generatedRuntimes: {
                typescript: this.runtime.typescript,
                python: this.runtime.python,
                java: this.runtime.java,
            },
        });
        this.handlers = {
            typescript: generatedHandlersProjects[languages_1.Language.TYPESCRIPT]
                ? generatedHandlersProjects[languages_1.Language.TYPESCRIPT]
                : undefined,
            java: generatedHandlersProjects[languages_1.Language.JAVA]
                ? generatedHandlersProjects[languages_1.Language.JAVA]
                : undefined,
            python: generatedHandlersProjects[languages_1.Language.PYTHON]
                ? generatedHandlersProjects[languages_1.Language.PYTHON]
                : undefined,
        };
        // Ensure the handlers project depends on the appropriate runtime projects
        if (this.handlers.typescript) {
            monorepo_1.NxProject.ensure(this.handlers.typescript).addImplicitDependency(this.runtime.typescript);
        }
        if (this.handlers.java) {
            monorepo_1.NxProject.ensure(this.handlers.java).addImplicitDependency(this.runtime.java);
        }
        if (this.handlers.python) {
            monorepo_1.NxProject.ensure(this.handlers.python).addImplicitDependency(this.runtime.python);
        }
        const infraDir = path.join(generatedDir, "infrastructure");
        const infraDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, infraDir)
            : infraDir;
        // Infrastructure project
        const infraProject = (0, generate_1.generateInfraProject)(options.infrastructure.language, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: infraDirRelativeToParent,
            isWithinMonorepo: isNxWorkspace,
            // Spec path relative to each generated infra package dir
            parsedSpecPath: parsedSpecRelativeToGeneratedPackageDir,
            typescriptOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected.defaultBase ?? "mainline",
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                commitGeneratedCode: options.infrastructure.options?.typescript?.commitGeneratedCode ??
                    options.commitGeneratedCode ??
                    false,
                ...options.infrastructure.options?.typescript,
            },
            pythonOptions: {
                authorName: "APJ Cope",
                authorEmail: "apj-cope@amazon.com",
                version: "0.0.0",
                commitGeneratedCode: options.infrastructure.options?.python?.commitGeneratedCode ??
                    options.commitGeneratedCode ??
                    false,
                ...options.infrastructure.options?.python,
            },
            javaOptions: {
                version: "0.0.0",
                commitGeneratedCode: options.infrastructure.options?.java?.commitGeneratedCode ??
                    options.commitGeneratedCode ??
                    false,
                ...options.infrastructure.options?.java,
            },
            generatedRuntimes: {
                typescript: this.runtime.typescript,
                python: this.runtime.python,
                java: this.runtime.java,
            },
            generatedHandlers: {
                typescript: this.handlers.typescript,
                python: this.handlers.python,
                java: this.handlers.java,
            },
        });
        const infraProjects = {};
        // Add implicit dependencies and assign the appropriate infrastructure project member
        switch (options.infrastructure.language) {
            case languages_1.Language.JAVA:
                monorepo_1.NxProject.ensure(infraProject).addImplicitDependency(this.runtime.java);
                infraProjects.java = infraProject;
                break;
            case languages_1.Language.PYTHON:
                monorepo_1.NxProject.ensure(infraProject).addImplicitDependency(this.runtime.python);
                infraProjects.python = infraProject;
                break;
            case languages_1.Language.TYPESCRIPT:
                monorepo_1.NxProject.ensure(infraProject).addImplicitDependency(this.runtime.typescript);
                infraProjects.typescript = infraProject;
                break;
            default:
                throw new Error(`Unknown infrastructure language ${options.infrastructure.language}`);
        }
        this.infrastructure = infraProjects;
        monorepo_1.NxProject.ensure(infraProject).addImplicitDependency(this.model);
        // Expose collections of projects
        const allRuntimes = Object.values(generatedRuntimeProjects);
        const allInfrastructure = [infraProject];
        const allLibraries = Object.values(generatedLibraryProjects);
        const allDocumentation = Object.values(generatedDocs);
        const allHandlers = Object.values(generatedHandlersProjects);
        this.all = {
            model: [this.model],
            runtimes: allRuntimes,
            infrastructure: allInfrastructure,
            libraries: allLibraries,
            documentation: allDocumentation,
            handlers: allHandlers,
            projects: [
                this.model,
                ...allRuntimes,
                ...allInfrastructure,
                ...allLibraries,
                ...allDocumentation,
                ...allHandlers,
            ],
        };
        if (!nxWorkspace) {
            // Add a task for the non-monorepo case to build the projects in the right order
            [
                this.model,
                ...Object.values(generatedRuntimeProjects),
                infraProject,
                ...Object.values(generatedLibraryProjects),
                ...Object.values(generatedDocs),
            ].forEach((project) => {
                this.compileTask.exec("npx projen build", {
                    cwd: path.relative(this.outdir, project.outdir),
                });
            });
        }
        // Add the README as a sample file which the user may edit
        new projen_1.SampleFile(this, "README.md", {
            sourcePath: path.resolve(__dirname, "..", "..", "samples", "type-safe-api", "readme", "TYPE_SAFE_API.md"),
        });
    }
    /**
     * @inheritDoc
     */
    postSynthesize() {
        // Migration code, since we've moved these generated directories under a parent "generated"
        // folder, we delete the generated projects which would otherwise be orphaned and still
        // checked into VCS
        ["runtime", "libraries", "infrastructure", "documentation"].forEach((dir) => fs.rmSync(path.join(this.outdir, dir), { force: true, recursive: true }));
        super.postSynthesize();
    }
}
exports.TypeSafeApiProject = TypeSafeApiProject;
_a = JSII_RTTI_SYMBOL_1;
TypeSafeApiProject[_a] = { fqn: "@aws/pdk.type_safe_api.TypeSafeApiProject", version: "0.23.61" };
//# sourceMappingURL=data:application/json;base64,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