"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypeSafeWebSocketApiProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const monorepo_1 = require("../../monorepo");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const type_safe_websocket_api_model_project_1 = require("./model/type-safe-websocket-api-model-project");
/**
 * Project for a Type Safe WebSocket API, defined using Smithy or OpenAPI.
 *
 * Generates a CDK construct to deploy your API, as well as client and server code to help build your API quickly.
 *
 * @experimental
 * @pjid type-safe-ws-api
 */
class TypeSafeWebSocketApiProject extends projen_1.Project {
    constructor(options) {
        super(options);
        this.getNxWorkspace = (options) => {
            return options.parent ? monorepo_1.NxWorkspace.of(options.parent) : undefined;
        };
        const nxWorkspace = this.getNxWorkspace(options);
        const isNxWorkspace = this.parent &&
            (monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, monorepo_1.MonorepoTsProject) ||
                monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, monorepo_1.MonorepoJavaProject) ||
                monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, monorepo_1.MonorepoPythonProject));
        const handlerLanguages = [...new Set(options.handlers?.languages ?? [])];
        // API Definition project containing the model
        const modelDir = "model";
        this.model = new type_safe_websocket_api_model_project_1.TypeSafeWebSocketApiModelProject({
            parent: nxWorkspace ? this.parent : this,
            outdir: nxWorkspace ? path.join(options.outdir, modelDir) : modelDir,
            name: `${options.name}-model`,
            modelLanguage: options.model.language,
            modelOptions: options.model.options,
            handlerLanguages,
        });
        // Ensure we always generate a runtime project for the infrastructure language, regardless of what was specified by
        // the user. Likewise we generate a runtime project for any handler languages specified
        const runtimeLanguages = [
            ...new Set([
                ...(options.runtime?.languages ?? []),
                options.infrastructure.language,
                ...(options.handlers?.languages ?? []),
            ]),
        ];
        // TODO: Delete when python/java support is implemented
        if (runtimeLanguages.includes(languages_1.Language.JAVA) ||
            runtimeLanguages.includes(languages_1.Language.PYTHON)) {
            throw new Error(`Python and Java are not yet supported by Type Safe WebSocket API! Support is coming soon...`);
        }
        const generatedDir = "generated";
        const runtimeDir = path.join(generatedDir, "runtime");
        const runtimeDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, runtimeDir)
            : runtimeDir;
        // Path from a generated package directory (eg api/generated/runtime/typescript) to the model dir (ie api/model)
        const relativePathToModelDirFromGeneratedPackageDir = path.relative(path.join(this.outdir, runtimeDir, "language"), path.join(this.outdir, modelDir));
        const parsedSpecRelativeToGeneratedPackageDir = path.join(relativePathToModelDirFromGeneratedPackageDir, this.model.parsedSpecFile);
        // Declare the generated runtime projects
        const generatedRuntimeProjects = (0, generate_1.generateAsyncRuntimeProjects)(runtimeLanguages, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: runtimeDirRelativeToParent,
            isWithinMonorepo: isNxWorkspace,
            // Spec path relative to each generated runtime dir
            parsedSpecPath: parsedSpecRelativeToGeneratedPackageDir,
            typescriptOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected?.defaultBase ?? "mainline",
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                ...options.runtime?.options?.typescript,
            },
            pythonOptions: {
                authorName: "APJ Cope",
                authorEmail: "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.runtime?.options?.python,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.runtime?.options?.java,
            },
        });
        const documentationFormats = [
            ...new Set(options.documentation?.formats ?? []),
        ];
        const docsDir = path.join(generatedDir, "documentation");
        const docsDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, docsDir)
            : docsDir;
        // AsyncAPI specification is used for WebSocket documentation generation
        const asyncapiJsonFilePathRelativeToGeneratedPackageDir = path.join(relativePathToModelDirFromGeneratedPackageDir, this.model.asyncApiSpecFile);
        const generatedDocs = (0, generate_1.generateDocsProjects)(documentationFormats, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedDocsDir: docsDirRelativeToParent,
            // Spec path relative to each generated doc format dir
            parsedSpecPath: asyncapiJsonFilePathRelativeToGeneratedPackageDir,
            asyncDocumentationOptions: options.documentation?.options,
        });
        // Documentation projects use AsyncAPI generator which can intermittently fail
        // when executed in parallel to other AsyncAPI generator commands. We protect against this
        // by ensuring documentation projects are built sequentially.
        const docProjects = Object.values(generatedDocs);
        docProjects.forEach((docProject, i) => {
            if (docProjects[i - 1]) {
                monorepo_1.NxProject.ensure(docProjects[i - 1]).addImplicitDependency(docProject);
            }
        });
        this.documentation = {
            html: generatedDocs[languages_1.WebSocketDocumentationFormat.HTML],
            markdown: generatedDocs[languages_1.WebSocketDocumentationFormat.MARKDOWN],
        };
        const librarySet = new Set(options.library?.libraries ?? []);
        // Hooks depend on client, so always add the client if we specified hooks
        if (librarySet.has(languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_HOOKS)) {
            librarySet.add(languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_CLIENT);
        }
        const libraries = [...librarySet];
        const libraryDir = path.join(generatedDir, "libraries");
        const libraryDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, libraryDir)
            : libraryDir;
        // Declare the generated runtime projects
        const generatedLibraryProjects = (0, generate_1.generateAsyncLibraryProjects)(libraries, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: libraryDirRelativeToParent,
            isWithinMonorepo: isNxWorkspace,
            // Spec path relative to each generated library dir
            parsedSpecPath: parsedSpecRelativeToGeneratedPackageDir,
            typescriptWebSocketClientOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected.defaultBase ?? "mainline",
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                ...options.library?.options?.typescriptWebSocketClient,
            },
            typescriptWebSocketHooksOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected.defaultBase ?? "mainline",
                clientPackageName: options.library?.options?.typescriptWebSocketClient?.name,
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                ...options.library?.options?.typescriptWebSocketHooks,
            },
        });
        // Ensure the generated runtime, libraries and docs projects have a dependency on the model project
        if (this.parent) {
            [
                ...Object.values(generatedRuntimeProjects),
                ...Object.values(generatedDocs),
                ...Object.values(generatedLibraryProjects),
            ].forEach((project) => {
                monorepo_1.NxProject.ensure(project).addImplicitDependency(this.model);
            });
        }
        this.runtime = {
            typescript: generatedRuntimeProjects[languages_1.Language.TYPESCRIPT]
                ? generatedRuntimeProjects[languages_1.Language.TYPESCRIPT]
                : undefined,
            java: generatedRuntimeProjects[languages_1.Language.JAVA]
                ? generatedRuntimeProjects[languages_1.Language.JAVA]
                : undefined,
            python: generatedRuntimeProjects[languages_1.Language.PYTHON]
                ? generatedRuntimeProjects[languages_1.Language.PYTHON]
                : undefined,
        };
        this.library = {
            typescriptWebSocketClient: generatedLibraryProjects[languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_CLIENT]
                ? generatedLibraryProjects[languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_CLIENT]
                : undefined,
            typescriptWebSocketHooks: generatedLibraryProjects[languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_HOOKS]
                ? generatedLibraryProjects[languages_1.WebSocketLibrary.TYPESCRIPT_WEBSOCKET_HOOKS]
                : undefined,
        };
        // For the hooks library, add a dependency on the client
        if (this.library.typescriptWebSocketHooks &&
            this.library.typescriptWebSocketClient) {
            this.library.typescriptWebSocketHooks.addDeps(this.library.typescriptWebSocketClient.package.packageName);
        }
        const handlersDir = "handlers";
        const handlersDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, handlersDir)
            : handlersDir;
        const relativePathToModelDirFromHandlersDir = path.relative(path.join(this.outdir, handlersDir, "language"), path.join(this.outdir, modelDir));
        const parsedSpecRelativeToHandlersDir = path.join(relativePathToModelDirFromHandlersDir, this.model.parsedSpecFile);
        // Declare the generated handlers projects
        const generatedHandlersProjects = (0, generate_1.generateAsyncHandlersProjects)(handlerLanguages, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: handlersDirRelativeToParent,
            isWithinMonorepo: isNxWorkspace,
            // Spec path relative to each generated handlers package dir
            parsedSpecPath: parsedSpecRelativeToHandlersDir,
            typescriptOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected.defaultBase ?? "mainline",
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                ...options.handlers?.options?.typescript,
            },
            pythonOptions: {
                authorName: "APJ Cope",
                authorEmail: "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.handlers?.options?.python,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.handlers?.options?.java,
            },
            generatedRuntimes: {
                typescript: this.runtime.typescript,
                python: this.runtime.python,
                java: this.runtime.java,
            },
        });
        this.handlers = {
            typescript: generatedHandlersProjects[languages_1.Language.TYPESCRIPT]
                ? generatedHandlersProjects[languages_1.Language.TYPESCRIPT]
                : undefined,
            java: generatedHandlersProjects[languages_1.Language.JAVA]
                ? generatedHandlersProjects[languages_1.Language.JAVA]
                : undefined,
            python: generatedHandlersProjects[languages_1.Language.PYTHON]
                ? generatedHandlersProjects[languages_1.Language.PYTHON]
                : undefined,
        };
        // Ensure the handlers project depends on the appropriate runtime projects
        if (this.handlers.typescript) {
            monorepo_1.NxProject.ensure(this.handlers.typescript).addImplicitDependency(this.runtime.typescript);
        }
        if (this.handlers.java) {
            monorepo_1.NxProject.ensure(this.handlers.java).addImplicitDependency(this.runtime.java);
        }
        if (this.handlers.python) {
            monorepo_1.NxProject.ensure(this.handlers.python).addImplicitDependency(this.runtime.python);
        }
        const infraDir = path.join(generatedDir, "infrastructure");
        const infraDirRelativeToParent = nxWorkspace
            ? path.join(options.outdir, infraDir)
            : infraDir;
        // Infrastructure project
        const infraProject = (0, generate_1.generateAsyncInfraProject)(options.infrastructure.language, {
            parent: nxWorkspace ? this.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: infraDirRelativeToParent,
            isWithinMonorepo: isNxWorkspace,
            // Spec path relative to each generated infra package dir
            parsedSpecPath: parsedSpecRelativeToGeneratedPackageDir,
            typescriptOptions: {
                // Try to infer monorepo default release branch, otherwise default to mainline unless overridden
                defaultReleaseBranch: nxWorkspace?.affected.defaultBase ?? "mainline",
                packageManager: this.parent &&
                    monorepo_1.ProjectUtils.isNamedInstanceOf(this.parent, javascript_1.NodeProject)
                    ? this.parent.package.packageManager
                    : javascript_1.NodePackageManager.PNPM,
                ...options.infrastructure.options?.typescript,
            },
            pythonOptions: {
                authorName: "APJ Cope",
                authorEmail: "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.infrastructure.options?.python,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.infrastructure.options?.java,
            },
            generatedRuntimes: {
                typescript: this.runtime.typescript,
                python: this.runtime.python,
                java: this.runtime.java,
            },
            generatedHandlers: {
                typescript: this.handlers.typescript,
                python: this.handlers.python,
                java: this.handlers.java,
            },
        });
        const infraProjects = {};
        // Add implicit dependencies and assign the appropriate infrastructure project member
        switch (options.infrastructure.language) {
            case languages_1.Language.JAVA:
                monorepo_1.NxProject.ensure(infraProject).addImplicitDependency(this.runtime.java);
                infraProjects.java = infraProject;
                break;
            case languages_1.Language.PYTHON:
                monorepo_1.NxProject.ensure(infraProject).addImplicitDependency(this.runtime.python);
                infraProjects.python = infraProject;
                break;
            case languages_1.Language.TYPESCRIPT:
                monorepo_1.NxProject.ensure(infraProject).addImplicitDependency(this.runtime.typescript);
                infraProjects.typescript = infraProject;
                break;
            default:
                throw new Error(`Unknown infrastructure language ${options.infrastructure.language}`);
        }
        this.infrastructure = infraProjects;
        monorepo_1.NxProject.ensure(infraProject).addImplicitDependency(this.model);
        // Expose collections of projects
        const allRuntimes = Object.values(generatedRuntimeProjects);
        const allInfrastructure = [infraProject];
        const allLibraries = Object.values(generatedLibraryProjects);
        const allDocumentation = Object.values(generatedDocs);
        const allHandlers = Object.values(generatedHandlersProjects);
        this.all = {
            model: [this.model],
            runtimes: allRuntimes,
            infrastructure: allInfrastructure,
            libraries: allLibraries,
            documentation: allDocumentation,
            handlers: allHandlers,
            projects: [
                this.model,
                ...allRuntimes,
                ...allInfrastructure,
                ...allLibraries,
                ...allDocumentation,
                ...allHandlers,
            ],
        };
        if (!nxWorkspace) {
            // Add a task for the non-monorepo case to build the projects in the right order
            [
                this.model,
                ...Object.values(generatedRuntimeProjects),
                infraProject,
                ...Object.values(generatedLibraryProjects),
                ...Object.values(generatedDocs),
            ].forEach((project) => {
                this.compileTask.exec("npx projen build", {
                    cwd: path.relative(this.outdir, project.outdir),
                });
            });
        }
        // Add the README as a sample file which the user may edit
        new projen_1.SampleFile(this, "README.md", {
            sourcePath: path.resolve(__dirname, "..", "..", "samples", "type-safe-api", "readme", "TYPE_SAFE_API.md"),
        });
    }
}
exports.TypeSafeWebSocketApiProject = TypeSafeWebSocketApiProject;
_a = JSII_RTTI_SYMBOL_1;
TypeSafeWebSocketApiProject[_a] = { fqn: "@aws/pdk.type_safe_api.TypeSafeWebSocketApiProject", version: "0.23.61" };
//# sourceMappingURL=data:application/json;base64,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