# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/06_player.ipynb.

# %% auto 0
__all__ = ['Player', 'head_to_head']

# %% ../nbs/06_player.ipynb 3
from .data import *
from .name import *
from .game import *
from .match import *
from .team import *
import pandas as pd
import numpy as np

# %% ../nbs/06_player.ipynb 5
class Player():

    def __init__(self, player_id: str):
        self.player_id = player_id
        self.name = get_player_name(self.player_id)
        self.teams = self.associated_teams()
        game_mask = game.w_team_id.isin(self.teams) | game.l_team_id.isin(self.teams)
        self.game = game[game_mask]

        self.matches_played = self.get_matches_played()
        self.num_matches_played = len(self.matches_played)
        self.matches_won = self.get_matches_won()
        self.num_matches_won = len(self.matches_won)

        self.games_played = self.game.game_id.values
        self.num_games_played = len(self.games_played)
        self.games_won = self.game[self.game.w_team_id.isin(self.teams)].game_id.values
        self.num_games_won = len(self.games_won)

        self.partner_ids = self.associated_partners()

    def __str__(self):
        return f"{self.name}, id: {self.player_id}"
    __repr__ = __str__

    def associated_teams(self):
        '''
        Returns the team_ids of the teams that the player played for.
        '''
        return team[team.player_id == self.player_id].team_id.values
    
    def associated_partners(self):
        '''
        Returns the player_ids of the players that the player has played with
        '''
        partners = []
        for team_id in self.teams:
            team = Team(team_id)
            for p_id in team.players:
                if p_id != self.player_id:
                    partners.append(p_id)
        return partners
    
    def get_matches_played(self):
        '''
        Returns the match_ids of matches played (in the database) by the player
        '''
        ms = []
        for team_id in self.teams:
            team_matches = match[(match.team_id_1 == team_id) | (match.team_id_2 == team_id)].match_id.values
            ms += team_matches.tolist()
        return ms
        
    def get_matches_won(self):
        ms = []
        for m_id in self.matches_played:
            m = Match(m_id)
            if m.w_team_id in self.teams:
                ms += [m_id]
        return ms
    
    def error_rate(self):
        '''
        Returns the error rate of the player accross all games played
        '''
        error_rates = []
        for g_id in self.games_played:
            g = Game(g_id)
            error_rate = g.get_error_rate(self.player_id)
            error_rates += [error_rate]
        return sum(error_rates)/len(error_rates)
    
    def partners_win_rate(self):
        '''
        Returns the win rate of the player's partners excluding games where the player was on the same team as their partner
        '''
        
        partners_exclusive_wins = 0

        for p_id in self.partner_ids:
            p2 = Player(p_id)
            games_w_partner = self.games_played_with_partner(p2)
            p2_games_wo_player_partner = set(self.games_played).difference(set(games_w_partner))

            for g_id in p2_games_wo_player_partner:
                g = Game(g_id)
                if g.w_team_id in p2.teams:
                    partners_exclusive_wins += 1
                
        if len(p2_games_wo_player_partner) == 0:
            return None

        return partners_exclusive_wins/len(p2_games_wo_player_partner)

    def partners_error_rate(self):
        '''
        Returns the average error rate of the player's partners
        '''
        error_rates = []
        for p_id in self.partner_ids:
            p = Player(p_id)
            error_rates += [p.error_rate()]
        return sum(error_rates)/len(error_rates)
            
    def games_played_with_partner(self, partner):
        '''
        Returns the games played on the same team as the given partner
        '''
        games_in_common = set(self.games_played).intersection(set(partner.games_played))
        games_w_partner = []
        for g_common in games_in_common:
            g = Game(g_common)
            #find if self and partner were on the same team
            if (g.w_team_id in self.teams and g.w_team_id in partner.teams) or (g.l_team_id in self.teams and g.l_team_id in partner.teams):
                games_w_partner.append(g.game_id)
        return games_w_partner

    def third_shot_profile(self):
        '''
        Returns the third shot profile of the player
        '''
        third_shots = np.zeros(4)
        for g_id in self.games_played:
            g = Game(g_id)
            thirds = np.array(g.player_third_shots(self.player_id))
            third_shots += thirds
        return third_shots

    def summarize_player(self):
        '''
        Prints a summary of the player's stats
        '''
        print(f"Player: {self.name}")
        print(f"Matches played: {self.num_matches_played}")
        print(f"Matches won: {self.num_matches_won}")
        print("Percentage of matches won: {:.2f}%".format(self.num_matches_won/self.num_matches_played*100))
        print(f"Games played: {self.num_games_played}")
        print(f"Games won: {self.num_games_won}")
        print("Percentage of games won: {:.2f}%".format(self.num_games_won/self.num_games_played*100))
        print(f"Teams: {[get_team_name(team) for team in self.teams]}")
    
    def overall_player_impact(self):
        '''
        Returns the mean and std of player impact taken over all games played in the database
        '''
        game_impacts = np.zeros(len(self.games_played))
        for i, g_id in enumerate(self.games_played):
            g = Game(g_id)
            game_impacts[i] = g.player_impact_flow(self.player_id, g.num_rallies)
        return np.mean(game_impacts), np.std(game_impacts)


# %% ../nbs/06_player.ipynb 21
def head_to_head(p1: Player, p2: Player):
    '''
    Returns the results of matches where p1 and p2 have played against each other
    '''
    p1_matches = p1.matches_played
    p2_matches = p2.matches_played

    p1_count = 0
    p2_count = 0
    for m_id in p1_matches:
        if m_id in p2_matches:
            m = Match(m_id)
            if (p.name in m.w_team_name and p2.name in m.l_team_name):
                p1_count += 1
            elif (p.name in m.l_team_name and p2.name in m.w_team_name):
                p2_count += 1
    print(f"{p1.name} has played against {p2.name} in {p1_count + p2_count} matches and has won {p1_count} times")
