"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Archiver = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib/");
const aws_cdk_lib_1 = require("aws-cdk-lib/");
const codebuild = require("aws-cdk-lib/aws-codebuild");
const iam = require("aws-cdk-lib/aws-iam");
const kms = require("aws-cdk-lib/aws-kms");
const logs = require("aws-cdk-lib/aws-logs");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const snsNotifications = require("aws-cdk-lib/aws-s3-notifications");
const sns = require("aws-cdk-lib/aws-sns");
const constructs_1 = require("constructs");
class Archiver extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.logGroup = this.createLogGroup();
        const topic = new sns.Topic(this, 'notifications', {
            displayName: 'archiver-notifications',
        });
        const bucket = this.createArchiveBucket();
        bucket.addEventNotification(s3.EventType.OBJECT_CREATED, new snsNotifications.SnsDestination(topic));
        this.createProjects(bucket);
    }
    createLogGroup() {
        const key = this.createLogGroupKey();
        const loggroup = new logs.LogGroup(this, 'loggroup', {
            encryptionKey: key,
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        loggroup.node.addDependency(key);
        return loggroup;
    }
    /**
     *Create the S3 bucket that will later store the repositories.
     *
     * @private
     * @return {*}
     * @memberof Archiver
     */
    createArchiveBucket() {
        return new s3.Bucket(this, 'destination', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.RETAIN,
            lifecycleRules: [
                {
                    expiration: cdk.Duration.days(360),
                    transitions: [
                        {
                            storageClass: s3.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: cdk.Duration.days(30),
                        },
                        {
                            storageClass: s3.StorageClass.GLACIER,
                            transitionAfter: cdk.Duration.days(90),
                        },
                        {
                            storageClass: s3.StorageClass.DEEP_ARCHIVE,
                            transitionAfter: cdk.Duration.days(180),
                        },
                    ],
                },
            ],
            publicReadAccess: false,
            versioned: true,
        });
    }
    createLogGroupKey() {
        const key = new kms.Key(this, 'loggroupKey', {
            description: 'Repository Archiver',
            enableKeyRotation: true,
            pendingWindow: cdk.Duration.days(7),
            keyUsage: kms.KeyUsage.ENCRYPT_DECRYPT,
            keySpec: kms.KeySpec.SYMMETRIC_DEFAULT,
            alias: 'archiver-loggroup-key',
        });
        key.grantEncryptDecrypt(new iam.ServicePrincipal(`logs.${cdk.Stack.of(this).region}.amazonaws.com`));
        return key;
    }
    /**
     * Creates for each backup configuration a separate CodeBuild project
     *
     * @private
     * @param {s3.Bucket} bucket
     * @memberof Archiver
     */
    createProjects(bucket) {
        this.props.backupConfigurations.forEach((element) => {
            const project = this.createProject(element, bucket);
            project.enableBatchBuilds();
            bucket.grantReadWrite(project);
        });
    }
    /**
     * Create a CodeBuild project
     *
     * @private
     * @param {BackupConfiguration} element
     * @param {cdk.aws_s3.Bucket} bucket
     * @return {*}
     * @memberof Archiver
     */
    createProject(element, bucket) {
        return new codebuild.Project(this, 'archiver-' + element.organizationName + '-' + element.projectName, {
            timeout: cdk.Duration.hours(5),
            projectName: 'AzureDevOpsGitBackup' +
                '-' +
                element.organizationName +
                '-' +
                element.projectName,
            description: 'Backup Azure DevOps git repositories to an S3 bucket.',
            checkSecretsInPlainTextEnvVariables: true,
            concurrentBuildLimit: 90,
            environment: {
                environmentVariables: {
                    TOKEN: {
                        value: element.secretArn + ':pat',
                        type: codebuild.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    },
                    ORGANIZATION: { value: element.organizationName },
                    PROJECT: { value: element.projectName },
                },
                buildImage: codebuild.LinuxBuildImage.STANDARD_6_0,
            },
            logging: {
                cloudWatch: {
                    enabled: true,
                    logGroup: this.logGroup,
                },
            },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: 0.2,
                batch: {
                    'fail-fast': false,
                    'build-list': this.createBatchConfiguration(element.repositoryNames),
                },
                phases: {
                    build: {
                        commands: [
                            'git clone --mirror "https://${TOKEN}@dev.azure.com/${ORGANIZATION}/${PROJECT}/_git/${REPOSITORY}"',
                            'tar czf ${REPOSITORY}.tgz ./${REPOSITORY}.git',
                            'aws s3 cp ./${REPOSITORY}.tgz ' +
                                bucket.s3UrlForObject() +
                                '/${ORGANIZATION}/${PROJECT}/${REPOSITORY}.tgz',
                        ],
                    },
                },
            }),
        });
    }
    createBatchConfiguration(repositoryNames) {
        const output = [];
        repositoryNames.forEach((element) => {
            output.push({
                identifier: 'build_' + element.replace(/-/g, '_'),
                env: {
                    variables: {
                        REPOSITORY: element,
                    },
                },
            });
        });
        return output;
    }
}
exports.Archiver = Archiver;
_a = JSII_RTTI_SYMBOL_1;
Archiver[_a] = { fqn: "azure-devops-repository-archiver.Archiver", version: "0.0.6" };
//# sourceMappingURL=data:application/json;base64,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